// Copyright 2017 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package oauth2

import (
	"fmt"
	"math/rand"
	"net/http"
	"time"
)

// default cookie name.
const cookieName = "_oauth_state_"

// createState generates and returns a new opaque state
// value that is also stored in the http.Response by
// creating a session cookie.
func createState(w http.ResponseWriter) string {
	cookie := &http.Cookie{
		Name:   cookieName,
		Value:  random(),
		MaxAge: 1800,
	}
	http.SetCookie(w, cookie)
	return cookie.Value
}

// validateState returns an error if the state value does
// not match the session cookie value.
func validateState(r *http.Request, state string) error {
	cookie, err := r.Cookie(cookieName)
	if err != nil {
		return err
	}
	if state != cookie.Value {
		return ErrState
	}
	return nil
}

// deleteState deletes the state from the session cookie.
func deleteState(w http.ResponseWriter) {
	http.SetCookie(w, &http.Cookie{
		Name:    cookieName,
		MaxAge:  -1,
		Expires: time.Unix(0, 0),
	})
}

// random creates an opaque value shared between the
// http.Request and the callback used to validate redirects.
func random() string {
	return fmt.Sprintf("%x", rand.Uint64())
}
