# 12 "cut.mll"
 

open Printf

type toc_style = Normal | Both | Special


exception Error of string

module type Config = sig
  val verbose : int
  val name_in : string
  val name_out : string
  val toc_style : toc_style
  val svg_arrows: bool
  val cross_links : bool
  val small_length : int
end

module Make (Config : Config) =
struct

open Config
open Lexing
open MyStack

module Out = CutOut.Make(Config)

let count = ref 0

let dir =
  let dir = Filename.dirname name_out in
  if dir = "." then
    None
  else
    Some dir

and base =
  let base = Filename.basename name_in in
  try Filename.chop_extension base
  with Invalid_argument _ -> base

let changed_t = Hashtbl.create 17

let record_changed oldname newname =
  try
    let _ = Hashtbl.find changed_t oldname in
    Hashtbl.replace changed_t oldname newname
  with Not_found ->
    Hashtbl.add changed_t oldname newname

let check_changed name =
  try Hashtbl.find changed_t name
  with Not_found -> name


let real_name name =
  let name = check_changed name in
  match dir with
  | None -> name
  | Some dir -> Filename.concat dir name

let real_open_out name = open_out (real_name name)

let some_links = ref false

let env = Hashtbl.create 17

let add_env key v =
  if verbose > 0 then
    eprintf "add_env '%s' -> '%s'\n" key v ;
  Hashtbl.add env key v

let replace_env key v =
  if verbose > 0 then
    eprintf "replace_env '%s' -> '%s'\n" key v ;
  Hashtbl.replace env key v

let imgsrc img alt = sprintf "<img src=\"%s\" alt=\"%s\">" img alt

let () =
  let ext = if Config.svg_arrows then "svg" else "gif" in
  let add_ext f = sprintf "%s.%s" f ext in
  add_env "UPTXT" (imgsrc (add_ext "contents_motif") "Up") ;
  add_env "PREVTXT" (imgsrc (add_ext "previous_motif") "Previous") ;
  add_env "NEXTTXT" (imgsrc (add_ext "next_motif")  "Next") ;
  ()

let get_env key = try Hashtbl.find env key with Not_found ->
  eprintf "Key '%s' not found\n" key ;
  assert false

let get_env_not_found key = Hashtbl.find env key

(* Accumulate all META, link and similar tags that appear in the preamble
   in order to output them in the preamble of every generated page. *)

let header_buff = Out.create_buff "header-buf"
let style_buff = Out.create_buff "style-buf"
let common_headers = ref ""
and link_style = ref ""

let adjoin_to_header s = Out.put header_buff s

and adjoin_to_header_char c = Out.put_char header_buff c

let finalize_header () =
  if not (Out.is_empty style_buff) then begin
    let css_name = sprintf "%s.css" base in
    link_style :=
       sprintf
         "<link rel=\"stylesheet\" type=\"text/css\" href=\"%s\">\n"
         css_name ;
    adjoin_to_header !link_style ;
    let chan = real_open_out css_name in
    Out.to_chan chan style_buff ;
    close_out chan
  end ;
  common_headers := Out.to_string header_buff

let html_buff = Out.create_buff "html-buf"
let html_head = ref ""
and html_foot = ref ""
and html_prefix = ref ""

let phase = ref (-1)
;;

let body = ref "<body>"
and doctype = ref ""
and html = ref "<html>"
;;

let new_filename _from =
  incr count ;
  sprintf "%s%03d.html" base !count

let out = ref (Out.create_null ())
and out_prefix = ref (Out.create_null ())
and outname = ref ""
and lastclosed = ref ""
and otheroutname = ref ""
and flowname_stack = (MyStack.create "flowname" : (string * bool) MyStack.t)
and flow_stack = (MyStack.create "flow" : Out.t MyStack.t)
;;

let toc = ref !out
and tocname = ref !outname
and otherout = ref !out
;;

let close_loc _ctx _name out =  Out.close out

let dont_change = ref ""

let change_name oldname name =
  if !phase <= 0 && oldname <> !dont_change then begin
    if verbose > 0 then
      eprintf "Change '%s' into '%s'\n" oldname name ;
    record_changed oldname name ;
  end

let start_phase () =
  incr phase ;
  if verbose > 0 then
    eprintf "Starting phase number: %d\n" !phase ;
  let base_out = Filename.basename name_out in
  dont_change := base_out ;
  outname := base_out ;
  tocname := base_out ;
  otheroutname := "" ;
  count := 0 ;
  if !phase > 0 then begin
    out := Out.create_chan (real_name base_out)
  end ;
  toc := !out
;;

let openlist out =
(*  eprintf "OPEN LIST: %s\n" (Out.get_name out) ; *)
  Out.put out "<ul>\n"

and closelist out =
(*  eprintf "CLOSE LIST: %s\n" (Out.get_name out) ; *)
  Out.put out "</li></ul>\n"

and itemref fst_item filename s out =
  let filename = check_changed filename in
  if not fst_item then Out.put out "</li>" ;
  Out.put out "<li>" ;
  Out.put out "<a href=\"" ;
  Out.put out filename ;
  Out.put out "\">" ;
  Out.put out s ;
  Out.put out "</a>\n"

and itemanchor fst_item filename label s out =
  let filename = check_changed filename in
  if not fst_item then Out.put out "</li>" ;
  Out.put out "<li>" ;
  Out.put out "<a href=\"" ;
  Out.put out filename ;
  Out.put_char out '#' ;
  Url.encode_fragment
    (fun c -> Out.put_char out c)
    (fun s -> Out.put out s)
    label ;
  Out.put out "\">" ;
  Out.put out s ;
  Out.put out "</a>\n"
;;

let delayed_anchor = ref false
and prev_anchor = ref None

let do_putanchor label out =
  Out.put out "<a id=\"" ;
  Out.put out label ;
  Out.put out "\"></a>"
;;

let putanchor label out =
  if !delayed_anchor then
    prev_anchor := Some (label, out)
  else
    do_putanchor label out

let putlink out name txt =
  let name = check_changed name in
  Out.put out "<a href=\"" ;
  Out.put out name ;
  Out.put out "\">" ;
  Out.put out txt ;
  Out.put out "</a>\n"
;;

let link_buff = Out.create_buff "link-buf"

let old_putlinks  name =
  let links_there = ref false in
  if verbose > 0 then
    eprintf "putlinks: %s\n" name ;
  begin try
    putlink link_buff (Thread.prev name) (get_env "PREVTXT") ;
    links_there := true
  with Not_found ->
    if verbose > 0 then
      eprintf "No prev link for %s\n" name
  end ;
  begin try
    putlink link_buff (Thread.up name) (get_env "UPTXT") ;
    links_there := true
  with Not_found -> () end ;
  begin try
    putlink link_buff (Thread.next name) (get_env "NEXTTXT") ;
    links_there := true
  with Not_found -> () end ;
  if !links_there then
    Some (Out.to_string link_buff)
  else
    None

let get_linkvar key = try get_env_not_found key with Not_found -> "VAR"

module SV =
  SubstVar.Make
    (struct
      let arg_buff = Out.create_buff "arg-buff"
      let put = Out.put arg_buff
      let put_char = Out.put_char arg_buff
      let to_string () = Out.to_string arg_buff
    end)

let putlinks name =
  let substlink key get =
    try
      let url = (check_changed (get name)) in
      let macro = get_linkvar key in
      SV.subst  macro url ;
      true
    with Not_found ->  false in

  try
    let linksmacro = get_env_not_found "LINKS" in
    let b1 = substlink "PREVLINK" Thread.prev in
    let b2 = substlink "UPLINK" Thread.up in
    let b3 = substlink "NEXTLINK" Thread.next in
    if b1 || b2 || b3 then begin
      let arg = SV.to_string () in
      SV.subst linksmacro arg ;
      Some (SV.to_string ())
    end else None

  with Not_found -> old_putlinks name

let putlinks_start out outname =
  if cross_links then
    match putlinks outname with
    | Some s ->
        some_links := true ;
        Out.put out s ;
        Out.put out "<hr>\n"
    | None -> ()

let putlinks_end out outname =
  if cross_links then
    match putlinks outname with
    | Some s ->
        some_links := true ;
        Out.put out "<hr>\n" ;
        Out.put out s
    | None -> ()


type file_opt = { with_footer : bool ; with_links : bool ; }

let std_file_opt = { with_footer=true; with_links=true ; }

let openhtml opt title out outname =
  Out.put out !doctype ; Out.put_char out '\n' ;
  Out.put out !html ; Out.put_char out '\n' ;
  Out.put out "<head>" ;
  Out.put out !common_headers;
  Out.put out "<title>" ;
  let title =
    let title = !html_prefix^title in
    try Tagout.tagout title
    with Tagout.Error ->
      Location.print_pos () ;
      eprintf  "Warning, cannot erase tags from '%s'\n%!" title ;
      title in
  Out.put out title ;
  Out.put out "</title>\n" ;
  Out.put out "</head>\n" ;
  Out.put out !body;
  Out.put out "\n" ;
  if opt.with_links then putlinks_start out outname ;
  if opt.with_footer then Out.put out !html_head


and closehtml opt name out =
  if opt.with_footer then Out.put out !html_foot ;
  if opt.with_links then putlinks_end out name ;
  Out.put out "</body>\n" ;
  Out.put out "</html>\n" ;
  close_loc "closehtml" name out
;;

let put s = Out.put !out s
and put_char c = Out.put_char !out c
;;

let cur_level = ref (Section.value "DOCUMENT")
and chapter = ref (Section.value "CHAPTER")
and depth = ref 2
;;


(* Open all lists in toc from chapter to sec, with sec > chapter *)
let rec do_open l1 l2 =
  if l1 < l2 then begin
     begin match toc_style with
     | Both -> openlist !toc ; openlist !out_prefix
     | Special -> openlist !out_prefix
     | Normal  -> openlist !toc
    end ;
    do_open (l1+1) l2
  end
;;

(* close from l1 down to l2 *)
let rec do_close l1 l2 =
  if l1 > l2 then begin
     begin match toc_style with
     | Both -> closelist  !toc ; closelist !out_prefix
     | Special -> closelist !out_prefix
     | Normal  -> closelist !toc
     end ;
     do_close (l1-1) l2
  end else
  cur_level := l1
;;

let anchor = ref 0
;;

let open_section sec name a =
  if !phase > 0 then begin
    let fst_item =
      if !cur_level > sec then begin
        do_close !cur_level sec ;
        false
      end else if !cur_level < sec then begin
        do_open  !cur_level sec ;
        true
      end else false in
    let label =
      match a with
      | Some label -> label
      | None ->
         incr anchor ;
         "toc"^string_of_int !anchor in
    begin match toc_style with
    | Normal ->
        itemanchor  fst_item !outname label name !toc ;
    | Both ->
        itemanchor  fst_item !outname label name !toc ;
        itemanchor  fst_item !outname label name !out_prefix
    | Special    ->
        itemanchor  fst_item !outname label name !out_prefix
    end ;
    begin match a with
    | Some _ -> ()
    | None -> putanchor label !out
    end ;
    cur_level := sec
  end else
    cur_level := sec

and close_section sec =
  if !phase > 0 then do_close !cur_level sec
  else
    cur_level := sec
;;

let close_chapter () =
  if verbose > 0 then
    eprintf "Close chapter out=%s toc=%s\n" !outname !tocname ;
  if !phase > 0 then begin
    if !outname <> !tocname then begin
      closehtml std_file_opt !outname !out ;
      let doout out what =
        if false then begin
          eprintf "DEBUG:\n" ;
          Out.debug stderr what ;
          eprintf "\n"
        end ;
        Out.to_chan out what in
      begin match toc_style with
      | Both|Special ->
          let real_out = real_open_out !outname in
(* Those hacking try with avoid failure for cuttingsection = document *)
          begin try
            doout real_out !out_prefix
          with Misc.Fatal _ -> () end ;
          begin try
            doout real_out !out ;
          with Misc.Fatal _ -> () end ;
          close_out real_out
      | Normal -> ()
      end ;
    end ;
    out := !toc
  end else begin
    lastclosed := !outname ;
    outname := !tocname
  end

and open_chapter fst_item name  =
  outname := new_filename ("open_chapter <<"^name^">>") ;
  if verbose > 0 then
    eprintf
      "Open chapter out=%s toc=%s cur_level=%d\n"
      !outname !tocname  !cur_level ;
  if !phase > 0 then begin
    begin match toc_style with
    | Both|Special ->
        out_prefix := Out.create_buff (!outname ^ "-prefix") ;
        out := !out_prefix ;
        openhtml std_file_opt name !out_prefix !outname
    | Normal ->
        out := Out.create_chan (real_name !outname) ;
        openhtml std_file_opt name !out !outname
    end ;
    itemref fst_item !outname name !toc ;
    cur_level := !chapter
  end else begin
    if verbose > 0 then
      eprintf "link prev=%s next=%s\n" !lastclosed !outname ;
    Thread.setup !outname !tocname ;
    Thread.setprevnext !lastclosed !outname ;
    cur_level := !chapter
  end
;;
let setlink set target =
  if !phase = 0 && target <> "" then
    set !outname target

let open_notes_pred sec_notes =
  (sec_notes <> !chapter) ||
  (!cur_level < sec_notes)

let open_notes sticky sec_notes =
  if verbose > 0 && !phase > 0 then
    eprintf "Notes flushed as %s (current cut is %s, current level is %s)\n"
      (Section.pretty sec_notes)
      (Section.pretty !chapter)
      (Section.pretty !cur_level) ;
  if
    not sticky && open_notes_pred sec_notes
  then begin
    otheroutname := !outname ;
    outname := new_filename "open_notes" ;
    if !phase > 0 then begin
      otherout := !out ;
      out := Out.create_chan (real_name !outname) ;
      Out.put !out !doctype ; Out.put_char !out '\n' ;
      Out.put !out !html ; Out.put_char !out '\n' ;
      Out.put !out "<head><title>Notes</title>\n" ;
      Out.put !out !common_headers ;
      Out.put !out "</head>\n" ;
      Out.put !out !body ;
      Out.put !out "\n"
    end
  end else
   otheroutname := ""

and close_notes () =
  if !otheroutname <> "" then begin
     Out.put !out "\n</body></html>\n" ;
     close_loc "notes" !outname !out ;
     outname := !otheroutname ;
     out := !otherout ;
     otheroutname := ""
  end
;;

let toc_buf = Out.create_buff "toc-buf"
and arg_buf = Out.create_buff "arg-buf"
;;

let stack = MyStack.create "main"
;;

let save_state newchapter newdepth =
  if verbose > 0 then
    eprintf "New state: %d\n" newchapter ;
  push stack
    (!outname, MyStack.save flowname_stack, MyStack.save flow_stack,
     !chapter,!depth,!toc,!tocname,!cur_level,!lastclosed,!out_prefix) ;
  chapter := newchapter ;
  depth := newdepth ;
  tocname := !outname ;
  lastclosed := "" ;
  toc := !out
;;

let restore_state () =
  if verbose > 0 then eprintf "Restore\n%!" ;
  let
    oldoutname, oldflowname, oldflow,
    oldchapter,olddepth,oldtoc,oldtocname,
    oldlevel,_oldlastclosed,oldprefix  = pop stack in
  outname := oldoutname ;
  MyStack.restore flowname_stack oldflowname ;
  MyStack.restore flow_stack oldflow ;
  chapter := oldchapter ;
  depth := olddepth ;
  toc := oldtoc ;
  tocname := oldtocname ;
  lastclosed := !lastclosed ;
  cur_level := oldlevel ;
  out_prefix := oldprefix
;;


let hevea_footer = ref false

let hevea_footer_buff = Out.create_buff "hevea-footer-buf"


let close_top lxm =
  Out.put !toc !html_foot ;
  putlinks_end !toc !tocname ;
  if !hevea_footer then begin
    Out.put !out "<!--FOOTER-->\n" ;
    Out.copy hevea_footer_buff !out
  end ;
  Out.put !toc lxm ;
  if !tocname = "" then
    Out.flush !toc
  else
   close_loc "top" !tocname !toc
;;

let open_toc () = if !phase > 0 then openlist !toc
and close_toc () = if !phase > 0 then closelist !toc
;;

let close_all () =
  if !cur_level > !chapter then begin
    close_section !chapter ;
    close_chapter () ;
    close_toc ()
  end else if !cur_level = !chapter then begin
    close_chapter () ;
    close_toc ()
  end ;
  cur_level := (Section.value "DOCUMENT")

let openflow with_footer title =
  let new_outname = new_filename "openflow" in
  push flowname_stack (!outname,with_footer) ;
  outname := new_outname ;
  if !phase > 0 then begin
    push flow_stack !out ;
    out := Out.create_chan (real_name !outname) ;
    let opt = { with_footer=with_footer; with_links=false; } in
    openhtml opt title !out !outname
  end

and closeflow () =
  let prev_out, with_footer = pop flowname_stack in
  if !phase > 0 then begin
    let opt = { with_links=false; with_footer=with_footer; } in
    closehtml opt !outname !out;
    out := pop flow_stack
  end ;
  outname := prev_out


# 624 "cut.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\229\255\230\255\000\000\114\000\027\000\228\000\086\001\
    \200\001\081\000\012\000\000\000\061\000\023\000\008\000\018\000\
    \015\000\030\000\018\000\050\000\061\000\066\000\064\000\067\000\
    \061\000\056\000\078\000\083\000\001\000\110\000\111\000\000\000\
    \255\255\074\000\173\000\073\000\084\000\089\000\089\000\129\000\
    \013\000\000\000\001\000\239\255\079\000\075\000\207\000\014\000\
    \002\000\002\000\238\255\081\000\096\000\089\000\000\000\002\000\
    \094\000\137\000\125\000\016\000\163\000\147\000\003\000\237\255\
    \174\000\003\000\141\000\162\000\156\000\004\000\005\000\175\000\
    \168\000\179\000\216\000\204\000\188\000\006\000\173\000\178\000\
    \171\000\007\000\008\000\186\000\189\000\189\000\175\000\009\000\
    \010\000\196\000\011\000\031\001\186\000\190\000\188\000\181\000\
    \241\000\224\000\208\000\004\000\250\255\187\000\203\000\191\000\
    \243\000\242\000\227\000\007\000\241\255\213\000\222\000\012\000\
    \004\001\253\000\088\001\058\002\172\002\112\001\221\000\005\001\
    \005\003\069\001\119\003\095\001\072\001\098\001\224\000\240\000\
    \067\001\045\001\022\001\033\001\028\001\026\001\040\001\048\001\
    \087\001\178\003\013\004\078\001\100\001\062\001\247\255\055\001\
    \094\001\072\004\163\004\170\001\043\002\061\001\101\001\086\001\
    \085\001\008\000\244\255\077\001\076\001\072\001\145\001\150\001\
    \094\001\100\001\096\001\112\001\128\001\180\001\222\004\024\005\
    \140\001\147\001\140\001\127\001\151\001\151\001\147\001\165\001\
    \152\001\204\001\083\005\141\005\170\001\154\001\150\001\160\001\
    \172\001\183\001\236\255\069\002\071\002\255\005\113\006\227\006\
    \085\007\235\255\184\001\001\006\199\007\057\008\233\255\185\001\
    \029\006\171\008\029\009\143\009\234\255\196\001\115\006\131\001\
    \143\001\123\001\197\001\004\001\201\001\253\255\254\255\229\001\
    \218\001\219\001\222\001\214\001\225\001\202\009\005\010\249\001\
    \038\002\233\001\009\000\255\255\139\002\252\255\236\001\035\002\
    \211\001\224\001\226\001\231\001\236\001\240\001\023\002\255\255\
    \226\001\235\001\243\001\040\002\254\255\224\001\252\001\004\002\
    \177\002\181\002\241\001\251\001\007\002\202\002\183\002\249\001\
    \009\002\247\001\252\001\066\002\015\002\000\002\001\002\083\002\
    \034\003\010\000\253\255\002\002\134\002\253\255\073\002\003\002\
    \017\002\021\002\055\002\053\002\092\002\255\255\068\002\087\002\
    \096\002\193\002\036\003\077\002\088\002\104\002\121\003\197\002\
    \092\002\109\002\094\002\100\002\171\002\120\002\105\002\106\002\
    \188\002\149\003\254\255\163\002\254\255\177\002\111\002\111\002\
    \107\002\122\002\131\002\171\002\011\000\255\255\072\003\254\255\
    \188\002\121\002\148\002\144\002\158\002\166\002\206\002\012\000\
    \255\255\073\003\254\255\251\002\187\002\199\002\189\002\198\002\
    \206\002\246\002\013\000\255\255\075\003\253\255\254\255\016\003\
    \222\002\210\002\223\002\204\002\035\003\098\002\054\003\254\255\
    \055\003\039\003\014\000\255\255\105\003\251\255\252\255\091\003\
    \040\003\024\003\037\003\109\003\040\003\032\003\039\003\029\003\
    \038\003\046\003\086\003\255\255\038\003\038\003\054\003\043\003\
    \098\003\254\255\099\003\015\000\253\255\001\004\252\255\253\255\
    \065\003\119\003\104\003\016\000\255\255\063\003\075\003\147\003\
    \047\010\255\255\108\010\209\003\253\255\254\255\133\003\086\003\
    \098\003\110\003\152\003\255\255\209\005\251\255\252\255\253\255\
    \133\003\128\003\202\003\171\003\242\003\240\003\172\003\202\003\
    \045\004\017\004\237\003\237\003\222\010\252\255\246\003\080\011\
    \194\011\052\012\189\004\166\012\194\004\024\013\138\013\143\006\
    \218\013\042\014\030\004\018\005\254\255\122\014\202\014\174\004\
    \019\005\253\255\035\015\113\015\020\005\176\004\255\255\021\004\
    \010\004\024\004\251\003\017\000\255\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\025\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\016\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\017\000\255\255\255\255\255\255\255\255\255\255\001\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\018\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\002\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\010\000\255\255\255\255\
    \255\255\255\255\003\000\255\255\255\255\255\255\255\255\255\255\
    \004\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\005\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\014\000\255\255\255\255\255\255\255\255\
    \006\000\255\255\255\255\255\255\255\255\007\000\255\255\255\255\
    \255\255\255\255\255\255\007\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\009\000\009\000\009\000\255\255\255\255\255\255\
    \255\255\011\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\012\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\013\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\024\000\015\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\024\000\255\255\255\255\255\255\255\255\
    \024\000\255\255\255\255\255\255\255\255\255\255\024\000\255\255\
    \255\255\255\255\255\255\023\000\255\255\255\255\255\255\001\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\003\000\003\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\002\000\255\255\255\255\255\255\255\255\002\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\001\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \001\000\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\001\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\001\000\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \001\000\255\255\000\000\255\255\255\255\255\255\255\255\004\000\
    \004\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\002\000\255\255\255\255\255\255\255\255\
    \003\000\003\000\255\255\000\000\255\255\255\255\255\255\001\000\
    \001\000\255\255\255\255\255\255\255\255\255\255\001\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \003\000\003\000\003\000\003\000\255\255\000\000\255\255\255\255\
    \255\255\001\000\255\255\255\255\004\000\255\255\255\255\002\000\
    \002\000\002\000\000\000\255\255\255\255\002\000\255\255\255\255\
    \001\000\255\255\255\255\255\255\255\255\002\000\255\255\255\255\
    \255\255\255\255\001\000\000\000\255\255\255\255\255\255\001\000\
    \255\255\255\255\255\255\000\000\255\255";
  Lexing.lex_default =
   "\002\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\028\000\028\000\028\000\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\124\000\255\255\255\255\124\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\185\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \194\000\000\000\194\000\194\000\255\255\199\000\000\000\199\000\
    \199\000\255\255\255\255\205\000\000\000\205\000\205\000\255\255\
    \255\255\255\255\255\255\211\000\214\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\229\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\243\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\013\001\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\044\001\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\055\001\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\066\001\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\078\001\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\085\001\087\001\000\000\
    \255\255\255\255\255\255\000\000\093\001\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\000\000\118\001\000\000\000\000\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\000\000\255\255\133\001\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\000\000\142\001\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\170\001\171\001\000\000\255\255\255\255\175\001\
    \176\001\000\000\255\255\255\255\180\001\181\001\000\000\255\255\
    \255\255\255\255\255\255\255\255\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\032\000\043\000\050\000\063\000\100\000\077\000\
    \077\000\108\000\154\000\227\000\010\001\053\001\064\001\075\001\
    \091\001\116\001\124\001\188\001\000\000\000\000\000\000\000\000\
    \055\000\009\000\055\000\071\000\070\000\070\000\077\000\082\000\
    \082\000\088\000\088\000\091\000\112\000\012\000\029\000\005\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\041\000\048\000\003\000\060\000\042\000\255\255\
    \049\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\180\000\155\000\127\000\113\000\109\000\
    \089\000\008\000\006\000\004\000\004\000\004\000\004\000\004\000\
    \007\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\187\000\187\000\207\000\011\000\013\000\
    \014\000\126\000\018\000\022\000\083\000\023\000\078\000\064\000\
    \025\000\020\000\033\000\017\000\052\000\019\000\026\000\051\000\
    \021\000\016\000\187\000\027\000\028\000\010\000\034\000\024\000\
    \044\000\037\000\038\000\030\000\030\000\039\000\045\000\046\000\
    \056\000\039\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\255\255\031\000\040\000\053\000\
    \054\000\015\000\057\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\034\000\058\000\059\000\
    \061\000\062\000\067\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\187\000\187\000\046\000\
    \068\000\065\000\069\000\035\000\072\000\036\000\073\000\074\000\
    \074\000\076\000\077\000\079\000\047\000\080\000\081\000\084\000\
    \001\000\255\255\066\000\085\000\187\000\075\000\086\000\087\000\
    \090\000\101\000\094\000\095\000\096\000\098\000\099\000\102\000\
    \103\000\096\000\104\000\104\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\097\000\106\000\
    \105\000\107\000\110\000\111\000\112\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\091\000\
    \114\000\119\000\120\000\114\000\128\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\201\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\187\000\
    \187\000\115\000\115\000\129\000\149\000\093\000\143\000\255\255\
    \123\000\123\000\125\000\123\000\123\000\092\000\255\255\255\255\
    \133\000\131\000\130\000\134\000\135\000\136\000\187\000\137\000\
    \115\000\117\000\117\000\141\000\142\000\144\000\145\000\123\000\
    \132\000\150\000\123\000\152\000\140\000\150\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \117\000\139\000\151\000\153\000\156\000\157\000\158\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\159\000\168\000\162\000\163\000\164\000\159\000\004\000\
    \004\000\004\000\004\000\189\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\190\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\187\000\187\000\165\000\166\000\169\000\170\000\171\000\
    \172\000\118\000\147\000\147\000\147\000\147\000\147\000\147\000\
    \147\000\147\000\147\000\147\000\161\000\173\000\174\000\175\000\
    \188\000\160\000\176\000\177\000\178\000\181\000\182\000\183\000\
    \184\000\185\000\208\000\209\000\210\000\186\000\193\000\198\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\204\000\211\000\255\255\215\000\216\000\217\000\
    \218\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\219\000\220\000\221\000\225\000\226\000\
    \011\001\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\115\000\115\000\255\255\224\000\245\000\
    \255\255\240\000\235\000\148\000\236\000\237\000\187\000\187\000\
    \187\000\187\000\234\000\223\000\238\000\239\000\241\000\242\000\
    \243\000\246\000\115\000\147\000\147\000\147\000\147\000\147\000\
    \147\000\147\000\147\000\147\000\147\000\187\000\244\000\188\000\
    \247\000\248\000\251\000\252\000\253\000\000\001\001\001\002\001\
    \003\001\004\001\005\001\006\001\007\001\008\001\233\000\022\001\
    \016\001\017\001\018\001\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\230\000\232\000\233\000\
    \019\001\020\001\021\001\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\117\000\117\000\255\255\
    \255\255\255\255\249\000\249\000\015\001\023\001\249\000\249\000\
    \254\000\254\000\014\001\024\001\255\255\025\001\028\001\231\000\
    \029\001\213\000\026\001\026\001\117\000\030\001\031\001\031\001\
    \033\001\249\000\034\001\253\000\253\000\249\000\035\001\254\000\
    \036\001\255\000\037\001\038\001\039\001\040\001\041\001\045\001\
    \046\001\026\001\047\001\048\001\049\001\031\001\050\001\032\001\
    \051\001\052\001\253\000\057\001\058\001\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\254\000\
    \059\001\060\001\061\001\062\001\063\001\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\121\000\
    \255\255\250\000\068\001\008\001\008\001\026\001\026\001\069\001\
    \070\001\071\001\072\001\073\001\074\001\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\080\001\
    \081\001\082\001\008\001\083\001\026\001\084\001\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \009\001\085\001\255\255\088\001\089\001\090\001\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \123\000\123\000\030\001\030\001\056\001\067\001\255\255\079\001\
    \114\001\097\001\098\001\255\255\099\001\100\001\101\001\108\001\
    \103\001\104\001\105\001\106\001\107\001\109\001\095\001\123\000\
    \027\001\030\001\110\001\111\001\102\001\112\001\041\001\041\001\
    \113\001\115\001\125\001\255\255\122\001\096\001\123\001\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\126\001\127\001\127\001\135\001\041\001\031\001\136\001\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\137\000\042\001\137\001\138\001\139\001\154\001\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\150\001\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\134\001\148\001\148\001\
    \149\001\151\001\148\001\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\140\000\121\001\149\001\
    \152\001\153\001\155\001\152\001\157\001\185\001\255\255\186\001\
    \185\001\187\001\139\000\000\000\000\000\120\001\000\000\000\000\
    \172\001\000\000\184\001\000\000\000\000\184\001\000\000\000\000\
    \255\255\255\255\000\000\077\001\000\000\152\001\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \145\000\094\001\153\001\000\000\000\000\000\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \000\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\148\000\000\000\000\000\162\001\162\001\
    \000\000\000\000\000\000\164\001\164\001\000\000\000\000\000\000\
    \177\001\132\001\182\001\147\000\147\000\147\000\147\000\147\000\
    \147\000\147\000\147\000\147\000\147\000\162\001\000\000\000\000\
    \000\000\000\000\164\001\000\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\166\000\163\001\
    \000\000\119\001\000\000\000\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\255\255\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\172\001\177\001\182\001\000\000\000\000\000\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\178\000\000\000\000\000\000\000\000\000\
    \000\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\000\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\255\255\000\000\
    \255\255\000\000\000\000\000\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \187\000\187\000\195\000\195\000\000\000\000\000\000\000\143\001\
    \000\000\000\000\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\144\001\146\001\000\000\000\000\000\000\000\000\187\000\
    \000\000\195\000\000\000\000\000\000\000\000\000\200\000\200\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\145\001\147\001\000\000\000\000\200\000\000\000\193\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\000\000\198\000\000\000\000\000\000\000\000\000\
    \196\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\187\000\187\000\206\000\206\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\187\000\000\000\206\000\000\000\000\000\000\000\000\000\
    \167\001\167\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\000\000\000\000\000\000\000\000\167\001\
    \000\000\204\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\166\001\000\000\000\000\000\000\
    \000\000\141\001\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\191\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\187\000\187\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\187\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\192\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\195\000\195\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\195\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\000\000\
    \000\000\000\000\000\000\193\000\000\000\000\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \187\000\187\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\187\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \004\000\004\000\004\000\197\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\200\000\200\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\200\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\000\000\000\000\000\000\000\000\198\000\
    \000\000\000\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\187\000\187\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\187\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\004\000\004\000\202\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\187\000\187\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\187\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\203\000\004\000\
    \206\000\206\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\206\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\000\000\000\000\000\000\000\000\204\000\000\000\000\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\221\000\000\000\000\000\000\000\000\000\000\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\000\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\224\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\223\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\129\001\000\000\000\000\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\000\000\000\000\000\000\000\000\000\000\255\255\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\129\001\000\000\000\000\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\162\001\
    \162\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\162\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\158\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\157\001\000\000\000\000\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\160\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\000\000\000\000\000\000\000\000\000\000\000\000\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\161\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\164\001\164\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \164\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\159\001\000\000\000\000\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\000\000\000\000\163\001\000\000\000\000\
    \000\000\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\164\001\164\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\164\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\159\001\
    \000\000\000\000\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\000\000\000\000\163\001\
    \000\000\000\000\000\000\159\001\159\001\159\001\165\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\164\001\164\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\164\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\159\001\000\000\000\000\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\000\000\
    \000\000\163\001\000\000\000\000\000\000\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\159\001\159\001\159\001\
    \165\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\163\001\
    \163\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\163\001\000\000\
    \175\001\174\001\000\000\000\000\000\000\176\001\000\000\000\000\
    \000\000\173\001\000\000\173\001\173\001\000\000\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\000\000\000\000\000\000\000\000\000\000\000\000\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\000\000\000\000\000\000\000\000\173\001\000\000\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\167\001\167\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \167\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\159\001\000\000\000\000\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\000\000\000\000\166\001\000\000\000\000\
    \000\000\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\166\001\166\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\166\001\000\000\170\001\169\001\000\000\000\000\
    \000\000\171\001\000\000\000\000\000\000\168\001\000\000\168\001\
    \168\001\000\000\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\000\000\000\000\000\000\
    \000\000\168\001\000\000\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\000\000\168\001\
    \168\001\000\000\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\000\000\000\000\000\000\
    \000\000\168\001\000\000\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\000\000\168\001\
    \168\001\000\000\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\000\000\000\000\000\000\
    \000\000\168\001\000\000\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\173\001\000\000\173\001\
    \173\001\000\000\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\000\000\000\000\000\000\
    \000\000\173\001\000\000\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\000\000\173\001\
    \173\001\000\000\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\000\000\000\000\000\000\
    \000\000\173\001\000\000\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\181\001\000\000\000\000\
    \000\000\000\000\180\001\000\000\000\000\000\000\000\000\000\000\
    \179\001\179\001\000\000\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\000\000\000\000\
    \000\000\000\000\179\001\000\000\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \000\000\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\000\000\000\000\000\000\000\000\
    \179\001\000\000\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\031\000\042\000\049\000\062\000\099\000\077\000\
    \077\000\107\000\153\000\226\000\009\001\052\001\063\001\074\001\
    \090\001\115\001\123\001\187\001\255\255\255\255\255\255\255\255\
    \054\000\003\000\055\000\065\000\069\000\070\000\077\000\081\000\
    \082\000\087\000\088\000\090\000\111\000\011\000\028\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\040\000\047\000\000\000\059\000\041\000\028\000\
    \048\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\010\000\013\000\014\000\016\000\017\000\
    \018\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\004\000\004\000\005\000\009\000\012\000\
    \012\000\015\000\012\000\012\000\019\000\012\000\020\000\021\000\
    \023\000\012\000\024\000\012\000\022\000\012\000\025\000\022\000\
    \012\000\012\000\004\000\026\000\027\000\009\000\033\000\023\000\
    \035\000\036\000\037\000\029\000\030\000\038\000\044\000\045\000\
    \051\000\039\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\029\000\030\000\039\000\052\000\
    \053\000\012\000\056\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\034\000\057\000\058\000\
    \060\000\061\000\066\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\006\000\006\000\046\000\
    \067\000\064\000\068\000\034\000\071\000\034\000\072\000\073\000\
    \074\000\075\000\076\000\078\000\046\000\079\000\080\000\083\000\
    \000\000\028\000\064\000\084\000\006\000\074\000\085\000\086\000\
    \089\000\092\000\093\000\094\000\095\000\097\000\098\000\101\000\
    \102\000\096\000\103\000\104\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\096\000\105\000\
    \104\000\106\000\109\000\110\000\112\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\091\000\
    \113\000\118\000\119\000\126\000\127\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\007\000\
    \007\000\114\000\114\000\128\000\130\000\091\000\131\000\121\000\
    \123\000\123\000\124\000\125\000\125\000\091\000\029\000\030\000\
    \132\000\129\000\129\000\133\000\134\000\135\000\007\000\136\000\
    \114\000\117\000\117\000\139\000\141\000\143\000\144\000\123\000\
    \129\000\149\000\125\000\151\000\140\000\150\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \117\000\140\000\150\000\152\000\155\000\156\000\157\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\158\000\160\000\161\000\162\000\163\000\159\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\008\000\008\000\164\000\165\000\168\000\169\000\170\000\
    \171\000\117\000\147\000\147\000\147\000\147\000\147\000\147\000\
    \147\000\147\000\147\000\147\000\159\000\172\000\173\000\174\000\
    \008\000\159\000\175\000\176\000\177\000\180\000\181\000\182\000\
    \183\000\184\000\207\000\208\000\209\000\185\000\194\000\199\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\205\000\210\000\211\000\212\000\215\000\216\000\
    \217\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\218\000\219\000\220\000\223\000\225\000\
    \230\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\115\000\115\000\121\000\224\000\232\000\
    \124\000\233\000\234\000\148\000\235\000\236\000\187\000\187\000\
    \188\000\188\000\231\000\224\000\237\000\238\000\240\000\241\000\
    \242\000\245\000\115\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\187\000\243\000\188\000\
    \246\000\247\000\250\000\251\000\252\000\255\000\000\001\001\001\
    \002\001\003\001\004\001\005\001\006\001\007\001\011\001\015\001\
    \014\001\016\001\017\001\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\228\000\231\000\231\000\
    \018\001\019\001\020\001\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\116\000\116\000\185\000\
    \194\000\199\000\248\000\248\000\014\001\022\001\249\000\249\000\
    \254\000\254\000\012\001\023\001\205\000\024\001\027\001\228\000\
    \028\001\212\000\025\001\025\001\116\000\029\001\031\001\031\001\
    \032\001\248\000\033\001\253\000\253\000\249\000\034\001\254\000\
    \035\001\254\000\036\001\037\001\038\001\039\001\040\001\043\001\
    \045\001\025\001\046\001\047\001\048\001\031\001\049\001\031\001\
    \050\001\051\001\253\000\056\001\057\001\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\253\000\
    \058\001\059\001\060\001\061\001\062\001\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\120\000\
    \243\000\249\000\067\001\008\001\008\001\026\001\026\001\068\001\
    \069\001\070\001\071\001\072\001\073\001\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\079\001\
    \080\001\081\001\008\001\082\001\026\001\083\001\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \008\001\084\001\085\001\086\001\088\001\089\001\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \122\000\122\000\030\001\030\001\054\001\065\001\012\001\076\001\
    \095\001\096\001\097\001\228\000\098\001\099\001\100\001\101\001\
    \102\001\103\001\104\001\105\001\106\001\108\001\092\001\122\000\
    \026\001\030\001\109\001\110\001\100\001\111\001\041\001\041\001\
    \112\001\114\001\120\001\043\001\121\001\092\001\122\001\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\125\001\126\001\127\001\134\001\041\001\030\001\135\001\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\137\000\041\001\136\001\137\001\138\001\144\001\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\145\001\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\131\001\146\001\147\001\
    \149\001\150\001\148\001\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\138\000\117\001\148\001\
    \151\001\153\001\154\001\155\001\158\001\183\001\086\001\184\001\
    \185\001\186\001\138\000\255\255\255\255\117\001\255\255\255\255\
    \170\001\255\255\183\001\255\255\255\255\185\001\255\255\255\255\
    \054\001\065\001\255\255\076\001\255\255\152\001\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \145\000\092\001\152\001\255\255\255\255\255\255\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \255\255\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\146\000\255\255\255\255\162\001\162\001\
    \255\255\255\255\255\255\164\001\164\001\255\255\255\255\255\255\
    \175\001\131\001\181\001\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\162\001\255\255\255\255\
    \255\255\255\255\164\001\255\255\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\166\000\164\001\
    \255\255\117\001\255\255\255\255\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\170\001\166\000\
    \166\000\166\000\166\000\166\000\166\000\166\000\166\000\166\000\
    \166\000\166\000\166\000\166\000\166\000\166\000\166\000\166\000\
    \166\000\166\000\166\000\166\000\166\000\166\000\166\000\166\000\
    \166\000\171\001\176\001\180\001\255\255\255\255\255\255\166\000\
    \166\000\166\000\166\000\166\000\166\000\166\000\166\000\166\000\
    \166\000\166\000\166\000\166\000\166\000\166\000\166\000\166\000\
    \166\000\166\000\166\000\166\000\166\000\166\000\166\000\166\000\
    \166\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\178\000\255\255\255\255\255\255\255\255\
    \255\255\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\255\255\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\175\001\255\255\
    \181\001\255\255\255\255\255\255\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \189\000\189\000\195\000\195\000\255\255\255\255\255\255\140\001\
    \255\255\255\255\171\001\176\001\180\001\255\255\255\255\255\255\
    \255\255\140\001\140\001\255\255\255\255\255\255\255\255\189\000\
    \255\255\195\000\255\255\255\255\255\255\255\255\200\000\200\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\000\
    \189\000\189\000\189\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\140\001\140\001\255\255\255\255\200\000\255\255\195\000\
    \189\000\189\000\189\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\189\000\189\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\189\000\189\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\189\000\255\255\200\000\255\255\255\255\255\255\255\255\
    \189\000\189\000\189\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\189\000\189\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\189\000\189\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\189\000\190\000\190\000\206\000\206\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\190\000\255\255\206\000\255\255\255\255\255\255\255\255\
    \167\001\167\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\190\000\190\000\190\000\190\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\255\255\255\255\255\255\255\255\167\001\
    \255\255\206\000\190\000\190\000\190\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\190\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\190\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\167\001\255\255\255\255\255\255\
    \255\255\140\001\190\000\190\000\190\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\190\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\190\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\191\000\191\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\195\000\255\255\191\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\200\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\192\000\192\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\206\000\255\255\192\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\255\255\
    \255\255\255\255\255\255\192\000\255\255\255\255\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \196\000\196\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\196\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\197\000\197\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\192\000\255\255\255\255\
    \255\255\197\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\255\255\255\255\255\255\255\255\197\000\
    \255\255\255\255\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\201\000\201\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\201\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\201\000\201\000\201\000\201\000\201\000\
    \201\000\201\000\201\000\201\000\201\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\201\000\201\000\201\000\201\000\
    \201\000\201\000\201\000\201\000\201\000\201\000\201\000\201\000\
    \201\000\201\000\201\000\201\000\201\000\201\000\201\000\201\000\
    \201\000\201\000\201\000\201\000\201\000\201\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\201\000\201\000\201\000\201\000\
    \201\000\201\000\201\000\201\000\201\000\201\000\201\000\201\000\
    \201\000\201\000\201\000\201\000\201\000\201\000\201\000\201\000\
    \201\000\201\000\201\000\201\000\201\000\201\000\202\000\202\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\197\000\255\255\255\255\255\255\202\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \202\000\202\000\202\000\202\000\202\000\202\000\202\000\202\000\
    \203\000\203\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\203\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\255\255\255\255\255\255\255\255\203\000\255\255\255\255\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\221\000\255\255\255\255\255\255\255\255\255\255\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\203\000\255\255\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\222\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\222\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\128\001\255\255\255\255\
    \128\001\128\001\128\001\128\001\128\001\128\001\128\001\128\001\
    \128\001\128\001\128\001\128\001\128\001\128\001\128\001\128\001\
    \128\001\128\001\128\001\128\001\128\001\128\001\128\001\128\001\
    \128\001\128\001\255\255\255\255\255\255\255\255\255\255\203\000\
    \128\001\128\001\128\001\128\001\128\001\128\001\128\001\128\001\
    \128\001\128\001\128\001\128\001\128\001\128\001\128\001\128\001\
    \128\001\128\001\128\001\128\001\128\001\128\001\128\001\128\001\
    \128\001\128\001\130\001\255\255\255\255\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\130\001\
    \130\001\130\001\130\001\130\001\130\001\130\001\130\001\156\001\
    \156\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\156\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\156\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\156\001\255\255\255\255\156\001\
    \156\001\156\001\156\001\156\001\156\001\156\001\156\001\156\001\
    \156\001\156\001\156\001\156\001\156\001\156\001\156\001\156\001\
    \156\001\156\001\156\001\156\001\156\001\156\001\156\001\156\001\
    \156\001\255\255\255\255\255\255\255\255\255\255\255\255\156\001\
    \156\001\156\001\156\001\156\001\156\001\156\001\156\001\156\001\
    \156\001\156\001\156\001\156\001\156\001\156\001\156\001\156\001\
    \156\001\156\001\156\001\156\001\156\001\156\001\156\001\156\001\
    \156\001\159\001\159\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \159\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\159\001\255\255\255\255\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\255\255\255\255\159\001\255\255\255\255\
    \255\255\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\159\001\159\001\159\001\159\001\159\001\
    \159\001\159\001\159\001\160\001\160\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\160\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\160\001\
    \255\255\255\255\160\001\160\001\160\001\160\001\160\001\160\001\
    \160\001\160\001\160\001\160\001\160\001\255\255\255\255\160\001\
    \255\255\255\255\255\255\160\001\160\001\160\001\160\001\160\001\
    \160\001\160\001\160\001\160\001\160\001\160\001\160\001\160\001\
    \160\001\160\001\160\001\160\001\160\001\160\001\160\001\160\001\
    \160\001\160\001\160\001\160\001\160\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\160\001\160\001\160\001\160\001\160\001\
    \160\001\160\001\160\001\160\001\160\001\160\001\160\001\160\001\
    \160\001\160\001\160\001\160\001\160\001\160\001\160\001\160\001\
    \160\001\160\001\160\001\160\001\160\001\161\001\161\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\161\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\161\001\255\255\255\255\161\001\161\001\161\001\161\001\
    \161\001\161\001\161\001\161\001\161\001\161\001\161\001\255\255\
    \255\255\161\001\255\255\255\255\255\255\161\001\161\001\161\001\
    \161\001\161\001\161\001\161\001\161\001\161\001\161\001\161\001\
    \161\001\161\001\161\001\161\001\161\001\161\001\161\001\161\001\
    \161\001\161\001\161\001\161\001\161\001\161\001\161\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\161\001\161\001\161\001\
    \161\001\161\001\161\001\161\001\161\001\161\001\161\001\161\001\
    \161\001\161\001\161\001\161\001\161\001\161\001\161\001\161\001\
    \161\001\161\001\161\001\161\001\161\001\161\001\161\001\163\001\
    \163\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\163\001\255\255\
    \163\001\163\001\255\255\255\255\255\255\163\001\255\255\255\255\
    \255\255\163\001\255\255\163\001\163\001\255\255\163\001\163\001\
    \163\001\163\001\163\001\163\001\163\001\163\001\163\001\163\001\
    \163\001\255\255\255\255\255\255\255\255\255\255\255\255\163\001\
    \163\001\163\001\163\001\163\001\163\001\163\001\163\001\163\001\
    \163\001\163\001\163\001\163\001\163\001\163\001\163\001\163\001\
    \163\001\163\001\163\001\163\001\163\001\163\001\163\001\163\001\
    \163\001\255\255\255\255\255\255\255\255\163\001\255\255\163\001\
    \163\001\163\001\163\001\163\001\163\001\163\001\163\001\163\001\
    \163\001\163\001\163\001\163\001\163\001\163\001\163\001\163\001\
    \163\001\163\001\163\001\163\001\163\001\163\001\163\001\163\001\
    \163\001\165\001\165\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \165\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\165\001\255\255\255\255\
    \165\001\165\001\165\001\165\001\165\001\165\001\165\001\165\001\
    \165\001\165\001\165\001\255\255\255\255\165\001\255\255\255\255\
    \255\255\165\001\165\001\165\001\165\001\165\001\165\001\165\001\
    \165\001\165\001\165\001\165\001\165\001\165\001\165\001\165\001\
    \165\001\165\001\165\001\165\001\165\001\165\001\165\001\165\001\
    \165\001\165\001\165\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\165\001\165\001\165\001\165\001\165\001\165\001\165\001\
    \165\001\165\001\165\001\165\001\165\001\165\001\165\001\165\001\
    \165\001\165\001\165\001\165\001\165\001\165\001\165\001\165\001\
    \165\001\165\001\165\001\166\001\166\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\166\001\255\255\166\001\166\001\255\255\255\255\
    \255\255\166\001\255\255\255\255\255\255\166\001\255\255\166\001\
    \166\001\255\255\166\001\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\166\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\166\001\255\255\255\255\255\255\
    \255\255\166\001\255\255\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\166\001\168\001\255\255\168\001\
    \168\001\255\255\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\255\255\255\255\255\255\
    \255\255\168\001\255\255\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\169\001\255\255\169\001\
    \169\001\255\255\169\001\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\169\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\169\001\255\255\255\255\255\255\
    \255\255\169\001\255\255\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\169\001\173\001\255\255\173\001\
    \173\001\255\255\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\255\255\255\255\255\255\
    \255\255\173\001\255\255\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\173\001\173\001\173\001\
    \173\001\173\001\173\001\173\001\173\001\174\001\255\255\174\001\
    \174\001\255\255\174\001\174\001\174\001\174\001\174\001\174\001\
    \174\001\174\001\174\001\174\001\174\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\174\001\174\001\174\001\174\001\174\001\
    \174\001\174\001\174\001\174\001\174\001\174\001\174\001\174\001\
    \174\001\174\001\174\001\174\001\174\001\174\001\174\001\174\001\
    \174\001\174\001\174\001\174\001\174\001\255\255\255\255\255\255\
    \255\255\174\001\255\255\174\001\174\001\174\001\174\001\174\001\
    \174\001\174\001\174\001\174\001\174\001\174\001\174\001\174\001\
    \174\001\174\001\174\001\174\001\174\001\174\001\174\001\174\001\
    \174\001\174\001\174\001\174\001\174\001\178\001\255\255\255\255\
    \255\255\255\255\178\001\255\255\255\255\255\255\255\255\255\255\
    \178\001\178\001\255\255\178\001\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\178\001\255\255\255\255\
    \255\255\255\255\178\001\255\255\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\178\001\179\001\179\001\
    \255\255\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\255\255\255\255\255\255\255\255\
    \179\001\255\255\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255";
  Lexing.lex_base_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\002\000\060\000\000\000\000\000\000\000\
    \149\000\016\000\225\000\000\000\028\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\028\001\086\001\000\000\000\000\000\000\035\000\000\000\
    \003\000\145\001\236\001\003\000\071\002\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\004\000\005\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\007\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\002\000\097\002\003\000\
    \177\002\001\003\031\000\027\000\098\000\000\000\000\000\000\000\
    \000\000\000\000\090\003\168\003\022\001\062\000\098\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\007\000\000\000\000\000\
    \000\000\000\000\000\000\020\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\048\000\066\000\048\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\078\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\084\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \098\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\098\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_default_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\017\000\000\000\000\000\017\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\095\000\095\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\095\000\095\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\001\000\001\000\001\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\029\000\001\000\040\000\075\000\075\000\081\000\081\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\014\000\000\000\087\000\
    \087\000\000\000\000\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\014\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \000\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\029\000\000\000\000\000\000\000\
    \000\000\000\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\017\000\017\000\017\000\017\000\
    \017\000\017\000\017\000\017\000\000\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\040\000\000\000\000\000\000\000\000\000\000\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\000\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\055\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\055\000\
    \000\000\000\000\087\000\087\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\087\000\000\000\090\000\000\000\000\000\000\000\000\000\
    \090\000\000\000\000\000\000\000\095\000\000\000\095\000\095\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\000\000\000\000\000\000\000\000\
    \095\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\000\000\095\000\095\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\000\000\000\000\000\000\000\000\
    \095\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\000\000\095\000\095\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\000\000\000\000\000\000\000\000\
    \095\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\090\000\000\000\000\000\000\000\
    \000\000\090\000\000\000\000\000\000\000\000\000\000\000\095\000\
    \095\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\000\000\000\000\000\000\
    \000\000\095\000\000\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\000\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\000\000\000\000\000\000\000\000\095\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000";
  Lexing.lex_check_code =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\114\000\114\000\115\000\115\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \114\000\136\000\115\000\144\000\165\000\166\000\177\000\178\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\121\000\147\000\147\000\147\000\147\000\147\000\
    \147\000\147\000\147\000\147\000\147\000\119\000\124\000\165\001\
    \167\001\170\001\171\001\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\255\255\255\255\255\255\
    \181\001\255\255\255\255\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\120\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \121\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\171\001\124\000\255\255\255\255\170\001\
    \255\255\255\255\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\137\000\180\001\181\001\255\255\
    \255\255\255\255\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\122\000\122\000\122\000\122\000\
    \122\000\122\000\122\000\122\000\255\255\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\145\000\255\255\255\255\255\255\255\255\255\255\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\255\255\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\145\000\145\000\145\000\145\000\
    \145\000\145\000\145\000\145\000\146\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\180\001\255\255\
    \255\255\255\255\255\255\255\255\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\146\000\
    \146\000\146\000\146\000\146\000\146\000\146\000\146\000\148\000\
    \255\255\255\255\166\001\166\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\166\001\255\255\166\001\255\255\255\255\255\255\255\255\
    \166\001\255\255\255\255\255\255\166\001\255\255\166\001\166\001\
    \255\255\166\001\166\001\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\166\001\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\255\255\255\255\255\255\255\255\
    \166\001\255\255\166\001\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\166\001\166\001\166\001\166\001\
    \166\001\166\001\166\001\166\001\168\001\255\255\168\001\168\001\
    \255\255\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\255\255\255\255\255\255\255\255\
    \168\001\255\255\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\168\001\168\001\168\001\168\001\
    \168\001\168\001\168\001\168\001\169\001\255\255\169\001\169\001\
    \255\255\169\001\169\001\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\169\001\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\255\255\255\255\255\255\255\255\
    \169\001\255\255\169\001\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\169\001\169\001\169\001\169\001\
    \169\001\169\001\169\001\169\001\178\001\255\255\255\255\255\255\
    \255\255\178\001\255\255\255\255\255\255\255\255\255\255\178\001\
    \178\001\255\255\178\001\178\001\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\178\001\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\255\255\255\255\255\255\
    \255\255\178\001\255\255\178\001\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\178\001\178\001\178\001\
    \178\001\178\001\178\001\178\001\178\001\179\001\179\001\255\255\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\255\255\255\255\255\255\255\255\179\001\
    \255\255\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\179\001\179\001\179\001\179\001\179\001\
    \179\001\179\001\179\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255";
  Lexing.lex_code =
   "\255\004\255\255\005\255\255\003\255\000\004\001\005\255\006\255\
    \255\007\255\255\000\004\001\005\003\006\002\007\255\008\255\255\
    \009\255\255\000\008\001\009\255\010\255\255\012\255\011\255\255\
    \003\255\000\010\001\011\255\012\255\255\013\255\255\011\255\012\
    \255\255\000\010\001\011\003\012\002\013\255\014\255\255\000\014\
    \255\015\255\255\000\015\255\002\255\255\003\255\002\255\255\003\
    \255\255\001\002\000\003\255";
}

let rec main lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 16 (-1); __ocaml_lex_main_rec lexbuf 0
and __ocaml_lex_main_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 644 "cut.mll"
    (let lxm = lexeme lexbuf in
    if !phase > 0 then begin
      put lxm ;
      put ("<!--HACHA command line is: ") ;
      for i = 0 to Array.length Sys.argv - 1 do
        put Sys.argv.(i) ;
        put_char ' '
      done ;
      put "-->\n"
    end ;
    main lexbuf)
# 2353 "cut.ml"

  | 1 ->
# 656 "cut.mll"
   (let title,b = flowline lexbuf in
   openflow b title ;
   main lexbuf)
# 2360 "cut.ml"

  | 2 ->
# 660 "cut.mll"
   ( let pairs = getargs lexbuf in
     if !phase = 0 then begin
       List.iter
         (fun (name, v) -> replace_env name v)
         pairs
     end ;
     main lexbuf )
# 2371 "cut.ml"

  | 3 ->
# 668 "cut.mll"
   (let links = getargs lexbuf in
   List.iter
     (fun (name,v) -> match name with
     | "UP" -> setlink Thread.setup v
     | "PREV" -> setlink Thread.setprev v
     | "NEXT" -> setlink Thread.setnext v
     | _ -> ())
     links ;
   main lexbuf)
# 2384 "cut.ml"

  | 4 ->
# 678 "cut.mll"
   (let l = getargs lexbuf in
   if !phase = 0 then begin
     match l with
     | [] -> ()
     | (_,v)::_ -> html_prefix := v
   end ;
   main lexbuf)
# 2395 "cut.ml"

  | 5 ->
# 686 "cut.mll"
   (closeflow () ;
   main lexbuf)
# 2401 "cut.ml"

  | 6 ->
# 689 "cut.mll"
    (let name = tocline lexbuf in
    change_name !outname name ;
    main lexbuf)
# 2408 "cut.ml"

  | 7 ->
let
# 697 "cut.mll"
                                           arg
# 2414 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 698 "cut.mll"
                       a
# 2419 "cut.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2) in
# 699 "cut.mll"
    (let sn =
      if String.uppercase_ascii arg = "NOW" then !chapter
      else Section.value arg in
    let name = tocline lexbuf in
    if verbose > 1 then begin
      eprintf "TOC arg=%s, name=%s, anchor=%s\n" arg name
        (match a with Some x -> x | None -> "-")
    end;
    if sn < !chapter then begin
      if !cur_level >= !chapter then begin
        close_section (!chapter) ;
        close_chapter () ;
        close_toc ()
      end ;
      cur_level := sn
    end else if sn = !chapter then begin
      let fst_item =
        if !cur_level < sn then begin
          open_toc () ;
          true
        end else begin
          close_section !chapter ;
          close_chapter  () ;
          false
        end in
      open_chapter  fst_item name
    end else if sn <= !chapter + !depth then begin (* sn > !chapter *)
      if !cur_level < !chapter then begin
        open_toc () ;
        open_chapter true ""
      end ;
      close_section sn ;
      open_section sn name a
    end ;
    main lexbuf)
# 2457 "cut.ml"

  | 8 ->
let
# 734 "cut.mll"
                                  style
# 2463 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1) in
# 735 "cut.mll"
    (
     Section.set_style style ;
     main lexbuf
    )
# 2470 "cut.ml"

  | 9 ->
let
# 739 "cut.mll"
                                 name
# 2476 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 739 "cut.mll"
                                                           i_opt
# 2481 "cut.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2) in
# 740 "cut.mll"
    (let chapter = Section.value (String.uppercase_ascii name) in
    let depth = match i_opt with
    | None -> !depth
    | Some s -> int_of_string s in
    skip_endcom lexbuf ;
    save_state chapter depth ;
    cur_level := Section.value "DOCUMENT" ;
    main lexbuf)
# 2492 "cut.ml"

  | 10 ->
# 749 "cut.mll"
    (if !phase > 0 then begin
      put_char '\n' ;
      match toc_style with
      | Both|Special when !out == !out_prefix ->
          out := Out.create_buff "out-buf"
      | _ -> ()
    end ;
    main lexbuf)
# 2504 "cut.ml"

  | 11 ->
# 758 "cut.mll"
    (close_all () ;
      restore_state () ;
      main lexbuf)
# 2511 "cut.ml"

  | 12 ->
let
# 761 "cut.mll"
                                            sec_notes
# 2517 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 762 "cut.mll"
    (skip_endcom lexbuf ;
    open_notes false (Section.value sec_notes) ;
    main lexbuf)
# 2523 "cut.ml"

  | 13 ->
let
# 765 "cut.mll"
                                                  sec_notes
# 2529 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 766 "cut.mll"
    (skip_endcom lexbuf ;
    open_notes true (Section.value sec_notes) ;
    main lexbuf)
# 2535 "cut.ml"

  | 14 ->
# 770 "cut.mll"
    (if !otheroutname <> "" then
      close_notes ();
      main lexbuf)
# 2542 "cut.ml"

  | 15 ->
# 774 "cut.mll"
    (if !phase > 0 then put (lexeme lexbuf) ;
    aargs lexbuf)
# 2548 "cut.ml"

  | 16 ->
# 777 "cut.mll"
    (let head = save_html lexbuf in
    if !phase = 0 then
      html_head := head
    else
      Out.put !out head;
    main lexbuf)
# 2558 "cut.ml"

  | 17 ->
# 784 "cut.mll"
    (let foot =  save_html lexbuf in
    if !phase = 0 then begin
      html_foot := foot
    end ;
    main lexbuf)
# 2567 "cut.ml"

  | 18 ->
# 790 "cut.mll"
    (if !phase = 0 then hevea_footer := true ;
     close_all () ;
     footer lexbuf)
# 2574 "cut.ml"

  | 19 ->
# 794 "cut.mll"
    (let lxm = lexeme lexbuf in
    if !phase = 0 then
      doctype := lxm
    else
      Out.put !out lxm;
    main lexbuf)
# 2584 "cut.ml"

  | 20 ->
# 801 "cut.mll"
    (let lxm = lexeme lexbuf in
    if !phase = 0 then
      html := lxm
    else
      Out.put !out lxm;
    main lexbuf)
# 2594 "cut.ml"

  | 21 ->
# 808 "cut.mll"
    (let lxm = lexeme lexbuf in
    if !phase = 0 then
      body := lxm
    else begin
      Out.put !out lxm ;
      putlinks_start !out !outname
    end ;
    main lexbuf)
# 2606 "cut.ml"

  | 22 ->
# 817 "cut.mll"
    (put (lexeme lexbuf);
      if !phase = 0 then begin
        if verbose > 0 then eprintf "Collect header\n" ;
        collect_header lexbuf
      end else begin
        repeat_header lexbuf
      end ;
      main lexbuf)
# 2618 "cut.ml"

  | 23 ->
# 826 "cut.mll"
    (let lxm = lexeme lexbuf in
    close_all () ;
    if !phase > 0 then begin
      close_top lxm
    end)
# 2627 "cut.ml"

  | 24 ->
let
# 831 "cut.mll"
                    lxm
# 2633 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 832 "cut.mll"
    ( if !phase > 0 then put lxm ;
      intag lexbuf ;
      main lexbuf )
# 2639 "cut.ml"

  | 25 ->
let
# 835 "cut.mll"
        lxm
# 2645 "cut.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 836 "cut.mll"
    (if !phase > 0 then put_char lxm ;
    main lexbuf)
# 2650 "cut.ml"

  | 26 ->
# 839 "cut.mll"
    (raise (Error ("No </body> tag in input file")))
# 2655 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_main_rec lexbuf __ocaml_lex_state

and save_html lexbuf =
   __ocaml_lex_save_html_rec lexbuf 212
and __ocaml_lex_save_html_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 843 "cut.mll"
    (let s = Out.to_string html_buff in
    if verbose > 0 then
      eprintf "save_html -> '%s'\n" s;
    s)
# 2670 "cut.ml"

  | 1 ->
# 848 "cut.mll"
    (let lxm = lexeme_char lexbuf 0 in
    Out.put_char html_buff lxm ;
    save_html lexbuf)
# 2677 "cut.ml"

  | 2 ->
# 852 "cut.mll"
    (raise (Misc.Fatal ("End of file in save_html")))
# 2682 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_save_html_rec lexbuf __ocaml_lex_state

and collect_header lexbuf =
   __ocaml_lex_collect_header_rec lexbuf 228
and __ocaml_lex_collect_header_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 856 "cut.mll"
    (finalize_header () ;
    if verbose > 0 then begin
      eprintf "Header is: '%s'\n" !common_headers
    end)
# 2697 "cut.ml"

  | 1 ->
# 861 "cut.mll"
    (skip_title lexbuf ; collect_header lexbuf)
# 2702 "cut.ml"

  | 2 ->
# 864 "cut.mll"
    (collect_style lexbuf ;  collect_header lexbuf)
# 2707 "cut.ml"

  | 3 ->
let
# 865 "cut.mll"
       lxm
# 2713 "cut.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 866 "cut.mll"
    (adjoin_to_header_char lxm;
    collect_header lexbuf)
# 2718 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_collect_header_rec lexbuf __ocaml_lex_state

and repeat_header lexbuf =
   __ocaml_lex_repeat_header_rec lexbuf 268
and __ocaml_lex_repeat_header_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 870 "cut.mll"
               lxm
# 2731 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_start_pos + 7) in
# 871 "cut.mll"
    (put (!link_style) ; put lxm )
# 2735 "cut.ml"

  | 1 ->
# 873 "cut.mll"
    (skip_style lexbuf ; repeat_header lexbuf)
# 2740 "cut.ml"

  | 2 ->
let
# 874 "cut.mll"
       lxm
# 2746 "cut.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 875 "cut.mll"
    (put_char lxm ; repeat_header lexbuf)
# 2750 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_repeat_header_rec lexbuf __ocaml_lex_state

and collect_style lexbuf =
   __ocaml_lex_collect_style_rec lexbuf 299
and __ocaml_lex_collect_style_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 878 "cut.mll"
                   ( () )
# 2762 "cut.ml"

  | 1 ->
let
# 879 "cut.mll"
       c
# 2768 "cut.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 880 "cut.mll"
    ( Out.put_char style_buff c ; collect_style lexbuf )
# 2772 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_collect_style_rec lexbuf __ocaml_lex_state

and skip_style lexbuf =
   __ocaml_lex_skip_style_rec lexbuf 310
and __ocaml_lex_skip_style_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 883 "cut.mll"
                   ( () )
# 2784 "cut.ml"

  | 1 ->
# 884 "cut.mll"
    ( skip_style lexbuf )
# 2789 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_skip_style_rec lexbuf __ocaml_lex_state

and skip_title lexbuf =
   __ocaml_lex_skip_title_rec lexbuf 321
and __ocaml_lex_skip_title_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 887 "cut.mll"
                    (())
# 2801 "cut.ml"

  | 1 ->
# 888 "cut.mll"
              (skip_title lexbuf)
# 2806 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_skip_title_rec lexbuf __ocaml_lex_state

and footer lexbuf =
   __ocaml_lex_footer_rec lexbuf 332
and __ocaml_lex_footer_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 891 "cut.mll"
                    lxm
# 2819 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 892 "cut.mll"
    (if !phase > 0 then begin
      close_top lxm
    end)
# 2825 "cut.ml"

  | 1 ->
let
# 895 "cut.mll"
        lxm
# 2831 "cut.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 895 "cut.mll"
            (if !phase = 0 then begin Out.put_char hevea_footer_buff lxm end ;
       footer lexbuf)
# 2836 "cut.ml"

  | 2 ->
# 897 "cut.mll"
      (raise (Misc.Fatal ("End of file in footer (no </body> tag)")))
# 2841 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_footer_rec lexbuf __ocaml_lex_state

and tocline lexbuf =
   __ocaml_lex_tocline_rec lexbuf 342
and __ocaml_lex_tocline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 901 "cut.mll"
                 (Out.to_string toc_buf)
# 2853 "cut.ml"

  | 1 ->
# 903 "cut.mll"
    (Out.put_char toc_buf (lexeme_char lexbuf 0) ;
      tocline lexbuf)
# 2859 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_tocline_rec lexbuf __ocaml_lex_state

and flowline lexbuf =
   __ocaml_lex_flowline_rec lexbuf 348
and __ocaml_lex_flowline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 909 "cut.mll"
    (let title = arg lexbuf in
    let _,b = flowline lexbuf in
    title,b)
# 2873 "cut.ml"

  | 1 ->
# 913 "cut.mll"
    (let yes_no = arg lexbuf in
    let b = match yes_no with "YES" -> true | _ -> false in
    let title,_ = flowline lexbuf in
    title,b)
# 2881 "cut.ml"

  | 2 ->
# 917 "cut.mll"
              ("",true)
# 2886 "cut.ml"

  | 3 ->
# 918 "cut.mll"
      (raise (Misc.Fatal "Unclosed comment"))
# 2891 "cut.ml"

  | 4 ->
# 919 "cut.mll"
      (flowline lexbuf)
# 2896 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_flowline_rec lexbuf __ocaml_lex_state

and getargs lexbuf =
   __ocaml_lex_getargs_rec lexbuf 373
and __ocaml_lex_getargs_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 922 "cut.mll"
              ([])
# 2908 "cut.ml"

  | 1 ->
# 924 "cut.mll"
    (let name = argname lexbuf in
    let r = arg lexbuf in
    (name,r)::getargs lexbuf)
# 2915 "cut.ml"

  | 2 ->
# 927 "cut.mll"
      (raise (Misc.Fatal "Unclosed comment"))
# 2920 "cut.ml"

  | 3 ->
# 928 "cut.mll"
      (getargs lexbuf)
# 2925 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_getargs_rec lexbuf __ocaml_lex_state

and argname lexbuf =
   __ocaml_lex_argname_rec lexbuf 384
and __ocaml_lex_argname_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 932 "cut.mll"
    (let lxm = lexeme lexbuf in
    String.sub lxm 0 (String.length lxm-1))
# 2938 "cut.ml"

  | 1 ->
# 934 "cut.mll"
     (raise (Misc.Fatal "arg title"))
# 2943 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_argname_rec lexbuf __ocaml_lex_state

and arg lexbuf =
   __ocaml_lex_arg_rec lexbuf 387
and __ocaml_lex_arg_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 937 "cut.mll"
            (Out.to_string arg_buf)
# 2955 "cut.ml"

  | 1 ->
# 938 "cut.mll"
            (Out.put_char arg_buf (Lexing.lexeme_char lexbuf 0) ; arg lexbuf)
# 2960 "cut.ml"

  | 2 ->
# 939 "cut.mll"
            (raise (Misc.Fatal "Unclosed arg"))
# 2965 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_arg_rec lexbuf __ocaml_lex_state

and aargs lexbuf =
   __ocaml_lex_aargs_rec lexbuf 396
and __ocaml_lex_aargs_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 944 "cut.mll"
  (if !phase = 0 then begin
    let name = refname lexbuf in
    Cross.add name !outname
  end else
    put (lexeme lexbuf) ;
  aargs lexbuf)
# 2982 "cut.ml"

  | 1 ->
# 951 "cut.mll"
   (if !phase > 0 then begin
      let lxm = lexeme lexbuf in
      let name = refname lexbuf in
      try
        let newname =
          if String.length name > 0 && String.get name 0 = '#' then
            let frag = String.sub name 1 (String.length name-1) in
            Cross.fullname check_changed !outname frag
          else name in
        put lxm ;
        put "\"" ;
        put newname ;
        put "\""
      with Not_found -> ()
    end ;
    aargs lexbuf)
# 3002 "cut.ml"

  | 2 ->
# 968 "cut.mll"
  (if !phase > 0 then put_char '>' ;
  main lexbuf)
# 3008 "cut.ml"

  | 3 ->
let
# 970 "cut.mll"
       c
# 3014 "cut.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 971 "cut.mll"
  (if !phase > 0 then put_char c ;
  aargs lexbuf)
# 3019 "cut.ml"

  | 4 ->
# 974 "cut.mll"
  (raise (Error "Bad <A ...> tag"))
# 3024 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_aargs_rec lexbuf __ocaml_lex_state

and intag lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 4 (-1); __ocaml_lex_intag_rec lexbuf 412
and __ocaml_lex_intag_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 977 "cut.mll"
            lxm
# 3037 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 978 "cut.mll"
    ( if !phase > 0 then put lxm ;
      intag lexbuf )
# 3042 "cut.ml"

  | 1 ->
let
# 981 "cut.mll"
                      v
# 3048 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_mem.(0)
and
# 983 "cut.mll"
                                                            lxm
# 3053 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 985 "cut.mll"
  ( if !phase = 0 then Cross.add v !outname else put lxm ;
    intag lexbuf )
# 3058 "cut.ml"

  | 2 ->
let
# 990 "cut.mll"
                                                      lxm
# 3064 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 992 "cut.mll"
  ( if !phase > 0 then put lxm ;
    intag lexbuf )
# 3069 "cut.ml"

  | 3 ->
let
# 994 "cut.mll"
              lxm
# 3075 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 994 "cut.mll"
                  ( if !phase > 0 then put lxm  )
# 3079 "cut.ml"

  | 4 ->
# 995 "cut.mll"
      ( raise (Error "intag: attribute syntax"))
# 3084 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_intag_rec lexbuf __ocaml_lex_state

and refname lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 4 (-1);(* L=1 [2] <- p ;  *)
  lexbuf.Lexing.lex_mem.(2) <- lexbuf.Lexing.lex_curr_pos ;
 __ocaml_lex_refname_rec lexbuf 434
and __ocaml_lex_refname_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 998 "cut.mll"
                   name
# 3099 "cut.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_mem.(0) in
# 1002 "cut.mll"
   ( name )
# 3103 "cut.ml"

  | 1 ->
# 1003 "cut.mll"
     (raise (Error "Bad reference name syntax"))
# 3108 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_refname_rec lexbuf __ocaml_lex_state

and skip_endcom lexbuf =
   __ocaml_lex_skip_endcom_rec lexbuf 439
and __ocaml_lex_skip_endcom_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 1006 "cut.mll"
                   (())
# 3120 "cut.ml"

  | 1 ->
# 1007 "cut.mll"
                   (raise (Error "Bad HTML comment syntax"))
# 3125 "cut.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_skip_endcom_rec lexbuf __ocaml_lex_state

;;

# 1009 "cut.mll"
 

let do_lex lexbuff =
  main lexbuff ;
  !some_links

end

# 3141 "cut.ml"
