/*********************                                                        */
/*! \file options_template.cpp
 ** \verbatim
 ** Top contributors (to current version):
 **   Morgan Deters, Tim King, Mathias Preiner
 ** This file is part of the CVC4 project.
 ** Copyright (c) 2009-2020 by the authors listed in the file AUTHORS
 ** in the top-level source directory) and their institutional affiliations.
 ** All rights reserved.  See the file COPYING in the top-level source
 ** directory for licensing information.\endverbatim
 **
 ** \brief Contains code for handling command-line options.
 **
 ** Contains code for handling command-line options
 **/

#if !defined(_BSD_SOURCE) && defined(__MINGW32__) && !defined(__MINGW64__)
// force use of optreset; mingw32 croaks on argv-switching otherwise
#  include "cvc4autoconfig.h"
#  define _BSD_SOURCE
#  undef HAVE_DECL_OPTRESET
#  define HAVE_DECL_OPTRESET 1
#  define CVC4_IS_NOT_REALLY_BSD
#endif /* !_BSD_SOURCE && __MINGW32__ && !__MINGW64__ */

#ifdef __MINGW64__
extern int optreset;
#endif /* __MINGW64__ */

#include <getopt.h>

// clean up
#ifdef CVC4_IS_NOT_REALLY_BSD
#  undef _BSD_SOURCE
#endif /* CVC4_IS_NOT_REALLY_BSD */

#include <unistd.h>
#include <string.h>
#include <stdint.h>
#include <time.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iomanip>
#include <new>
#include <string>
#include <sstream>
#include <limits>

#include "base/check.h"
#include "base/exception.h"
#include "base/output.h"
#include "options/didyoumean.h"
#include "options/language.h"
#include "options/options_handler.h"

#include "options/arith_options.h"
#include "options/arrays_options.h"
#include "options/base_options.h"
#include "options/booleans_options.h"
#include "options/builtin_options.h"
#include "options/bv_options.h"
#include "options/datatypes_options.h"
#include "options/decision_options.h"
#include "options/expr_options.h"
#include "options/fp_options.h"
#include "options/main_options.h"
#include "options/parser_options.h"
#include "options/printer_options.h"
#include "options/proof_options.h"
#include "options/prop_options.h"
#include "options/quantifiers_options.h"
#include "options/sep_options.h"
#include "options/sets_options.h"
#include "options/smt_options.h"
#include "options/strings_options.h"
#include "options/theory_options.h"
#include "options/uf_options.h"


#include "options/options_holder.h"
#include "cvc4autoconfig.h"
#include "options/base_handlers.h"




using namespace CVC4;
using namespace CVC4::options;

namespace CVC4 {

thread_local Options* Options::s_current = NULL;

/**
 * This is a default handler for options of built-in C++ type.  This
 * template is really just a helper for the handleOption() template,
 * below.  Variants of this template handle numeric and non-numeric,
 * integral and non-integral, signed and unsigned C++ types.
 * handleOption() makes sure to instantiate the right one.
 *
 * This implements default behavior when e.g. an option is
 * unsigned but the user specifies a negative argument; etc.
 */
template <class T, bool is_numeric, bool is_integer>
struct OptionHandler {
  static T handle(std::string option, std::string optionarg);
};/* struct OptionHandler<> */

/** Variant for integral C++ types */
template <class T>
struct OptionHandler<T, true, true> {
  static bool stringToInt(T& t, const std::string& str) {
    std::istringstream ss(str);
    ss >> t;
    char tmp;
    return !(ss.fail() || ss.get(tmp));
  }

  static bool containsMinus(const std::string& str) {
    return str.find('-') != std::string::npos;
  }

  static T handle(const std::string& option, const std::string& optionarg) {
    try {
      T i;
      bool success = stringToInt(i, optionarg);

      if(!success){
        throw OptionException(option + ": failed to parse "+ optionarg +
                              " as an integer of the appropriate type.");
      }

      // Depending in the platform unsigned numbers with '-' signs may parse.
      // Reject these by looking for any minus if it is not signed.
      if( (! std::numeric_limits<T>::is_signed) && containsMinus(optionarg) ) {
        // unsigned type but user gave negative argument
        throw OptionException(option + " requires a nonnegative argument");
      } else if(i < std::numeric_limits<T>::min()) {
        // negative overflow for type
        std::stringstream ss;
        ss << option << " requires an argument >= "
           << std::numeric_limits<T>::min();
        throw OptionException(ss.str());
      } else if(i > std::numeric_limits<T>::max()) {
        // positive overflow for type
        std::stringstream ss;
        ss << option << " requires an argument <= "
           << std::numeric_limits<T>::max();
        throw OptionException(ss.str());
      }

      return i;

      // if(std::numeric_limits<T>::is_signed) {
      //   return T(i.getLong());
      // } else {
      //   return T(i.getUnsignedLong());
      // }
    } catch(std::invalid_argument&) {
      // user gave something other than an integer
      throw OptionException(option + " requires an integer argument");
    }
  }
};/* struct OptionHandler<T, true, true> */

/** Variant for numeric but non-integral C++ types */
template <class T>
struct OptionHandler<T, true, false> {
  static T handle(std::string option, std::string optionarg) {
    std::stringstream in(optionarg);
    long double r;
    in >> r;
    if(! in.eof()) {
      // we didn't consume the whole string (junk at end)
      throw OptionException(option + " requires a numeric argument");
    }

    if(! std::numeric_limits<T>::is_signed && r < 0.0) {
      // unsigned type but user gave negative value
      throw OptionException(option + " requires a nonnegative argument");
    } else if(r < -std::numeric_limits<T>::max()) {
      // negative overflow for type
      std::stringstream ss;
      ss << option << " requires an argument >= "
         << -std::numeric_limits<T>::max();
      throw OptionException(ss.str());
    } else if(r > std::numeric_limits<T>::max()) {
      // positive overflow for type
      std::stringstream ss;
      ss << option << " requires an argument <= "
         << std::numeric_limits<T>::max();
      throw OptionException(ss.str());
    }

    return T(r);
  }
};/* struct OptionHandler<T, true, false> */

/** Variant for non-numeric C++ types */
template <class T>
struct OptionHandler<T, false, false> {
  static T handle(std::string option, std::string optionarg) {
    T::unsupported_handleOption_call___please_write_me;
    // The above line causes a compiler error if this version of the template
    // is ever instantiated (meaning that a specialization is missing).  So
    // don't worry about the segfault in the next line, the "return" is only
    // there to keep the compiler from giving additional, distracting errors
    // and warnings.
    return *(T*)0;
  }
};/* struct OptionHandler<T, false, false> */

/** Handle an option of type T in the default way. */
template <class T>
T handleOption(std::string option, std::string optionarg) {
  return OptionHandler<T, std::numeric_limits<T>::is_specialized, std::numeric_limits<T>::is_integer>::handle(option, optionarg);
}

/** Handle an option of type std::string in the default way. */
template <>
std::string handleOption<std::string>(std::string option, std::string optionarg) {
  return optionarg;
}

/**
 * Run handler, and any user-given predicates, for option T.
 * If a user specifies a :handler or :predicates, it overrides this.
 */
template <class T>
typename T::type runHandlerAndPredicates(T, std::string option, std::string optionarg, options::OptionsHandler* handler) {
  // By default, parse the option argument in a way appropriate for its type.
  // E.g., for "unsigned int" options, ensure that the provided argument is
  // a nonnegative integer that fits in the unsigned int type.

  return handleOption<typename T::type>(option, optionarg);
}

template <class T>
void runBoolPredicates(T, std::string option, bool b, options::OptionsHandler* handler) {
  // By default, nothing to do for bool.  Users add things with
  // :predicate in options files to provide custom checking routines
  // that can throw exceptions.
}


Options::Options()
    : d_holder(new options::OptionsHolder())
    , d_handler(new options::OptionsHandler(this))
    , d_beforeSearchListeners()
    , d_tlimitListeners()
    , d_tlimitPerListeners()
    , d_rlimitListeners()
    , d_rlimitPerListeners()
{}

Options::~Options() {
  delete d_handler;
  delete d_holder;
}

void Options::copyValues(const Options& options){
  if(this != &options) {
    delete d_holder;
    d_holder = new options::OptionsHolder(*options.d_holder);
  }
}

std::string Options::formatThreadOptionException(const std::string& option) {
  std::stringstream ss;
  ss << "can't understand option `" << option
     << "': expected something like --threadN=\"--option1 --option2\","
     << " where N is a nonnegative integer";
  return ss.str();
}

ListenerCollection::Registration* Options::registerAndNotify(
    ListenerCollection& collection, Listener* listener, bool notify)
{
  ListenerCollection::Registration* registration =
      collection.registerListener(listener);
  if(notify) {
    try
    {
      listener->notify();
    }
    catch (OptionException& e)
    {
      // It can happen that listener->notify() throws an OptionException. In
      // that case, we have to make sure that we delete the registration of our
      // listener before rethrowing the exception. Otherwise the
      // ListenerCollection deconstructor will complain that not all
      // registrations have been removed before invoking the deconstructor.
      delete registration;
      throw OptionException(e.getRawMessage());
    }
  }
  return registration;
}

ListenerCollection::Registration* Options::registerBeforeSearchListener(
   Listener* listener)
{
  return d_beforeSearchListeners.registerListener(listener);
}

ListenerCollection::Registration* Options::registerTlimitListener(
   Listener* listener, bool notifyIfSet)
{
  bool notify = notifyIfSet &&
      wasSetByUser(options::cumulativeMillisecondLimit);
  return registerAndNotify(d_tlimitListeners, listener, notify);
}

ListenerCollection::Registration* Options::registerTlimitPerListener(
   Listener* listener, bool notifyIfSet)
{
  bool notify = notifyIfSet && wasSetByUser(options::perCallMillisecondLimit);
  return registerAndNotify(d_tlimitPerListeners, listener, notify);
}

ListenerCollection::Registration* Options::registerRlimitListener(
   Listener* listener, bool notifyIfSet)
{
  bool notify = notifyIfSet && wasSetByUser(options::cumulativeResourceLimit);
  return registerAndNotify(d_rlimitListeners, listener, notify);
}

ListenerCollection::Registration* Options::registerRlimitPerListener(
   Listener* listener, bool notifyIfSet)
{
  bool notify = notifyIfSet && wasSetByUser(options::perCallResourceLimit);
  return registerAndNotify(d_rlimitPerListeners, listener, notify);
}

ListenerCollection::Registration* Options::registerSetDefaultExprDepthListener(
    Listener* listener, bool notifyIfSet)
{
  bool notify = notifyIfSet && wasSetByUser(options::defaultExprDepth);
  return registerAndNotify(d_setDefaultExprDepthListeners, listener, notify);
}

ListenerCollection::Registration* Options::registerSetDefaultExprDagListener(
    Listener* listener, bool notifyIfSet)
{
  bool notify = notifyIfSet && wasSetByUser(options::defaultDagThresh);
  return registerAndNotify(d_setDefaultDagThreshListeners, listener, notify);
}

ListenerCollection::Registration* Options::registerSetPrintExprTypesListener(
    Listener* listener, bool notifyIfSet)
{
  bool notify = notifyIfSet && wasSetByUser(options::printExprTypes);
  return registerAndNotify(d_setPrintExprTypesListeners, listener, notify);
}

ListenerCollection::Registration* Options::registerSetDumpModeListener(
    Listener* listener, bool notifyIfSet)
{
  bool notify = notifyIfSet && wasSetByUser(options::dumpModeString);
  return registerAndNotify(d_setDumpModeListeners, listener, notify);
}

ListenerCollection::Registration* Options::registerSetPrintSuccessListener(
    Listener* listener, bool notifyIfSet)
{
  bool notify = notifyIfSet && wasSetByUser(options::printSuccess);
  return registerAndNotify(d_setPrintSuccessListeners, listener, notify);
}

ListenerCollection::Registration* Options::registerDumpToFileNameListener(
    Listener* listener, bool notifyIfSet)
{
  bool notify = notifyIfSet && wasSetByUser(options::dumpToFileName);
  return registerAndNotify(d_dumpToFileListeners, listener, notify);
}

ListenerCollection::Registration*
Options::registerSetRegularOutputChannelListener(
    Listener* listener, bool notifyIfSet)
{
  bool notify = notifyIfSet && wasSetByUser(options::regularChannelName);
  return registerAndNotify(d_setRegularChannelListeners, listener, notify);
}

ListenerCollection::Registration*
Options::registerSetDiagnosticOutputChannelListener(
    Listener* listener, bool notifyIfSet)
{
  bool notify = notifyIfSet && wasSetByUser(options::diagnosticChannelName);
  return registerAndNotify(d_setDiagnosticChannelListeners, listener, notify);
}

template <> options::maxApproxDepth__option_t::type runHandlerAndPredicates(
    options::maxApproxDepth__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::maxApproxDepth__option_t::type retval = handleOption<int16_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::maxApproxDepth__option_t,
    std::string option,
    std::string value)
{
  d_holder->maxApproxDepth =
    runHandlerAndPredicates(options::maxApproxDepth, option, value, d_handler);
  d_holder->maxApproxDepth__setByUser__ = true;
  Trace("options") << "user assigned option maxApproxDepth" << std::endl;
  
}
template <> void Options::assignBool(
    options::brabTest__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::brabTest, option, value, d_handler);
  d_holder->brabTest = value;
  d_holder->brabTest__setByUser__ = true;
  Trace("options") << "user assigned option brabTest" << std::endl;
  
}
template <> void Options::assignBool(
    options::arithNoPartialFun__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::arithNoPartialFun, option, value, d_handler);
  d_holder->arithNoPartialFun = value;
  d_holder->arithNoPartialFun__setByUser__ = true;
  Trace("options") << "user assigned option arithNoPartialFun" << std::endl;
  
}
template <> options::arithPropAsLemmaLength__option_t::type runHandlerAndPredicates(
    options::arithPropAsLemmaLength__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::arithPropAsLemmaLength__option_t::type retval = handleOption<uint16_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::arithPropAsLemmaLength__option_t,
    std::string option,
    std::string value)
{
  d_holder->arithPropAsLemmaLength =
    runHandlerAndPredicates(options::arithPropAsLemmaLength, option, value, d_handler);
  d_holder->arithPropAsLemmaLength__setByUser__ = true;
  Trace("options") << "user assigned option arithPropAsLemmaLength" << std::endl;
  
}
template <> options::arithPropagationMode__option_t::type runHandlerAndPredicates(
    options::arithPropagationMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::arithPropagationMode__option_t::type retval = stringToArithPropagationMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::arithPropagationMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->arithPropagationMode =
    runHandlerAndPredicates(options::arithPropagationMode, option, value, d_handler);
  d_holder->arithPropagationMode__setByUser__ = true;
  Trace("options") << "user assigned option arithPropagationMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::arithRewriteEq__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::arithRewriteEq, option, value, d_handler);
  d_holder->arithRewriteEq = value;
  d_holder->arithRewriteEq__setByUser__ = true;
  Trace("options") << "user assigned option arithRewriteEq" << std::endl;
  
}
template <> void Options::assignBool(
    options::collectPivots__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::collectPivots, option, value, d_handler);
  d_holder->collectPivots = value;
  d_holder->collectPivots__setByUser__ = true;
  Trace("options") << "user assigned option collectPivots" << std::endl;
  
}
template <> void Options::assignBool(
    options::doCutAllBounded__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::doCutAllBounded, option, value, d_handler);
  d_holder->doCutAllBounded = value;
  d_holder->doCutAllBounded__setByUser__ = true;
  Trace("options") << "user assigned option doCutAllBounded" << std::endl;
  
}
template <> void Options::assignBool(
    options::exportDioDecompositions__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::exportDioDecompositions, option, value, d_handler);
  d_holder->exportDioDecompositions = value;
  d_holder->exportDioDecompositions__setByUser__ = true;
  Trace("options") << "user assigned option exportDioDecompositions" << std::endl;
  
}
template <> void Options::assignBool(
    options::dioRepeat__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dioRepeat, option, value, d_handler);
  d_holder->dioRepeat = value;
  d_holder->dioRepeat__setByUser__ = true;
  Trace("options") << "user assigned option dioRepeat" << std::endl;
  
}
template <> void Options::assignBool(
    options::arithDioSolver__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::arithDioSolver, option, value, d_handler);
  d_holder->arithDioSolver = value;
  d_holder->arithDioSolver__setByUser__ = true;
  Trace("options") << "user assigned option arithDioSolver" << std::endl;
  
}
template <> options::dioSolverTurns__option_t::type runHandlerAndPredicates(
    options::dioSolverTurns__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::dioSolverTurns__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::dioSolverTurns__option_t,
    std::string option,
    std::string value)
{
  d_holder->dioSolverTurns =
    runHandlerAndPredicates(options::dioSolverTurns, option, value, d_handler);
  d_holder->dioSolverTurns__setByUser__ = true;
  Trace("options") << "user assigned option dioSolverTurns" << std::endl;
  
}
template <> options::arithErrorSelectionRule__option_t::type runHandlerAndPredicates(
    options::arithErrorSelectionRule__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::arithErrorSelectionRule__option_t::type retval = stringToErrorSelectionRule(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::arithErrorSelectionRule__option_t,
    std::string option,
    std::string value)
{
  d_holder->arithErrorSelectionRule =
    runHandlerAndPredicates(options::arithErrorSelectionRule, option, value, d_handler);
  d_holder->arithErrorSelectionRule__setByUser__ = true;
  Trace("options") << "user assigned option arithErrorSelectionRule" << std::endl;
  
}
template <> void Options::assignBool(
    options::havePenalties__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::havePenalties, option, value, d_handler);
  d_holder->havePenalties = value;
  d_holder->havePenalties__setByUser__ = true;
  Trace("options") << "user assigned option havePenalties" << std::endl;
  
}
template <> options::arithHeuristicPivots__option_t::type runHandlerAndPredicates(
    options::arithHeuristicPivots__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::arithHeuristicPivots__option_t::type retval = handleOption<int16_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::arithHeuristicPivots__option_t,
    std::string option,
    std::string value)
{
  d_holder->arithHeuristicPivots =
    runHandlerAndPredicates(options::arithHeuristicPivots, option, value, d_handler);
  d_holder->arithHeuristicPivots__setByUser__ = true;
  Trace("options") << "user assigned option arithHeuristicPivots" << std::endl;
  
}
template <> void Options::assignBool(
    options::replayFailureLemma__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::replayFailureLemma, option, value, d_handler);
  d_holder->replayFailureLemma = value;
  d_holder->replayFailureLemma__setByUser__ = true;
  Trace("options") << "user assigned option replayFailureLemma" << std::endl;
  
}
template <> options::maxCutsInContext__option_t::type runHandlerAndPredicates(
    options::maxCutsInContext__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::maxCutsInContext__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::maxCutsInContext__option_t,
    std::string option,
    std::string value)
{
  d_holder->maxCutsInContext =
    runHandlerAndPredicates(options::maxCutsInContext, option, value, d_handler);
  d_holder->maxCutsInContext__setByUser__ = true;
  Trace("options") << "user assigned option maxCutsInContext" << std::endl;
  
}
template <> void Options::assignBool(
    options::arithMLTrick__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::arithMLTrick, option, value, d_handler);
  d_holder->arithMLTrick = value;
  d_holder->arithMLTrick__setByUser__ = true;
  Trace("options") << "user assigned option arithMLTrick" << std::endl;
  
}
template <> options::arithMLTrickSubstitutions__option_t::type runHandlerAndPredicates(
    options::arithMLTrickSubstitutions__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::arithMLTrickSubstitutions__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::arithMLTrickSubstitutions__option_t,
    std::string option,
    std::string value)
{
  d_holder->arithMLTrickSubstitutions =
    runHandlerAndPredicates(options::arithMLTrickSubstitutions, option, value, d_handler);
  d_holder->arithMLTrickSubstitutions__setByUser__ = true;
  Trace("options") << "user assigned option arithMLTrickSubstitutions" << std::endl;
  
}
template <> void Options::assignBool(
    options::newProp__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::newProp, option, value, d_handler);
  d_holder->newProp = value;
  d_holder->newProp__setByUser__ = true;
  Trace("options") << "user assigned option newProp" << std::endl;
  
}
template <> void Options::assignBool(
    options::nlExt__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::nlExt, option, value, d_handler);
  d_holder->nlExt = value;
  d_holder->nlExt__setByUser__ = true;
  Trace("options") << "user assigned option nlExt" << std::endl;
  
}
template <> void Options::assignBool(
    options::nlExtEntailConflicts__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::nlExtEntailConflicts, option, value, d_handler);
  d_holder->nlExtEntailConflicts = value;
  d_holder->nlExtEntailConflicts__setByUser__ = true;
  Trace("options") << "user assigned option nlExtEntailConflicts" << std::endl;
  
}
template <> void Options::assignBool(
    options::nlExtFactor__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::nlExtFactor, option, value, d_handler);
  d_holder->nlExtFactor = value;
  d_holder->nlExtFactor__setByUser__ = true;
  Trace("options") << "user assigned option nlExtFactor" << std::endl;
  
}
template <> void Options::assignBool(
    options::nlExtIncPrecision__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::nlExtIncPrecision, option, value, d_handler);
  d_holder->nlExtIncPrecision = value;
  d_holder->nlExtIncPrecision__setByUser__ = true;
  Trace("options") << "user assigned option nlExtIncPrecision" << std::endl;
  
}
template <> void Options::assignBool(
    options::nlExtPurify__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::nlExtPurify, option, value, d_handler);
  d_holder->nlExtPurify = value;
  d_holder->nlExtPurify__setByUser__ = true;
  Trace("options") << "user assigned option nlExtPurify" << std::endl;
  
}
template <> void Options::assignBool(
    options::nlExtResBound__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::nlExtResBound, option, value, d_handler);
  d_holder->nlExtResBound = value;
  d_holder->nlExtResBound__setByUser__ = true;
  Trace("options") << "user assigned option nlExtResBound" << std::endl;
  
}
template <> void Options::assignBool(
    options::nlExtRewrites__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::nlExtRewrites, option, value, d_handler);
  d_holder->nlExtRewrites = value;
  d_holder->nlExtRewrites__setByUser__ = true;
  Trace("options") << "user assigned option nlExtRewrites" << std::endl;
  
}
template <> void Options::assignBool(
    options::nlExtSplitZero__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::nlExtSplitZero, option, value, d_handler);
  d_holder->nlExtSplitZero = value;
  d_holder->nlExtSplitZero__setByUser__ = true;
  Trace("options") << "user assigned option nlExtSplitZero" << std::endl;
  
}
template <> options::nlExtTfTaylorDegree__option_t::type runHandlerAndPredicates(
    options::nlExtTfTaylorDegree__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::nlExtTfTaylorDegree__option_t::type retval = handleOption<int16_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::nlExtTfTaylorDegree__option_t,
    std::string option,
    std::string value)
{
  d_holder->nlExtTfTaylorDegree =
    runHandlerAndPredicates(options::nlExtTfTaylorDegree, option, value, d_handler);
  d_holder->nlExtTfTaylorDegree__setByUser__ = true;
  Trace("options") << "user assigned option nlExtTfTaylorDegree" << std::endl;
  
}
template <> void Options::assignBool(
    options::nlExtTfTangentPlanes__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::nlExtTfTangentPlanes, option, value, d_handler);
  d_holder->nlExtTfTangentPlanes = value;
  d_holder->nlExtTfTangentPlanes__setByUser__ = true;
  Trace("options") << "user assigned option nlExtTfTangentPlanes" << std::endl;
  
}
template <> void Options::assignBool(
    options::nlExtTangentPlanes__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::nlExtTangentPlanes, option, value, d_handler);
  d_holder->nlExtTangentPlanes = value;
  d_holder->nlExtTangentPlanes__setByUser__ = true;
  Trace("options") << "user assigned option nlExtTangentPlanes" << std::endl;
  
}
template <> void Options::assignBool(
    options::nlExtTangentPlanesInterleave__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::nlExtTangentPlanesInterleave, option, value, d_handler);
  d_holder->nlExtTangentPlanesInterleave = value;
  d_holder->nlExtTangentPlanesInterleave__setByUser__ = true;
  Trace("options") << "user assigned option nlExtTangentPlanesInterleave" << std::endl;
  
}
template <> void Options::assignBool(
    options::pbRewrites__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::pbRewrites, option, value, d_handler);
  d_holder->pbRewrites = value;
  d_holder->pbRewrites__setByUser__ = true;
  Trace("options") << "user assigned option pbRewrites" << std::endl;
  
}
template <> options::arithPivotThreshold__option_t::type runHandlerAndPredicates(
    options::arithPivotThreshold__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::arithPivotThreshold__option_t::type retval = handleOption<uint16_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::arithPivotThreshold__option_t,
    std::string option,
    std::string value)
{
  d_holder->arithPivotThreshold =
    runHandlerAndPredicates(options::arithPivotThreshold, option, value, d_handler);
  d_holder->arithPivotThreshold__setByUser__ = true;
  Trace("options") << "user assigned option arithPivotThreshold" << std::endl;
  
}
template <> options::ppAssertMaxSubSize__option_t::type runHandlerAndPredicates(
    options::ppAssertMaxSubSize__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::ppAssertMaxSubSize__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::ppAssertMaxSubSize__option_t,
    std::string option,
    std::string value)
{
  d_holder->ppAssertMaxSubSize =
    runHandlerAndPredicates(options::ppAssertMaxSubSize, option, value, d_handler);
  d_holder->ppAssertMaxSubSize__setByUser__ = true;
  Trace("options") << "user assigned option ppAssertMaxSubSize" << std::endl;
  
}
template <> options::arithPropagateMaxLength__option_t::type runHandlerAndPredicates(
    options::arithPropagateMaxLength__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::arithPropagateMaxLength__option_t::type retval = handleOption<uint16_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::arithPropagateMaxLength__option_t,
    std::string option,
    std::string value)
{
  d_holder->arithPropagateMaxLength =
    runHandlerAndPredicates(options::arithPropagateMaxLength, option, value, d_handler);
  d_holder->arithPropagateMaxLength__setByUser__ = true;
  Trace("options") << "user assigned option arithPropagateMaxLength" << std::endl;
  
}
template <> options::replayEarlyCloseDepths__option_t::type runHandlerAndPredicates(
    options::replayEarlyCloseDepths__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::replayEarlyCloseDepths__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::replayEarlyCloseDepths__option_t,
    std::string option,
    std::string value)
{
  d_holder->replayEarlyCloseDepths =
    runHandlerAndPredicates(options::replayEarlyCloseDepths, option, value, d_handler);
  d_holder->replayEarlyCloseDepths__setByUser__ = true;
  Trace("options") << "user assigned option replayEarlyCloseDepths" << std::endl;
  
}
template <> options::replayFailurePenalty__option_t::type runHandlerAndPredicates(
    options::replayFailurePenalty__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::replayFailurePenalty__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::replayFailurePenalty__option_t,
    std::string option,
    std::string value)
{
  d_holder->replayFailurePenalty =
    runHandlerAndPredicates(options::replayFailurePenalty, option, value, d_handler);
  d_holder->replayFailurePenalty__setByUser__ = true;
  Trace("options") << "user assigned option replayFailurePenalty" << std::endl;
  
}
template <> options::lemmaRejectCutSize__option_t::type runHandlerAndPredicates(
    options::lemmaRejectCutSize__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::lemmaRejectCutSize__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::lemmaRejectCutSize__option_t,
    std::string option,
    std::string value)
{
  d_holder->lemmaRejectCutSize =
    runHandlerAndPredicates(options::lemmaRejectCutSize, option, value, d_handler);
  d_holder->lemmaRejectCutSize__setByUser__ = true;
  Trace("options") << "user assigned option lemmaRejectCutSize" << std::endl;
  
}
template <> options::replayNumericFailurePenalty__option_t::type runHandlerAndPredicates(
    options::replayNumericFailurePenalty__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::replayNumericFailurePenalty__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::replayNumericFailurePenalty__option_t,
    std::string option,
    std::string value)
{
  d_holder->replayNumericFailurePenalty =
    runHandlerAndPredicates(options::replayNumericFailurePenalty, option, value, d_handler);
  d_holder->replayNumericFailurePenalty__setByUser__ = true;
  Trace("options") << "user assigned option replayNumericFailurePenalty" << std::endl;
  
}
template <> options::replayRejectCutSize__option_t::type runHandlerAndPredicates(
    options::replayRejectCutSize__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::replayRejectCutSize__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::replayRejectCutSize__option_t,
    std::string option,
    std::string value)
{
  d_holder->replayRejectCutSize =
    runHandlerAndPredicates(options::replayRejectCutSize, option, value, d_handler);
  d_holder->replayRejectCutSize__setByUser__ = true;
  Trace("options") << "user assigned option replayRejectCutSize" << std::endl;
  
}
template <> options::soiApproxMajorFailurePen__option_t::type runHandlerAndPredicates(
    options::soiApproxMajorFailurePen__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::soiApproxMajorFailurePen__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::soiApproxMajorFailurePen__option_t,
    std::string option,
    std::string value)
{
  d_holder->soiApproxMajorFailurePen =
    runHandlerAndPredicates(options::soiApproxMajorFailurePen, option, value, d_handler);
  d_holder->soiApproxMajorFailurePen__setByUser__ = true;
  Trace("options") << "user assigned option soiApproxMajorFailurePen" << std::endl;
  
}
template <> options::soiApproxMajorFailure__option_t::type runHandlerAndPredicates(
    options::soiApproxMajorFailure__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::soiApproxMajorFailure__option_t::type retval = handleOption<double>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::soiApproxMajorFailure__option_t,
    std::string option,
    std::string value)
{
  d_holder->soiApproxMajorFailure =
    runHandlerAndPredicates(options::soiApproxMajorFailure, option, value, d_handler);
  d_holder->soiApproxMajorFailure__setByUser__ = true;
  Trace("options") << "user assigned option soiApproxMajorFailure" << std::endl;
  
}
template <> options::soiApproxMinorFailurePen__option_t::type runHandlerAndPredicates(
    options::soiApproxMinorFailurePen__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::soiApproxMinorFailurePen__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::soiApproxMinorFailurePen__option_t,
    std::string option,
    std::string value)
{
  d_holder->soiApproxMinorFailurePen =
    runHandlerAndPredicates(options::soiApproxMinorFailurePen, option, value, d_handler);
  d_holder->soiApproxMinorFailurePen__setByUser__ = true;
  Trace("options") << "user assigned option soiApproxMinorFailurePen" << std::endl;
  
}
template <> options::soiApproxMinorFailure__option_t::type runHandlerAndPredicates(
    options::soiApproxMinorFailure__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::soiApproxMinorFailure__option_t::type retval = handleOption<double>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::soiApproxMinorFailure__option_t,
    std::string option,
    std::string value)
{
  d_holder->soiApproxMinorFailure =
    runHandlerAndPredicates(options::soiApproxMinorFailure, option, value, d_handler);
  d_holder->soiApproxMinorFailure__setByUser__ = true;
  Trace("options") << "user assigned option soiApproxMinorFailure" << std::endl;
  
}
template <> void Options::assignBool(
    options::restrictedPivots__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::restrictedPivots, option, value, d_handler);
  d_holder->restrictedPivots = value;
  d_holder->restrictedPivots__setByUser__ = true;
  Trace("options") << "user assigned option restrictedPivots" << std::endl;
  
}
template <> void Options::assignBool(
    options::revertArithModels__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::revertArithModels, option, value, d_handler);
  d_holder->revertArithModels = value;
  d_holder->revertArithModels__setByUser__ = true;
  Trace("options") << "user assigned option revertArithModels" << std::endl;
  
}
template <> options::rrTurns__option_t::type runHandlerAndPredicates(
    options::rrTurns__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::rrTurns__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::rrTurns__option_t,
    std::string option,
    std::string value)
{
  d_holder->rrTurns =
    runHandlerAndPredicates(options::rrTurns, option, value, d_handler);
  d_holder->rrTurns__setByUser__ = true;
  Trace("options") << "user assigned option rrTurns" << std::endl;
  
}
template <> void Options::assignBool(
    options::trySolveIntStandardEffort__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::trySolveIntStandardEffort, option, value, d_handler);
  d_holder->trySolveIntStandardEffort = value;
  d_holder->trySolveIntStandardEffort__setByUser__ = true;
  Trace("options") << "user assigned option trySolveIntStandardEffort" << std::endl;
  
}
template <> options::arithSimplexCheckPeriod__option_t::type runHandlerAndPredicates(
    options::arithSimplexCheckPeriod__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::arithSimplexCheckPeriod__option_t::type retval = handleOption<uint16_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::arithSimplexCheckPeriod__option_t,
    std::string option,
    std::string value)
{
  d_holder->arithSimplexCheckPeriod =
    runHandlerAndPredicates(options::arithSimplexCheckPeriod, option, value, d_handler);
  d_holder->arithSimplexCheckPeriod__setByUser__ = true;
  Trace("options") << "user assigned option arithSimplexCheckPeriod" << std::endl;
  
}
template <> void Options::assignBool(
    options::soiQuickExplain__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::soiQuickExplain, option, value, d_handler);
  d_holder->soiQuickExplain = value;
  d_holder->soiQuickExplain__setByUser__ = true;
  Trace("options") << "user assigned option soiQuickExplain" << std::endl;
  
}
template <> options::arithStandardCheckVarOrderPivots__option_t::type runHandlerAndPredicates(
    options::arithStandardCheckVarOrderPivots__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::arithStandardCheckVarOrderPivots__option_t::type retval = handleOption<int16_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::arithStandardCheckVarOrderPivots__option_t,
    std::string option,
    std::string value)
{
  d_holder->arithStandardCheckVarOrderPivots =
    runHandlerAndPredicates(options::arithStandardCheckVarOrderPivots, option, value, d_handler);
  d_holder->arithStandardCheckVarOrderPivots__setByUser__ = true;
  Trace("options") << "user assigned option arithStandardCheckVarOrderPivots" << std::endl;
  
}
template <> options::arithUnateLemmaMode__option_t::type runHandlerAndPredicates(
    options::arithUnateLemmaMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::arithUnateLemmaMode__option_t::type retval = stringToArithUnateLemmaMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::arithUnateLemmaMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->arithUnateLemmaMode =
    runHandlerAndPredicates(options::arithUnateLemmaMode, option, value, d_handler);
  d_holder->arithUnateLemmaMode__setByUser__ = true;
  Trace("options") << "user assigned option arithUnateLemmaMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::useApprox__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::useApprox, option, value, d_handler);
  d_holder->useApprox = value;
  d_holder->useApprox__setByUser__ = true;
  Trace("options") << "user assigned option useApprox" << std::endl;
  
}
template <> void Options::assignBool(
    options::useFC__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::useFC, option, value, d_handler);
  d_holder->useFC = value;
  d_holder->useFC__setByUser__ = true;
  Trace("options") << "user assigned option useFC" << std::endl;
  
}
template <> void Options::assignBool(
    options::useSOI__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::useSOI, option, value, d_handler);
  d_holder->useSOI = value;
  d_holder->useSOI__setByUser__ = true;
  Trace("options") << "user assigned option useSOI" << std::endl;
  
}
template <> options::arraysConfig__option_t::type runHandlerAndPredicates(
    options::arraysConfig__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::arraysConfig__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::arraysConfig__option_t,
    std::string option,
    std::string value)
{
  d_holder->arraysConfig =
    runHandlerAndPredicates(options::arraysConfig, option, value, d_handler);
  d_holder->arraysConfig__setByUser__ = true;
  Trace("options") << "user assigned option arraysConfig" << std::endl;
  
}
template <> void Options::assignBool(
    options::arraysEagerIndexSplitting__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::arraysEagerIndexSplitting, option, value, d_handler);
  d_holder->arraysEagerIndexSplitting = value;
  d_holder->arraysEagerIndexSplitting__setByUser__ = true;
  Trace("options") << "user assigned option arraysEagerIndexSplitting" << std::endl;
  
}
template <> void Options::assignBool(
    options::arraysEagerLemmas__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::arraysEagerLemmas, option, value, d_handler);
  d_holder->arraysEagerLemmas = value;
  d_holder->arraysEagerLemmas__setByUser__ = true;
  Trace("options") << "user assigned option arraysEagerLemmas" << std::endl;
  
}
template <> void Options::assignBool(
    options::arraysLazyRIntro1__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::arraysLazyRIntro1, option, value, d_handler);
  d_holder->arraysLazyRIntro1 = value;
  d_holder->arraysLazyRIntro1__setByUser__ = true;
  Trace("options") << "user assigned option arraysLazyRIntro1" << std::endl;
  
}
template <> void Options::assignBool(
    options::arraysModelBased__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::arraysModelBased, option, value, d_handler);
  d_holder->arraysModelBased = value;
  d_holder->arraysModelBased__setByUser__ = true;
  Trace("options") << "user assigned option arraysModelBased" << std::endl;
  
}
template <> void Options::assignBool(
    options::arraysOptimizeLinear__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::arraysOptimizeLinear, option, value, d_handler);
  d_holder->arraysOptimizeLinear = value;
  d_holder->arraysOptimizeLinear__setByUser__ = true;
  Trace("options") << "user assigned option arraysOptimizeLinear" << std::endl;
  
}
template <> options::arraysPropagate__option_t::type runHandlerAndPredicates(
    options::arraysPropagate__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::arraysPropagate__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::arraysPropagate__option_t,
    std::string option,
    std::string value)
{
  d_holder->arraysPropagate =
    runHandlerAndPredicates(options::arraysPropagate, option, value, d_handler);
  d_holder->arraysPropagate__setByUser__ = true;
  Trace("options") << "user assigned option arraysPropagate" << std::endl;
  
}
template <> void Options::assignBool(
    options::arraysReduceSharing__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::arraysReduceSharing, option, value, d_handler);
  d_holder->arraysReduceSharing = value;
  d_holder->arraysReduceSharing__setByUser__ = true;
  Trace("options") << "user assigned option arraysReduceSharing" << std::endl;
  
}
template <> void Options::assignBool(
    options::arraysWeakEquivalence__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::arraysWeakEquivalence, option, value, d_handler);
  d_holder->arraysWeakEquivalence = value;
  d_holder->arraysWeakEquivalence__setByUser__ = true;
  Trace("options") << "user assigned option arraysWeakEquivalence" << std::endl;
  
}
template <> options::inputLanguage__option_t::type runHandlerAndPredicates(
    options::inputLanguage__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::inputLanguage__option_t::type retval = handler->stringToInputLanguage(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::inputLanguage__option_t,
    std::string option,
    std::string value)
{
  d_holder->inputLanguage =
    runHandlerAndPredicates(options::inputLanguage, option, value, d_handler);
  d_holder->inputLanguage__setByUser__ = true;
  Trace("options") << "user assigned option inputLanguage" << std::endl;
  
}
template <> void Options::assignBool(
    options::languageHelp__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::languageHelp, option, value, d_handler);
  d_holder->languageHelp = value;
  d_holder->languageHelp__setByUser__ = true;
  Trace("options") << "user assigned option languageHelp" << std::endl;
  
}
template <> options::outputLanguage__option_t::type runHandlerAndPredicates(
    options::outputLanguage__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::outputLanguage__option_t::type retval = handler->stringToOutputLanguage(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::outputLanguage__option_t,
    std::string option,
    std::string value)
{
  d_holder->outputLanguage =
    runHandlerAndPredicates(options::outputLanguage, option, value, d_handler);
  d_holder->outputLanguage__setByUser__ = true;
  Trace("options") << "user assigned option outputLanguage" << std::endl;
  
}
template <> void Options::assignBool(
    options::parseOnly__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::parseOnly, option, value, d_handler);
  d_holder->parseOnly = value;
  d_holder->parseOnly__setByUser__ = true;
  Trace("options") << "user assigned option parseOnly" << std::endl;
  
}
template <> void Options::assignBool(
    options::preprocessOnly__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::preprocessOnly, option, value, d_handler);
  d_holder->preprocessOnly = value;
  d_holder->preprocessOnly__setByUser__ = true;
  Trace("options") << "user assigned option preprocessOnly" << std::endl;
  
}
template <> void Options::assignBool(
    options::printSuccess__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::printSuccess, option, value, d_handler);
  d_holder->printSuccess = value;
  d_holder->printSuccess__setByUser__ = true;
  Trace("options") << "user assigned option printSuccess" << std::endl;
  d_handler->notifyPrintSuccess(option);
}
template <> void runBoolPredicates(
    options::statistics__option_t,
    std::string option,
    bool b,
    options::OptionsHandler* handler)
{
  handler->statsEnabledBuild(option, b);
}
template <> void Options::assignBool(
    options::statistics__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::statistics, option, value, d_handler);
  d_holder->statistics = value;
  d_holder->statistics__setByUser__ = true;
  Trace("options") << "user assigned option statistics" << std::endl;
  
}
template <> void Options::assignBool(
    options::statsEveryQuery__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::statsEveryQuery, option, value, d_handler);
  d_holder->statsEveryQuery = value;
  d_holder->statsEveryQuery__setByUser__ = true;
  Trace("options") << "user assigned option statsEveryQuery" << std::endl;
  
}
template <> void Options::assignBool(
    options::statsHideZeros__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::statsHideZeros, option, value, d_handler);
  d_holder->statsHideZeros = value;
  d_holder->statsHideZeros__setByUser__ = true;
  Trace("options") << "user assigned option statsHideZeros" << std::endl;
  
}
template <> void Options::assign(
    options::verbosity__option_t,
    std::string option,
    std::string value)
{
  d_holder->verbosity =
    runHandlerAndPredicates(options::verbosity, option, value, d_handler);
  d_holder->verbosity__setByUser__ = true;
  Trace("options") << "user assigned option verbosity" << std::endl;
  
}
template <> void runBoolPredicates(
    options::bitvectorAig__option_t,
    std::string option,
    bool b,
    options::OptionsHandler* handler)
{
  handler->abcEnabledBuild(option, b);
handler->setBitblastAig(option, b);
}
template <> void Options::assignBool(
    options::bitvectorAig__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bitvectorAig, option, value, d_handler);
  d_holder->bitvectorAig = value;
  d_holder->bitvectorAig__setByUser__ = true;
  Trace("options") << "user assigned option bitvectorAig" << std::endl;
  
}
template <> options::bitblastMode__option_t::type runHandlerAndPredicates(
    options::bitblastMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::bitblastMode__option_t::type retval = stringToBitblastMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::bitblastMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->bitblastMode =
    runHandlerAndPredicates(options::bitblastMode, option, value, d_handler);
  d_holder->bitblastMode__setByUser__ = true;
  Trace("options") << "user assigned option bitblastMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::bitwiseEq__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bitwiseEq, option, value, d_handler);
  d_holder->bitwiseEq = value;
  d_holder->bitwiseEq__setByUser__ = true;
  Trace("options") << "user assigned option bitwiseEq" << std::endl;
  
}
template <> options::boolToBitvector__option_t::type runHandlerAndPredicates(
    options::boolToBitvector__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::boolToBitvector__option_t::type retval = stringToBoolToBVMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::boolToBitvector__option_t,
    std::string option,
    std::string value)
{
  d_holder->boolToBitvector =
    runHandlerAndPredicates(options::boolToBitvector, option, value, d_handler);
  d_holder->boolToBitvector__setByUser__ = true;
  Trace("options") << "user assigned option boolToBitvector" << std::endl;
  
}
template <> void Options::assignBool(
    options::bvAbstraction__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bvAbstraction, option, value, d_handler);
  d_holder->bvAbstraction = value;
  d_holder->bvAbstraction__setByUser__ = true;
  Trace("options") << "user assigned option bvAbstraction" << std::endl;
  
}
template <> options::bitvectorAigSimplifications__option_t::type runHandlerAndPredicates(
    options::bitvectorAigSimplifications__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::bitvectorAigSimplifications__option_t::type retval = handleOption<std::string>(option, optionarg);
  handler->abcEnabledBuild(option, retval);
  return retval;
}
template <> void Options::assign(
    options::bitvectorAigSimplifications__option_t,
    std::string option,
    std::string value)
{
  d_holder->bitvectorAigSimplifications =
    runHandlerAndPredicates(options::bitvectorAigSimplifications, option, value, d_handler);
  d_holder->bitvectorAigSimplifications__setByUser__ = true;
  Trace("options") << "user assigned option bitvectorAigSimplifications" << std::endl;
  
}
template <> void Options::assignBool(
    options::bvAlgExtf__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bvAlgExtf, option, value, d_handler);
  d_holder->bvAlgExtf = value;
  d_holder->bvAlgExtf__setByUser__ = true;
  Trace("options") << "user assigned option bvAlgExtf" << std::endl;
  
}
template <> options::bitvectorAlgebraicBudget__option_t::type runHandlerAndPredicates(
    options::bitvectorAlgebraicBudget__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::bitvectorAlgebraicBudget__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::bitvectorAlgebraicBudget__option_t,
    std::string option,
    std::string value)
{
  d_holder->bitvectorAlgebraicBudget =
    runHandlerAndPredicates(options::bitvectorAlgebraicBudget, option, value, d_handler);
  d_holder->bitvectorAlgebraicBudget__setByUser__ = true;
  Trace("options") << "user assigned option bitvectorAlgebraicBudget" << std::endl;
  
}
template <> void Options::assignBool(
    options::bitvectorAlgebraicSolver__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bitvectorAlgebraicSolver, option, value, d_handler);
  d_holder->bitvectorAlgebraicSolver = value;
  d_holder->bitvectorAlgebraicSolver__setByUser__ = true;
  Trace("options") << "user assigned option bitvectorAlgebraicSolver" << std::endl;
  
}
template <> void Options::assignBool(
    options::bitvectorDivByZeroConst__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bitvectorDivByZeroConst, option, value, d_handler);
  d_holder->bitvectorDivByZeroConst = value;
  d_holder->bitvectorDivByZeroConst__setByUser__ = true;
  Trace("options") << "user assigned option bitvectorDivByZeroConst" << std::endl;
  
}
template <> void Options::assignBool(
    options::bvEagerExplanations__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bvEagerExplanations, option, value, d_handler);
  d_holder->bvEagerExplanations = value;
  d_holder->bvEagerExplanations__setByUser__ = true;
  Trace("options") << "user assigned option bvEagerExplanations" << std::endl;
  
}
template <> options::bitvectorEqualitySlicer__option_t::type runHandlerAndPredicates(
    options::bitvectorEqualitySlicer__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::bitvectorEqualitySlicer__option_t::type retval = stringToBvSlicerMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::bitvectorEqualitySlicer__option_t,
    std::string option,
    std::string value)
{
  d_holder->bitvectorEqualitySlicer =
    runHandlerAndPredicates(options::bitvectorEqualitySlicer, option, value, d_handler);
  d_holder->bitvectorEqualitySlicer__setByUser__ = true;
  Trace("options") << "user assigned option bitvectorEqualitySlicer" << std::endl;
  
}
template <> void Options::assignBool(
    options::bitvectorEqualitySolver__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bitvectorEqualitySolver, option, value, d_handler);
  d_holder->bitvectorEqualitySolver = value;
  d_holder->bitvectorEqualitySolver__setByUser__ = true;
  Trace("options") << "user assigned option bitvectorEqualitySolver" << std::endl;
  
}
template <> void Options::assignBool(
    options::bvExtractArithRewrite__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bvExtractArithRewrite, option, value, d_handler);
  d_holder->bvExtractArithRewrite = value;
  d_holder->bvExtractArithRewrite__setByUser__ = true;
  Trace("options") << "user assigned option bvExtractArithRewrite" << std::endl;
  
}
template <> void Options::assignBool(
    options::bvGaussElim__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bvGaussElim, option, value, d_handler);
  d_holder->bvGaussElim = value;
  d_holder->bvGaussElim__setByUser__ = true;
  Trace("options") << "user assigned option bvGaussElim" << std::endl;
  
}
template <> void Options::assignBool(
    options::bitvectorInequalitySolver__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bitvectorInequalitySolver, option, value, d_handler);
  d_holder->bitvectorInequalitySolver = value;
  d_holder->bitvectorInequalitySolver__setByUser__ = true;
  Trace("options") << "user assigned option bitvectorInequalitySolver" << std::endl;
  
}
template <> void Options::assignBool(
    options::bvIntroducePow2__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bvIntroducePow2, option, value, d_handler);
  d_holder->bvIntroducePow2 = value;
  d_holder->bvIntroducePow2__setByUser__ = true;
  Trace("options") << "user assigned option bvIntroducePow2" << std::endl;
  
}
template <> void Options::assignBool(
    options::bvLazyReduceExtf__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bvLazyReduceExtf, option, value, d_handler);
  d_holder->bvLazyReduceExtf = value;
  d_holder->bvLazyReduceExtf__setByUser__ = true;
  Trace("options") << "user assigned option bvLazyReduceExtf" << std::endl;
  
}
template <> void Options::assignBool(
    options::bvLazyRewriteExtf__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bvLazyRewriteExtf, option, value, d_handler);
  d_holder->bvLazyRewriteExtf = value;
  d_holder->bvLazyRewriteExtf__setByUser__ = true;
  Trace("options") << "user assigned option bvLazyRewriteExtf" << std::endl;
  
}
template <> options::bvNumFunc__option_t::type runHandlerAndPredicates(
    options::bvNumFunc__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::bvNumFunc__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::bvNumFunc__option_t,
    std::string option,
    std::string value)
{
  d_holder->bvNumFunc =
    runHandlerAndPredicates(options::bvNumFunc, option, value, d_handler);
  d_holder->bvNumFunc__setByUser__ = true;
  Trace("options") << "user assigned option bvNumFunc" << std::endl;
  
}
template <> options::bvOptimizeSatProof__option_t::type runHandlerAndPredicates(
    options::bvOptimizeSatProof__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::bvOptimizeSatProof__option_t::type retval = stringToBvOptimizeSatProof(option, optionarg);
  handler->checkSatSolverEnabled<BvOptimizeSatProof>(option, retval);
  return retval;
}
template <> void Options::assign(
    options::bvOptimizeSatProof__option_t,
    std::string option,
    std::string value)
{
  d_holder->bvOptimizeSatProof =
    runHandlerAndPredicates(options::bvOptimizeSatProof, option, value, d_handler);
  d_holder->bvOptimizeSatProof__setByUser__ = true;
  Trace("options") << "user assigned option bvOptimizeSatProof" << std::endl;
  
}
template <> void Options::assignBool(
    options::bvPrintConstsAsIndexedSymbols__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bvPrintConstsAsIndexedSymbols, option, value, d_handler);
  d_holder->bvPrintConstsAsIndexedSymbols = value;
  d_holder->bvPrintConstsAsIndexedSymbols__setByUser__ = true;
  Trace("options") << "user assigned option bvPrintConstsAsIndexedSymbols" << std::endl;
  
}
template <> options::bvProofFormat__option_t::type runHandlerAndPredicates(
    options::bvProofFormat__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::bvProofFormat__option_t::type retval = stringToBvProofFormat(option, optionarg);
  handler->checkSatSolverEnabled<BvProofFormat>(option, retval);
  return retval;
}
template <> void Options::assign(
    options::bvProofFormat__option_t,
    std::string option,
    std::string value)
{
  d_holder->bvProofFormat =
    runHandlerAndPredicates(options::bvProofFormat, option, value, d_handler);
  d_holder->bvProofFormat__setByUser__ = true;
  Trace("options") << "user assigned option bvProofFormat" << std::endl;
  
}
template <> void Options::assignBool(
    options::bitvectorPropagate__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bitvectorPropagate, option, value, d_handler);
  d_holder->bitvectorPropagate = value;
  d_holder->bitvectorPropagate__setByUser__ = true;
  Trace("options") << "user assigned option bitvectorPropagate" << std::endl;
  
}
template <> void Options::assignBool(
    options::bitvectorQuickXplain__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bitvectorQuickXplain, option, value, d_handler);
  d_holder->bitvectorQuickXplain = value;
  d_holder->bitvectorQuickXplain__setByUser__ = true;
  Trace("options") << "user assigned option bitvectorQuickXplain" << std::endl;
  
}
template <> options::bvSatSolver__option_t::type runHandlerAndPredicates(
    options::bvSatSolver__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::bvSatSolver__option_t::type retval = stringToSatSolverMode(option, optionarg);
  handler->checkBvSatSolver(option, retval);
  return retval;
}
template <> void Options::assign(
    options::bvSatSolver__option_t,
    std::string option,
    std::string value)
{
  d_holder->bvSatSolver =
    runHandlerAndPredicates(options::bvSatSolver, option, value, d_handler);
  d_holder->bvSatSolver__setByUser__ = true;
  Trace("options") << "user assigned option bvSatSolver" << std::endl;
  
}
template <> void Options::assignBool(
    options::skolemizeArguments__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::skolemizeArguments, option, value, d_handler);
  d_holder->skolemizeArguments = value;
  d_holder->skolemizeArguments__setByUser__ = true;
  Trace("options") << "user assigned option skolemizeArguments" << std::endl;
  
}
template <> void Options::assignBool(
    options::bitvectorToBool__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::bitvectorToBool, option, value, d_handler);
  d_holder->bitvectorToBool = value;
  d_holder->bitvectorToBool__setByUser__ = true;
  Trace("options") << "user assigned option bitvectorToBool" << std::endl;
  
}
template <> void Options::assignBool(
    options::cdtBisimilar__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cdtBisimilar, option, value, d_handler);
  d_holder->cdtBisimilar = value;
  d_holder->cdtBisimilar__setByUser__ = true;
  Trace("options") << "user assigned option cdtBisimilar" << std::endl;
  
}
template <> void Options::assignBool(
    options::dtBinarySplit__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dtBinarySplit, option, value, d_handler);
  d_holder->dtBinarySplit = value;
  d_holder->dtBinarySplit__setByUser__ = true;
  Trace("options") << "user assigned option dtBinarySplit" << std::endl;
  
}
template <> void Options::assignBool(
    options::dtBlastSplits__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dtBlastSplits, option, value, d_handler);
  d_holder->dtBlastSplits = value;
  d_holder->dtBlastSplits__setByUser__ = true;
  Trace("options") << "user assigned option dtBlastSplits" << std::endl;
  
}
template <> void Options::assignBool(
    options::dtCyclic__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dtCyclic, option, value, d_handler);
  d_holder->dtCyclic = value;
  d_holder->dtCyclic__setByUser__ = true;
  Trace("options") << "user assigned option dtCyclic" << std::endl;
  
}
template <> void Options::assignBool(
    options::dtForceAssignment__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dtForceAssignment, option, value, d_handler);
  d_holder->dtForceAssignment = value;
  d_holder->dtForceAssignment__setByUser__ = true;
  Trace("options") << "user assigned option dtForceAssignment" << std::endl;
  
}
template <> void Options::assignBool(
    options::dtInferAsLemmas__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dtInferAsLemmas, option, value, d_handler);
  d_holder->dtInferAsLemmas = value;
  d_holder->dtInferAsLemmas__setByUser__ = true;
  Trace("options") << "user assigned option dtInferAsLemmas" << std::endl;
  
}
template <> void Options::assignBool(
    options::dtNestedRec__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dtNestedRec, option, value, d_handler);
  d_holder->dtNestedRec = value;
  d_holder->dtNestedRec__setByUser__ = true;
  Trace("options") << "user assigned option dtNestedRec" << std::endl;
  
}
template <> void Options::assignBool(
    options::dtRewriteErrorSel__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dtRewriteErrorSel, option, value, d_handler);
  d_holder->dtRewriteErrorSel = value;
  d_holder->dtRewriteErrorSel__setByUser__ = true;
  Trace("options") << "user assigned option dtRewriteErrorSel" << std::endl;
  
}
template <> void Options::assignBool(
    options::dtSharedSelectors__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dtSharedSelectors, option, value, d_handler);
  d_holder->dtSharedSelectors = value;
  d_holder->dtSharedSelectors__setByUser__ = true;
  Trace("options") << "user assigned option dtSharedSelectors" << std::endl;
  
}
template <> options::sygusAbortSize__option_t::type runHandlerAndPredicates(
    options::sygusAbortSize__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusAbortSize__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusAbortSize__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusAbortSize =
    runHandlerAndPredicates(options::sygusAbortSize, option, value, d_handler);
  d_holder->sygusAbortSize__setByUser__ = true;
  Trace("options") << "user assigned option sygusAbortSize" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusFairMax__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusFairMax, option, value, d_handler);
  d_holder->sygusFairMax = value;
  d_holder->sygusFairMax__setByUser__ = true;
  Trace("options") << "user assigned option sygusFairMax" << std::endl;
  
}
template <> options::sygusFair__option_t::type runHandlerAndPredicates(
    options::sygusFair__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusFair__option_t::type retval = stringToSygusFairMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusFair__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusFair =
    runHandlerAndPredicates(options::sygusFair, option, value, d_handler);
  d_holder->sygusFair__setByUser__ = true;
  Trace("options") << "user assigned option sygusFair" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusSymBreak__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusSymBreak, option, value, d_handler);
  d_holder->sygusSymBreak = value;
  d_holder->sygusSymBreak__setByUser__ = true;
  Trace("options") << "user assigned option sygusSymBreak" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusSymBreakAgg__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusSymBreakAgg, option, value, d_handler);
  d_holder->sygusSymBreakAgg = value;
  d_holder->sygusSymBreakAgg__setByUser__ = true;
  Trace("options") << "user assigned option sygusSymBreakAgg" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusSymBreakDynamic__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusSymBreakDynamic, option, value, d_handler);
  d_holder->sygusSymBreakDynamic = value;
  d_holder->sygusSymBreakDynamic__setByUser__ = true;
  Trace("options") << "user assigned option sygusSymBreakDynamic" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusSymBreakLazy__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusSymBreakLazy, option, value, d_handler);
  d_holder->sygusSymBreakLazy = value;
  d_holder->sygusSymBreakLazy__setByUser__ = true;
  Trace("options") << "user assigned option sygusSymBreakLazy" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusSymBreakPbe__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusSymBreakPbe, option, value, d_handler);
  d_holder->sygusSymBreakPbe = value;
  d_holder->sygusSymBreakPbe__setByUser__ = true;
  Trace("options") << "user assigned option sygusSymBreakPbe" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusSymBreakRlv__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusSymBreakRlv, option, value, d_handler);
  d_holder->sygusSymBreakRlv = value;
  d_holder->sygusSymBreakRlv__setByUser__ = true;
  Trace("options") << "user assigned option sygusSymBreakRlv" << std::endl;
  
}
template <> options::decisionRandomWeight__option_t::type runHandlerAndPredicates(
    options::decisionRandomWeight__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::decisionRandomWeight__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::decisionRandomWeight__option_t,
    std::string option,
    std::string value)
{
  d_holder->decisionRandomWeight =
    runHandlerAndPredicates(options::decisionRandomWeight, option, value, d_handler);
  d_holder->decisionRandomWeight__setByUser__ = true;
  Trace("options") << "user assigned option decisionRandomWeight" << std::endl;
  
}
template <> options::decisionThreshold__option_t::type runHandlerAndPredicates(
    options::decisionThreshold__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::decisionThreshold__option_t::type retval = handleOption<decision::DecisionWeight>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::decisionThreshold__option_t,
    std::string option,
    std::string value)
{
  d_holder->decisionThreshold =
    runHandlerAndPredicates(options::decisionThreshold, option, value, d_handler);
  d_holder->decisionThreshold__setByUser__ = true;
  Trace("options") << "user assigned option decisionThreshold" << std::endl;
  
}
template <> void Options::assignBool(
    options::decisionUseWeight__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::decisionUseWeight, option, value, d_handler);
  d_holder->decisionUseWeight = value;
  d_holder->decisionUseWeight__setByUser__ = true;
  Trace("options") << "user assigned option decisionUseWeight" << std::endl;
  
}
template <> options::decisionWeightInternal__option_t::type runHandlerAndPredicates(
    options::decisionWeightInternal__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::decisionWeightInternal__option_t::type retval = stringToDecisionWeightInternal(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::decisionWeightInternal__option_t,
    std::string option,
    std::string value)
{
  d_holder->decisionWeightInternal =
    runHandlerAndPredicates(options::decisionWeightInternal, option, value, d_handler);
  d_holder->decisionWeightInternal__setByUser__ = true;
  Trace("options") << "user assigned option decisionWeightInternal" << std::endl;
  
}
template <> options::decisionMode__option_t::type runHandlerAndPredicates(
    options::decisionMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::decisionMode__option_t::type retval = stringToDecisionMode(option, optionarg);
  handler->setDecisionModeStopOnly(option, retval);
  return retval;
}
template <> void Options::assign(
    options::decisionMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->decisionMode =
    runHandlerAndPredicates(options::decisionMode, option, value, d_handler);
  d_holder->decisionMode__setByUser__ = true;
  Trace("options") << "user assigned option decisionMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::decisionStopOnly__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::decisionStopOnly, option, value, d_handler);
  d_holder->decisionStopOnly = value;
  d_holder->decisionStopOnly__setByUser__ = true;
  Trace("options") << "user assigned option decisionStopOnly" << std::endl;
  
}
template <> options::defaultDagThresh__option_t::type runHandlerAndPredicates(
    options::defaultDagThresh__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::defaultDagThresh__option_t::type retval = handleOption<int>(option, optionarg);
  handler->setDefaultDagThreshPredicate(option, retval);
  return retval;
}
template <> void Options::assign(
    options::defaultDagThresh__option_t,
    std::string option,
    std::string value)
{
  d_holder->defaultDagThresh =
    runHandlerAndPredicates(options::defaultDagThresh, option, value, d_handler);
  d_holder->defaultDagThresh__setByUser__ = true;
  Trace("options") << "user assigned option defaultDagThresh" << std::endl;
  d_handler->notifySetDefaultDagThresh(option);
}
template <> options::defaultExprDepth__option_t::type runHandlerAndPredicates(
    options::defaultExprDepth__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::defaultExprDepth__option_t::type retval = handleOption<int>(option, optionarg);
  handler->setDefaultExprDepthPredicate(option, retval);
  return retval;
}
template <> void Options::assign(
    options::defaultExprDepth__option_t,
    std::string option,
    std::string value)
{
  d_holder->defaultExprDepth =
    runHandlerAndPredicates(options::defaultExprDepth, option, value, d_handler);
  d_holder->defaultExprDepth__setByUser__ = true;
  Trace("options") << "user assigned option defaultExprDepth" << std::endl;
  d_handler->notifySetDefaultExprDepth(option);
}
template <> void Options::assignBool(
    options::printExprTypes__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::printExprTypes, option, value, d_handler);
  d_holder->printExprTypes = value;
  d_holder->printExprTypes__setByUser__ = true;
  Trace("options") << "user assigned option printExprTypes" << std::endl;
  d_handler->notifySetPrintExprTypes(option);
}
template <> void Options::assignBool(
    options::typeChecking__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::typeChecking, option, value, d_handler);
  d_holder->typeChecking = value;
  d_holder->typeChecking__setByUser__ = true;
  Trace("options") << "user assigned option typeChecking" << std::endl;
  
}
template <> void Options::assignBool(
    options::fpExp__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fpExp, option, value, d_handler);
  d_holder->fpExp = value;
  d_holder->fpExp__setByUser__ = true;
  Trace("options") << "user assigned option fpExp" << std::endl;
  
}
template <> void Options::assignBool(
    options::earlyExit__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::earlyExit, option, value, d_handler);
  d_holder->earlyExit = value;
  d_holder->earlyExit__setByUser__ = true;
  Trace("options") << "user assigned option earlyExit" << std::endl;
  
}
template <> void Options::assignBool(
    options::help__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::help, option, value, d_handler);
  d_holder->help = value;
  d_holder->help__setByUser__ = true;
  Trace("options") << "user assigned option help" << std::endl;
  
}
template <> void Options::assignBool(
    options::interactive__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::interactive, option, value, d_handler);
  d_holder->interactive = value;
  d_holder->interactive__setByUser__ = true;
  Trace("options") << "user assigned option interactive" << std::endl;
  
}
template <> void Options::assignBool(
    options::interactivePrompt__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::interactivePrompt, option, value, d_handler);
  d_holder->interactivePrompt = value;
  d_holder->interactivePrompt__setByUser__ = true;
  Trace("options") << "user assigned option interactivePrompt" << std::endl;
  
}
template <> options::seed__option_t::type runHandlerAndPredicates(
    options::seed__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::seed__option_t::type retval = handleOption<uint64_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::seed__option_t,
    std::string option,
    std::string value)
{
  d_holder->seed =
    runHandlerAndPredicates(options::seed, option, value, d_handler);
  d_holder->seed__setByUser__ = true;
  Trace("options") << "user assigned option seed" << std::endl;
  
}
template <> void Options::assignBool(
    options::segvSpin__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::segvSpin, option, value, d_handler);
  d_holder->segvSpin = value;
  d_holder->segvSpin__setByUser__ = true;
  Trace("options") << "user assigned option segvSpin" << std::endl;
  
}
template <> options::tearDownIncremental__option_t::type runHandlerAndPredicates(
    options::tearDownIncremental__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::tearDownIncremental__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::tearDownIncremental__option_t,
    std::string option,
    std::string value)
{
  d_holder->tearDownIncremental =
    runHandlerAndPredicates(options::tearDownIncremental, option, value, d_handler);
  d_holder->tearDownIncremental__setByUser__ = true;
  Trace("options") << "user assigned option tearDownIncremental" << std::endl;
  
}
template <> void Options::assignBool(
    options::version__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::version, option, value, d_handler);
  d_holder->version = value;
  d_holder->version__setByUser__ = true;
  Trace("options") << "user assigned option version" << std::endl;
  
}
template <> void Options::assignBool(
    options::filesystemAccess__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::filesystemAccess, option, value, d_handler);
  d_holder->filesystemAccess = value;
  d_holder->filesystemAccess__setByUser__ = true;
  Trace("options") << "user assigned option filesystemAccess" << std::endl;
  
}
template <> options::forceLogicString__option_t::type runHandlerAndPredicates(
    options::forceLogicString__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::forceLogicString__option_t::type retval = handleOption<std::string>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::forceLogicString__option_t,
    std::string option,
    std::string value)
{
  d_holder->forceLogicString =
    runHandlerAndPredicates(options::forceLogicString, option, value, d_handler);
  d_holder->forceLogicString__setByUser__ = true;
  Trace("options") << "user assigned option forceLogicString" << std::endl;
  
}
template <> void Options::assignBool(
    options::globalDeclarations__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::globalDeclarations, option, value, d_handler);
  d_holder->globalDeclarations = value;
  d_holder->globalDeclarations__setByUser__ = true;
  Trace("options") << "user assigned option globalDeclarations" << std::endl;
  
}
template <> void Options::assignBool(
    options::memoryMap__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::memoryMap, option, value, d_handler);
  d_holder->memoryMap = value;
  d_holder->memoryMap__setByUser__ = true;
  Trace("options") << "user assigned option memoryMap" << std::endl;
  
}
template <> void Options::assignBool(
    options::semanticChecks__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::semanticChecks, option, value, d_handler);
  d_holder->semanticChecks = value;
  d_holder->semanticChecks__setByUser__ = true;
  Trace("options") << "user assigned option semanticChecks" << std::endl;
  
}
template <> void Options::assignBool(
    options::strictParsing__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::strictParsing, option, value, d_handler);
  d_holder->strictParsing = value;
  d_holder->strictParsing__setByUser__ = true;
  Trace("options") << "user assigned option strictParsing" << std::endl;
  
}
template <> void Options::assignBool(
    options::flattenHOChains__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::flattenHOChains, option, value, d_handler);
  d_holder->flattenHOChains = value;
  d_holder->flattenHOChains__setByUser__ = true;
  Trace("options") << "user assigned option flattenHOChains" << std::endl;
  
}
template <> options::instFormatMode__option_t::type runHandlerAndPredicates(
    options::instFormatMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::instFormatMode__option_t::type retval = handler->stringToInstFormatMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::instFormatMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->instFormatMode =
    runHandlerAndPredicates(options::instFormatMode, option, value, d_handler);
  d_holder->instFormatMode__setByUser__ = true;
  Trace("options") << "user assigned option instFormatMode" << std::endl;
  
}
template <> options::modelFormatMode__option_t::type runHandlerAndPredicates(
    options::modelFormatMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::modelFormatMode__option_t::type retval = stringToModelFormatMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::modelFormatMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->modelFormatMode =
    runHandlerAndPredicates(options::modelFormatMode, option, value, d_handler);
  d_holder->modelFormatMode__setByUser__ = true;
  Trace("options") << "user assigned option modelFormatMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::aggressiveCoreMin__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::aggressiveCoreMin, option, value, d_handler);
  d_holder->aggressiveCoreMin = value;
  d_holder->aggressiveCoreMin__setByUser__ = true;
  Trace("options") << "user assigned option aggressiveCoreMin" << std::endl;
  
}
template <> void Options::assignBool(
    options::allowEmptyDependencies__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::allowEmptyDependencies, option, value, d_handler);
  d_holder->allowEmptyDependencies = value;
  d_holder->allowEmptyDependencies__setByUser__ = true;
  Trace("options") << "user assigned option allowEmptyDependencies" << std::endl;
  
}
template <> void Options::assignBool(
    options::fewerPreprocessingHoles__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fewerPreprocessingHoles, option, value, d_handler);
  d_holder->fewerPreprocessingHoles = value;
  d_holder->fewerPreprocessingHoles__setByUser__ = true;
  Trace("options") << "user assigned option fewerPreprocessingHoles" << std::endl;
  
}
template <> void Options::assignBool(
    options::lfscLetification__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::lfscLetification, option, value, d_handler);
  d_holder->lfscLetification = value;
  d_holder->lfscLetification__setByUser__ = true;
  Trace("options") << "user assigned option lfscLetification" << std::endl;
  
}
template <> void Options::assignBool(
    options::minisatDumpDimacs__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::minisatDumpDimacs, option, value, d_handler);
  d_holder->minisatDumpDimacs = value;
  d_holder->minisatDumpDimacs__setByUser__ = true;
  Trace("options") << "user assigned option minisatDumpDimacs" << std::endl;
  
}
template <> void Options::assignBool(
    options::minisatUseElim__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::minisatUseElim, option, value, d_handler);
  d_holder->minisatUseElim = value;
  d_holder->minisatUseElim__setByUser__ = true;
  Trace("options") << "user assigned option minisatUseElim" << std::endl;
  
}
template <> options::satRandomFreq__option_t::type runHandlerAndPredicates(
    options::satRandomFreq__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::satRandomFreq__option_t::type retval = handleOption<double>(option, optionarg);
  handler->doubleGreaterOrEqual0(option, retval);
handler->doubleLessOrEqual1(option, retval);
  return retval;
}
template <> void Options::assign(
    options::satRandomFreq__option_t,
    std::string option,
    std::string value)
{
  d_holder->satRandomFreq =
    runHandlerAndPredicates(options::satRandomFreq, option, value, d_handler);
  d_holder->satRandomFreq__setByUser__ = true;
  Trace("options") << "user assigned option satRandomFreq" << std::endl;
  
}
template <> options::satRandomSeed__option_t::type runHandlerAndPredicates(
    options::satRandomSeed__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::satRandomSeed__option_t::type retval = handleOption<uint32_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::satRandomSeed__option_t,
    std::string option,
    std::string value)
{
  d_holder->satRandomSeed =
    runHandlerAndPredicates(options::satRandomSeed, option, value, d_handler);
  d_holder->satRandomSeed__setByUser__ = true;
  Trace("options") << "user assigned option satRandomSeed" << std::endl;
  
}
template <> void Options::assignBool(
    options::sat_refine_conflicts__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sat_refine_conflicts, option, value, d_handler);
  d_holder->sat_refine_conflicts = value;
  d_holder->sat_refine_conflicts__setByUser__ = true;
  Trace("options") << "user assigned option sat_refine_conflicts" << std::endl;
  
}
template <> options::satRestartFirst__option_t::type runHandlerAndPredicates(
    options::satRestartFirst__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::satRestartFirst__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::satRestartFirst__option_t,
    std::string option,
    std::string value)
{
  d_holder->satRestartFirst =
    runHandlerAndPredicates(options::satRestartFirst, option, value, d_handler);
  d_holder->satRestartFirst__setByUser__ = true;
  Trace("options") << "user assigned option satRestartFirst" << std::endl;
  
}
template <> options::satRestartInc__option_t::type runHandlerAndPredicates(
    options::satRestartInc__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::satRestartInc__option_t::type retval = handleOption<double>(option, optionarg);
  handler->doubleGreaterOrEqual0(option, retval);
  return retval;
}
template <> void Options::assign(
    options::satRestartInc__option_t,
    std::string option,
    std::string value)
{
  d_holder->satRestartInc =
    runHandlerAndPredicates(options::satRestartInc, option, value, d_handler);
  d_holder->satRestartInc__setByUser__ = true;
  Trace("options") << "user assigned option satRestartInc" << std::endl;
  
}
template <> void Options::assignBool(
    options::aggressiveMiniscopeQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::aggressiveMiniscopeQuant, option, value, d_handler);
  d_holder->aggressiveMiniscopeQuant = value;
  d_holder->aggressiveMiniscopeQuant__setByUser__ = true;
  Trace("options") << "user assigned option aggressiveMiniscopeQuant" << std::endl;
  
}
template <> options::cegisSample__option_t::type runHandlerAndPredicates(
    options::cegisSample__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::cegisSample__option_t::type retval = stringToCegisSampleMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::cegisSample__option_t,
    std::string option,
    std::string value)
{
  d_holder->cegisSample =
    runHandlerAndPredicates(options::cegisSample, option, value, d_handler);
  d_holder->cegisSample__setByUser__ = true;
  Trace("options") << "user assigned option cegisSample" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqi__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqi, option, value, d_handler);
  d_holder->cegqi = value;
  d_holder->cegqi__setByUser__ = true;
  Trace("options") << "user assigned option cegqi" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiAll__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiAll, option, value, d_handler);
  d_holder->cegqiAll = value;
  d_holder->cegqiAll__setByUser__ = true;
  Trace("options") << "user assigned option cegqiAll" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiBv__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiBv, option, value, d_handler);
  d_holder->cegqiBv = value;
  d_holder->cegqiBv__setByUser__ = true;
  Trace("options") << "user assigned option cegqiBv" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiBvConcInv__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiBvConcInv, option, value, d_handler);
  d_holder->cegqiBvConcInv = value;
  d_holder->cegqiBvConcInv__setByUser__ = true;
  Trace("options") << "user assigned option cegqiBvConcInv" << std::endl;
  
}
template <> options::cegqiBvIneqMode__option_t::type runHandlerAndPredicates(
    options::cegqiBvIneqMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::cegqiBvIneqMode__option_t::type retval = stringToCegqiBvIneqMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::cegqiBvIneqMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->cegqiBvIneqMode =
    runHandlerAndPredicates(options::cegqiBvIneqMode, option, value, d_handler);
  d_holder->cegqiBvIneqMode__setByUser__ = true;
  Trace("options") << "user assigned option cegqiBvIneqMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiBvInterleaveValue__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiBvInterleaveValue, option, value, d_handler);
  d_holder->cegqiBvInterleaveValue = value;
  d_holder->cegqiBvInterleaveValue__setByUser__ = true;
  Trace("options") << "user assigned option cegqiBvInterleaveValue" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiBvLinearize__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiBvLinearize, option, value, d_handler);
  d_holder->cegqiBvLinearize = value;
  d_holder->cegqiBvLinearize__setByUser__ = true;
  Trace("options") << "user assigned option cegqiBvLinearize" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiBvRmExtract__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiBvRmExtract, option, value, d_handler);
  d_holder->cegqiBvRmExtract = value;
  d_holder->cegqiBvRmExtract__setByUser__ = true;
  Trace("options") << "user assigned option cegqiBvRmExtract" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiBvSolveNl__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiBvSolveNl, option, value, d_handler);
  d_holder->cegqiBvSolveNl = value;
  d_holder->cegqiBvSolveNl__setByUser__ = true;
  Trace("options") << "user assigned option cegqiBvSolveNl" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiFullEffort__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiFullEffort, option, value, d_handler);
  d_holder->cegqiFullEffort = value;
  d_holder->cegqiFullEffort__setByUser__ = true;
  Trace("options") << "user assigned option cegqiFullEffort" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiInnermost__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiInnermost, option, value, d_handler);
  d_holder->cegqiInnermost = value;
  d_holder->cegqiInnermost__setByUser__ = true;
  Trace("options") << "user assigned option cegqiInnermost" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiMidpoint__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiMidpoint, option, value, d_handler);
  d_holder->cegqiMidpoint = value;
  d_holder->cegqiMidpoint__setByUser__ = true;
  Trace("options") << "user assigned option cegqiMidpoint" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiMinBounds__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiMinBounds, option, value, d_handler);
  d_holder->cegqiMinBounds = value;
  d_holder->cegqiMinBounds__setByUser__ = true;
  Trace("options") << "user assigned option cegqiMinBounds" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiModel__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiModel, option, value, d_handler);
  d_holder->cegqiModel = value;
  d_holder->cegqiModel__setByUser__ = true;
  Trace("options") << "user assigned option cegqiModel" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiMultiInst__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiMultiInst, option, value, d_handler);
  d_holder->cegqiMultiInst = value;
  d_holder->cegqiMultiInst__setByUser__ = true;
  Trace("options") << "user assigned option cegqiMultiInst" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiNestedQE__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiNestedQE, option, value, d_handler);
  d_holder->cegqiNestedQE = value;
  d_holder->cegqiNestedQE__setByUser__ = true;
  Trace("options") << "user assigned option cegqiNestedQE" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiNopt__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiNopt, option, value, d_handler);
  d_holder->cegqiNopt = value;
  d_holder->cegqiNopt__setByUser__ = true;
  Trace("options") << "user assigned option cegqiNopt" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiPreRegInst__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiPreRegInst, option, value, d_handler);
  d_holder->cegqiPreRegInst = value;
  d_holder->cegqiPreRegInst__setByUser__ = true;
  Trace("options") << "user assigned option cegqiPreRegInst" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiRepeatLit__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiRepeatLit, option, value, d_handler);
  d_holder->cegqiRepeatLit = value;
  d_holder->cegqiRepeatLit__setByUser__ = true;
  Trace("options") << "user assigned option cegqiRepeatLit" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiRoundUpLowerLia__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiRoundUpLowerLia, option, value, d_handler);
  d_holder->cegqiRoundUpLowerLia = value;
  d_holder->cegqiRoundUpLowerLia__setByUser__ = true;
  Trace("options") << "user assigned option cegqiRoundUpLowerLia" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiSat__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiSat, option, value, d_handler);
  d_holder->cegqiSat = value;
  d_holder->cegqiSat__setByUser__ = true;
  Trace("options") << "user assigned option cegqiSat" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiUseInfInt__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiUseInfInt, option, value, d_handler);
  d_holder->cegqiUseInfInt = value;
  d_holder->cegqiUseInfInt__setByUser__ = true;
  Trace("options") << "user assigned option cegqiUseInfInt" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiUseInfReal__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiUseInfReal, option, value, d_handler);
  d_holder->cegqiUseInfReal = value;
  d_holder->cegqiUseInfReal__setByUser__ = true;
  Trace("options") << "user assigned option cegqiUseInfReal" << std::endl;
  
}
template <> void Options::assignBool(
    options::condVarSplitQuantAgg__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::condVarSplitQuantAgg, option, value, d_handler);
  d_holder->condVarSplitQuantAgg = value;
  d_holder->condVarSplitQuantAgg__setByUser__ = true;
  Trace("options") << "user assigned option condVarSplitQuantAgg" << std::endl;
  
}
template <> void Options::assignBool(
    options::condVarSplitQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::condVarSplitQuant, option, value, d_handler);
  d_holder->condVarSplitQuant = value;
  d_holder->condVarSplitQuant__setByUser__ = true;
  Trace("options") << "user assigned option condVarSplitQuant" << std::endl;
  
}
template <> void Options::assignBool(
    options::conjectureFilterActiveTerms__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::conjectureFilterActiveTerms, option, value, d_handler);
  d_holder->conjectureFilterActiveTerms = value;
  d_holder->conjectureFilterActiveTerms__setByUser__ = true;
  Trace("options") << "user assigned option conjectureFilterActiveTerms" << std::endl;
  
}
template <> void Options::assignBool(
    options::conjectureFilterCanonical__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::conjectureFilterCanonical, option, value, d_handler);
  d_holder->conjectureFilterCanonical = value;
  d_holder->conjectureFilterCanonical__setByUser__ = true;
  Trace("options") << "user assigned option conjectureFilterCanonical" << std::endl;
  
}
template <> void Options::assignBool(
    options::conjectureFilterModel__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::conjectureFilterModel, option, value, d_handler);
  d_holder->conjectureFilterModel = value;
  d_holder->conjectureFilterModel__setByUser__ = true;
  Trace("options") << "user assigned option conjectureFilterModel" << std::endl;
  
}
template <> void Options::assignBool(
    options::conjectureGen__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::conjectureGen, option, value, d_handler);
  d_holder->conjectureGen = value;
  d_holder->conjectureGen__setByUser__ = true;
  Trace("options") << "user assigned option conjectureGen" << std::endl;
  
}
template <> options::conjectureGenGtEnum__option_t::type runHandlerAndPredicates(
    options::conjectureGenGtEnum__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::conjectureGenGtEnum__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::conjectureGenGtEnum__option_t,
    std::string option,
    std::string value)
{
  d_holder->conjectureGenGtEnum =
    runHandlerAndPredicates(options::conjectureGenGtEnum, option, value, d_handler);
  d_holder->conjectureGenGtEnum__setByUser__ = true;
  Trace("options") << "user assigned option conjectureGenGtEnum" << std::endl;
  
}
template <> options::conjectureGenMaxDepth__option_t::type runHandlerAndPredicates(
    options::conjectureGenMaxDepth__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::conjectureGenMaxDepth__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::conjectureGenMaxDepth__option_t,
    std::string option,
    std::string value)
{
  d_holder->conjectureGenMaxDepth =
    runHandlerAndPredicates(options::conjectureGenMaxDepth, option, value, d_handler);
  d_holder->conjectureGenMaxDepth__setByUser__ = true;
  Trace("options") << "user assigned option conjectureGenMaxDepth" << std::endl;
  
}
template <> options::conjectureGenPerRound__option_t::type runHandlerAndPredicates(
    options::conjectureGenPerRound__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::conjectureGenPerRound__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::conjectureGenPerRound__option_t,
    std::string option,
    std::string value)
{
  d_holder->conjectureGenPerRound =
    runHandlerAndPredicates(options::conjectureGenPerRound, option, value, d_handler);
  d_holder->conjectureGenPerRound__setByUser__ = true;
  Trace("options") << "user assigned option conjectureGenPerRound" << std::endl;
  
}
template <> void Options::assignBool(
    options::conjectureUeeIntro__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::conjectureUeeIntro, option, value, d_handler);
  d_holder->conjectureUeeIntro = value;
  d_holder->conjectureUeeIntro__setByUser__ = true;
  Trace("options") << "user assigned option conjectureUeeIntro" << std::endl;
  
}
template <> void Options::assignBool(
    options::conjectureNoFilter__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::conjectureNoFilter, option, value, d_handler);
  d_holder->conjectureNoFilter = value;
  d_holder->conjectureNoFilter__setByUser__ = true;
  Trace("options") << "user assigned option conjectureNoFilter" << std::endl;
  
}
template <> void Options::assignBool(
    options::dtStcInduction__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dtStcInduction, option, value, d_handler);
  d_holder->dtStcInduction = value;
  d_holder->dtStcInduction__setByUser__ = true;
  Trace("options") << "user assigned option dtStcInduction" << std::endl;
  
}
template <> void Options::assignBool(
    options::dtVarExpandQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dtVarExpandQuant, option, value, d_handler);
  d_holder->dtVarExpandQuant = value;
  d_holder->dtVarExpandQuant__setByUser__ = true;
  Trace("options") << "user assigned option dtVarExpandQuant" << std::endl;
  
}
template <> void Options::assignBool(
    options::eMatching__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::eMatching, option, value, d_handler);
  d_holder->eMatching = value;
  d_holder->eMatching__setByUser__ = true;
  Trace("options") << "user assigned option eMatching" << std::endl;
  
}
template <> void Options::assignBool(
    options::elimExtArithQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::elimExtArithQuant, option, value, d_handler);
  d_holder->elimExtArithQuant = value;
  d_holder->elimExtArithQuant__setByUser__ = true;
  Trace("options") << "user assigned option elimExtArithQuant" << std::endl;
  
}
template <> void Options::assignBool(
    options::elimTautQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::elimTautQuant, option, value, d_handler);
  d_holder->elimTautQuant = value;
  d_holder->elimTautQuant__setByUser__ = true;
  Trace("options") << "user assigned option elimTautQuant" << std::endl;
  
}
template <> void Options::assignBool(
    options::extRewriteQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::extRewriteQuant, option, value, d_handler);
  d_holder->extRewriteQuant = value;
  d_holder->extRewriteQuant__setByUser__ = true;
  Trace("options") << "user assigned option extRewriteQuant" << std::endl;
  
}
template <> void Options::assignBool(
    options::finiteModelFind__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::finiteModelFind, option, value, d_handler);
  d_holder->finiteModelFind = value;
  d_holder->finiteModelFind__setByUser__ = true;
  Trace("options") << "user assigned option finiteModelFind" << std::endl;
  
}
template <> void Options::assignBool(
    options::fmfBound__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fmfBound, option, value, d_handler);
  d_holder->fmfBound = value;
  d_holder->fmfBound__setByUser__ = true;
  Trace("options") << "user assigned option fmfBound" << std::endl;
  
}
template <> void Options::assignBool(
    options::fmfBoundInt__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fmfBoundInt, option, value, d_handler);
  d_holder->fmfBoundInt = value;
  d_holder->fmfBoundInt__setByUser__ = true;
  Trace("options") << "user assigned option fmfBoundInt" << std::endl;
  
}
template <> void Options::assignBool(
    options::fmfBoundLazy__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fmfBoundLazy, option, value, d_handler);
  d_holder->fmfBoundLazy = value;
  d_holder->fmfBoundLazy__setByUser__ = true;
  Trace("options") << "user assigned option fmfBoundLazy" << std::endl;
  
}
template <> void Options::assignBool(
    options::fmfFmcSimple__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fmfFmcSimple, option, value, d_handler);
  d_holder->fmfFmcSimple = value;
  d_holder->fmfFmcSimple__setByUser__ = true;
  Trace("options") << "user assigned option fmfFmcSimple" << std::endl;
  
}
template <> void Options::assignBool(
    options::fmfFreshDistConst__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fmfFreshDistConst, option, value, d_handler);
  d_holder->fmfFreshDistConst = value;
  d_holder->fmfFreshDistConst__setByUser__ = true;
  Trace("options") << "user assigned option fmfFreshDistConst" << std::endl;
  
}
template <> void Options::assignBool(
    options::fmfFunWellDefined__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fmfFunWellDefined, option, value, d_handler);
  d_holder->fmfFunWellDefined = value;
  d_holder->fmfFunWellDefined__setByUser__ = true;
  Trace("options") << "user assigned option fmfFunWellDefined" << std::endl;
  
}
template <> void Options::assignBool(
    options::fmfFunWellDefinedRelevant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fmfFunWellDefinedRelevant, option, value, d_handler);
  d_holder->fmfFunWellDefinedRelevant = value;
  d_holder->fmfFunWellDefinedRelevant__setByUser__ = true;
  Trace("options") << "user assigned option fmfFunWellDefinedRelevant" << std::endl;
  
}
template <> void Options::assignBool(
    options::fmfInstEngine__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fmfInstEngine, option, value, d_handler);
  d_holder->fmfInstEngine = value;
  d_holder->fmfInstEngine__setByUser__ = true;
  Trace("options") << "user assigned option fmfInstEngine" << std::endl;
  
}
template <> options::fmfTypeCompletionThresh__option_t::type runHandlerAndPredicates(
    options::fmfTypeCompletionThresh__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::fmfTypeCompletionThresh__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::fmfTypeCompletionThresh__option_t,
    std::string option,
    std::string value)
{
  d_holder->fmfTypeCompletionThresh =
    runHandlerAndPredicates(options::fmfTypeCompletionThresh, option, value, d_handler);
  d_holder->fmfTypeCompletionThresh__setByUser__ = true;
  Trace("options") << "user assigned option fmfTypeCompletionThresh" << std::endl;
  
}
template <> void Options::assignBool(
    options::fullSaturateInterleave__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fullSaturateInterleave, option, value, d_handler);
  d_holder->fullSaturateInterleave = value;
  d_holder->fullSaturateInterleave__setByUser__ = true;
  Trace("options") << "user assigned option fullSaturateInterleave" << std::endl;
  
}
template <> void Options::assignBool(
    options::fullSaturateStratify__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fullSaturateStratify, option, value, d_handler);
  d_holder->fullSaturateStratify = value;
  d_holder->fullSaturateStratify__setByUser__ = true;
  Trace("options") << "user assigned option fullSaturateStratify" << std::endl;
  
}
template <> void Options::assignBool(
    options::fullSaturateQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fullSaturateQuant, option, value, d_handler);
  d_holder->fullSaturateQuant = value;
  d_holder->fullSaturateQuant__setByUser__ = true;
  Trace("options") << "user assigned option fullSaturateQuant" << std::endl;
  
}
template <> options::fullSaturateLimit__option_t::type runHandlerAndPredicates(
    options::fullSaturateLimit__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::fullSaturateLimit__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::fullSaturateLimit__option_t,
    std::string option,
    std::string value)
{
  d_holder->fullSaturateLimit =
    runHandlerAndPredicates(options::fullSaturateLimit, option, value, d_handler);
  d_holder->fullSaturateLimit__setByUser__ = true;
  Trace("options") << "user assigned option fullSaturateLimit" << std::endl;
  
}
template <> void Options::assignBool(
    options::fullSaturateQuantRd__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fullSaturateQuantRd, option, value, d_handler);
  d_holder->fullSaturateQuantRd = value;
  d_holder->fullSaturateQuantRd__setByUser__ = true;
  Trace("options") << "user assigned option fullSaturateQuantRd" << std::endl;
  
}
template <> void Options::assignBool(
    options::globalNegate__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::globalNegate, option, value, d_handler);
  d_holder->globalNegate = value;
  d_holder->globalNegate__setByUser__ = true;
  Trace("options") << "user assigned option globalNegate" << std::endl;
  
}
template <> void Options::assignBool(
    options::hoElim__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::hoElim, option, value, d_handler);
  d_holder->hoElim = value;
  d_holder->hoElim__setByUser__ = true;
  Trace("options") << "user assigned option hoElim" << std::endl;
  
}
template <> void Options::assignBool(
    options::hoElimStoreAx__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::hoElimStoreAx, option, value, d_handler);
  d_holder->hoElimStoreAx = value;
  d_holder->hoElimStoreAx__setByUser__ = true;
  Trace("options") << "user assigned option hoElimStoreAx" << std::endl;
  
}
template <> void Options::assignBool(
    options::hoMatching__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::hoMatching, option, value, d_handler);
  d_holder->hoMatching = value;
  d_holder->hoMatching__setByUser__ = true;
  Trace("options") << "user assigned option hoMatching" << std::endl;
  
}
template <> void Options::assignBool(
    options::hoMatchingVarArgPriority__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::hoMatchingVarArgPriority, option, value, d_handler);
  d_holder->hoMatchingVarArgPriority = value;
  d_holder->hoMatchingVarArgPriority__setByUser__ = true;
  Trace("options") << "user assigned option hoMatchingVarArgPriority" << std::endl;
  
}
template <> void Options::assignBool(
    options::hoMergeTermDb__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::hoMergeTermDb, option, value, d_handler);
  d_holder->hoMergeTermDb = value;
  d_holder->hoMergeTermDb__setByUser__ = true;
  Trace("options") << "user assigned option hoMergeTermDb" << std::endl;
  
}
template <> void Options::assignBool(
    options::incrementTriggers__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::incrementTriggers, option, value, d_handler);
  d_holder->incrementTriggers = value;
  d_holder->incrementTriggers__setByUser__ = true;
  Trace("options") << "user assigned option incrementTriggers" << std::endl;
  
}
template <> void Options::assignBool(
    options::instLevelInputOnly__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::instLevelInputOnly, option, value, d_handler);
  d_holder->instLevelInputOnly = value;
  d_holder->instLevelInputOnly__setByUser__ = true;
  Trace("options") << "user assigned option instLevelInputOnly" << std::endl;
  
}
template <> options::instMaxLevel__option_t::type runHandlerAndPredicates(
    options::instMaxLevel__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::instMaxLevel__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::instMaxLevel__option_t,
    std::string option,
    std::string value)
{
  d_holder->instMaxLevel =
    runHandlerAndPredicates(options::instMaxLevel, option, value, d_handler);
  d_holder->instMaxLevel__setByUser__ = true;
  Trace("options") << "user assigned option instMaxLevel" << std::endl;
  
}
template <> void Options::assignBool(
    options::instNoEntail__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::instNoEntail, option, value, d_handler);
  d_holder->instNoEntail = value;
  d_holder->instNoEntail__setByUser__ = true;
  Trace("options") << "user assigned option instNoEntail" << std::endl;
  
}
template <> void Options::assignBool(
    options::instNoModelTrue__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::instNoModelTrue, option, value, d_handler);
  d_holder->instNoModelTrue = value;
  d_holder->instNoModelTrue__setByUser__ = true;
  Trace("options") << "user assigned option instNoModelTrue" << std::endl;
  
}
template <> options::instWhenPhase__option_t::type runHandlerAndPredicates(
    options::instWhenPhase__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::instWhenPhase__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::instWhenPhase__option_t,
    std::string option,
    std::string value)
{
  d_holder->instWhenPhase =
    runHandlerAndPredicates(options::instWhenPhase, option, value, d_handler);
  d_holder->instWhenPhase__setByUser__ = true;
  Trace("options") << "user assigned option instWhenPhase" << std::endl;
  
}
template <> void Options::assignBool(
    options::instWhenStrictInterleave__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::instWhenStrictInterleave, option, value, d_handler);
  d_holder->instWhenStrictInterleave = value;
  d_holder->instWhenStrictInterleave__setByUser__ = true;
  Trace("options") << "user assigned option instWhenStrictInterleave" << std::endl;
  
}
template <> void Options::assignBool(
    options::instWhenTcFirst__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::instWhenTcFirst, option, value, d_handler);
  d_holder->instWhenTcFirst = value;
  d_holder->instWhenTcFirst__setByUser__ = true;
  Trace("options") << "user assigned option instWhenTcFirst" << std::endl;
  
}
template <> options::instWhenMode__option_t::type runHandlerAndPredicates(
    options::instWhenMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::instWhenMode__option_t::type retval = stringToInstWhenMode(option, optionarg);
  handler->checkInstWhenMode(option, retval);
  return retval;
}
template <> void Options::assign(
    options::instWhenMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->instWhenMode =
    runHandlerAndPredicates(options::instWhenMode, option, value, d_handler);
  d_holder->instWhenMode__setByUser__ = true;
  Trace("options") << "user assigned option instWhenMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::intWfInduction__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::intWfInduction, option, value, d_handler);
  d_holder->intWfInduction = value;
  d_holder->intWfInduction__setByUser__ = true;
  Trace("options") << "user assigned option intWfInduction" << std::endl;
  
}
template <> void Options::assignBool(
    options::iteDtTesterSplitQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::iteDtTesterSplitQuant, option, value, d_handler);
  d_holder->iteDtTesterSplitQuant = value;
  d_holder->iteDtTesterSplitQuant__setByUser__ = true;
  Trace("options") << "user assigned option iteDtTesterSplitQuant" << std::endl;
  
}
template <> options::iteLiftQuant__option_t::type runHandlerAndPredicates(
    options::iteLiftQuant__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::iteLiftQuant__option_t::type retval = stringToIteLiftQuantMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::iteLiftQuant__option_t,
    std::string option,
    std::string value)
{
  d_holder->iteLiftQuant =
    runHandlerAndPredicates(options::iteLiftQuant, option, value, d_handler);
  d_holder->iteLiftQuant__setByUser__ = true;
  Trace("options") << "user assigned option iteLiftQuant" << std::endl;
  
}
template <> options::literalMatchMode__option_t::type runHandlerAndPredicates(
    options::literalMatchMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::literalMatchMode__option_t::type retval = stringToLiteralMatchMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::literalMatchMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->literalMatchMode =
    runHandlerAndPredicates(options::literalMatchMode, option, value, d_handler);
  d_holder->literalMatchMode__setByUser__ = true;
  Trace("options") << "user assigned option literalMatchMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::lteRestrictInstClosure__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::lteRestrictInstClosure, option, value, d_handler);
  d_holder->lteRestrictInstClosure = value;
  d_holder->lteRestrictInstClosure__setByUser__ = true;
  Trace("options") << "user assigned option lteRestrictInstClosure" << std::endl;
  
}
template <> void Options::assignBool(
    options::macrosQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::macrosQuant, option, value, d_handler);
  d_holder->macrosQuant = value;
  d_holder->macrosQuant__setByUser__ = true;
  Trace("options") << "user assigned option macrosQuant" << std::endl;
  
}
template <> options::macrosQuantMode__option_t::type runHandlerAndPredicates(
    options::macrosQuantMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::macrosQuantMode__option_t::type retval = stringToMacrosQuantMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::macrosQuantMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->macrosQuantMode =
    runHandlerAndPredicates(options::macrosQuantMode, option, value, d_handler);
  d_holder->macrosQuantMode__setByUser__ = true;
  Trace("options") << "user assigned option macrosQuantMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::mbqiInterleave__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::mbqiInterleave, option, value, d_handler);
  d_holder->mbqiInterleave = value;
  d_holder->mbqiInterleave__setByUser__ = true;
  Trace("options") << "user assigned option mbqiInterleave" << std::endl;
  
}
template <> void Options::assignBool(
    options::fmfOneInstPerRound__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::fmfOneInstPerRound, option, value, d_handler);
  d_holder->fmfOneInstPerRound = value;
  d_holder->fmfOneInstPerRound__setByUser__ = true;
  Trace("options") << "user assigned option fmfOneInstPerRound" << std::endl;
  
}
template <> options::mbqiMode__option_t::type runHandlerAndPredicates(
    options::mbqiMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::mbqiMode__option_t::type retval = stringToMbqiMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::mbqiMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->mbqiMode =
    runHandlerAndPredicates(options::mbqiMode, option, value, d_handler);
  d_holder->mbqiMode__setByUser__ = true;
  Trace("options") << "user assigned option mbqiMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::miniscopeQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::miniscopeQuant, option, value, d_handler);
  d_holder->miniscopeQuant = value;
  d_holder->miniscopeQuant__setByUser__ = true;
  Trace("options") << "user assigned option miniscopeQuant" << std::endl;
  
}
template <> void Options::assignBool(
    options::miniscopeQuantFreeVar__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::miniscopeQuantFreeVar, option, value, d_handler);
  d_holder->miniscopeQuantFreeVar = value;
  d_holder->miniscopeQuantFreeVar__setByUser__ = true;
  Trace("options") << "user assigned option miniscopeQuantFreeVar" << std::endl;
  
}
template <> void Options::assignBool(
    options::multiTriggerCache__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::multiTriggerCache, option, value, d_handler);
  d_holder->multiTriggerCache = value;
  d_holder->multiTriggerCache__setByUser__ = true;
  Trace("options") << "user assigned option multiTriggerCache" << std::endl;
  
}
template <> void Options::assignBool(
    options::multiTriggerLinear__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::multiTriggerLinear, option, value, d_handler);
  d_holder->multiTriggerLinear = value;
  d_holder->multiTriggerLinear__setByUser__ = true;
  Trace("options") << "user assigned option multiTriggerLinear" << std::endl;
  
}
template <> void Options::assignBool(
    options::multiTriggerPriority__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::multiTriggerPriority, option, value, d_handler);
  d_holder->multiTriggerPriority = value;
  d_holder->multiTriggerPriority__setByUser__ = true;
  Trace("options") << "user assigned option multiTriggerPriority" << std::endl;
  
}
template <> void Options::assignBool(
    options::multiTriggerWhenSingle__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::multiTriggerWhenSingle, option, value, d_handler);
  d_holder->multiTriggerWhenSingle = value;
  d_holder->multiTriggerWhenSingle__setByUser__ = true;
  Trace("options") << "user assigned option multiTriggerWhenSingle" << std::endl;
  
}
template <> void Options::assignBool(
    options::partialTriggers__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::partialTriggers, option, value, d_handler);
  d_holder->partialTriggers = value;
  d_holder->partialTriggers__setByUser__ = true;
  Trace("options") << "user assigned option partialTriggers" << std::endl;
  
}
template <> void Options::assignBool(
    options::preSkolemQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::preSkolemQuant, option, value, d_handler);
  d_holder->preSkolemQuant = value;
  d_holder->preSkolemQuant__setByUser__ = true;
  Trace("options") << "user assigned option preSkolemQuant" << std::endl;
  
}
template <> void Options::assignBool(
    options::preSkolemQuantAgg__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::preSkolemQuantAgg, option, value, d_handler);
  d_holder->preSkolemQuantAgg = value;
  d_holder->preSkolemQuantAgg__setByUser__ = true;
  Trace("options") << "user assigned option preSkolemQuantAgg" << std::endl;
  
}
template <> void Options::assignBool(
    options::preSkolemQuantNested__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::preSkolemQuantNested, option, value, d_handler);
  d_holder->preSkolemQuantNested = value;
  d_holder->preSkolemQuantNested__setByUser__ = true;
  Trace("options") << "user assigned option preSkolemQuantNested" << std::endl;
  
}
template <> void Options::assignBool(
    options::prenexQuantUser__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::prenexQuantUser, option, value, d_handler);
  d_holder->prenexQuantUser = value;
  d_holder->prenexQuantUser__setByUser__ = true;
  Trace("options") << "user assigned option prenexQuantUser" << std::endl;
  
}
template <> options::prenexQuant__option_t::type runHandlerAndPredicates(
    options::prenexQuant__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::prenexQuant__option_t::type retval = stringToPrenexQuantMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::prenexQuant__option_t,
    std::string option,
    std::string value)
{
  d_holder->prenexQuant =
    runHandlerAndPredicates(options::prenexQuant, option, value, d_handler);
  d_holder->prenexQuant__setByUser__ = true;
  Trace("options") << "user assigned option prenexQuant" << std::endl;
  
}
template <> void Options::assignBool(
    options::pureThTriggers__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::pureThTriggers, option, value, d_handler);
  d_holder->pureThTriggers = value;
  d_holder->pureThTriggers__setByUser__ = true;
  Trace("options") << "user assigned option pureThTriggers" << std::endl;
  
}
template <> void Options::assignBool(
    options::purifyDtTriggers__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::purifyDtTriggers, option, value, d_handler);
  d_holder->purifyDtTriggers = value;
  d_holder->purifyDtTriggers__setByUser__ = true;
  Trace("options") << "user assigned option purifyDtTriggers" << std::endl;
  
}
template <> void Options::assignBool(
    options::purifyTriggers__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::purifyTriggers, option, value, d_handler);
  d_holder->purifyTriggers = value;
  d_holder->purifyTriggers__setByUser__ = true;
  Trace("options") << "user assigned option purifyTriggers" << std::endl;
  
}
template <> void Options::assignBool(
    options::qcfAllConflict__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::qcfAllConflict, option, value, d_handler);
  d_holder->qcfAllConflict = value;
  d_holder->qcfAllConflict__setByUser__ = true;
  Trace("options") << "user assigned option qcfAllConflict" << std::endl;
  
}
template <> void Options::assignBool(
    options::qcfEagerCheckRd__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::qcfEagerCheckRd, option, value, d_handler);
  d_holder->qcfEagerCheckRd = value;
  d_holder->qcfEagerCheckRd__setByUser__ = true;
  Trace("options") << "user assigned option qcfEagerCheckRd" << std::endl;
  
}
template <> void Options::assignBool(
    options::qcfEagerTest__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::qcfEagerTest, option, value, d_handler);
  d_holder->qcfEagerTest = value;
  d_holder->qcfEagerTest__setByUser__ = true;
  Trace("options") << "user assigned option qcfEagerTest" << std::endl;
  
}
template <> void Options::assignBool(
    options::qcfNestedConflict__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::qcfNestedConflict, option, value, d_handler);
  d_holder->qcfNestedConflict = value;
  d_holder->qcfNestedConflict__setByUser__ = true;
  Trace("options") << "user assigned option qcfNestedConflict" << std::endl;
  
}
template <> void Options::assignBool(
    options::qcfSkipRd__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::qcfSkipRd, option, value, d_handler);
  d_holder->qcfSkipRd = value;
  d_holder->qcfSkipRd__setByUser__ = true;
  Trace("options") << "user assigned option qcfSkipRd" << std::endl;
  
}
template <> void Options::assignBool(
    options::qcfTConstraint__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::qcfTConstraint, option, value, d_handler);
  d_holder->qcfTConstraint = value;
  d_holder->qcfTConstraint__setByUser__ = true;
  Trace("options") << "user assigned option qcfTConstraint" << std::endl;
  
}
template <> void Options::assignBool(
    options::qcfVoExp__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::qcfVoExp, option, value, d_handler);
  d_holder->qcfVoExp = value;
  d_holder->qcfVoExp__setByUser__ = true;
  Trace("options") << "user assigned option qcfVoExp" << std::endl;
  
}
template <> void Options::assignBool(
    options::quantAlphaEquiv__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::quantAlphaEquiv, option, value, d_handler);
  d_holder->quantAlphaEquiv = value;
  d_holder->quantAlphaEquiv__setByUser__ = true;
  Trace("options") << "user assigned option quantAlphaEquiv" << std::endl;
  
}
template <> void Options::assignBool(
    options::quantAntiSkolem__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::quantAntiSkolem, option, value, d_handler);
  d_holder->quantAntiSkolem = value;
  d_holder->quantAntiSkolem__setByUser__ = true;
  Trace("options") << "user assigned option quantAntiSkolem" << std::endl;
  
}
template <> void Options::assignBool(
    options::quantConflictFind__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::quantConflictFind, option, value, d_handler);
  d_holder->quantConflictFind = value;
  d_holder->quantConflictFind__setByUser__ = true;
  Trace("options") << "user assigned option quantConflictFind" << std::endl;
  
}
template <> options::qcfMode__option_t::type runHandlerAndPredicates(
    options::qcfMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::qcfMode__option_t::type retval = stringToQcfMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::qcfMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->qcfMode =
    runHandlerAndPredicates(options::qcfMode, option, value, d_handler);
  d_holder->qcfMode__setByUser__ = true;
  Trace("options") << "user assigned option qcfMode" << std::endl;
  
}
template <> options::qcfWhenMode__option_t::type runHandlerAndPredicates(
    options::qcfWhenMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::qcfWhenMode__option_t::type retval = stringToQcfWhenMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::qcfWhenMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->qcfWhenMode =
    runHandlerAndPredicates(options::qcfWhenMode, option, value, d_handler);
  d_holder->qcfWhenMode__setByUser__ = true;
  Trace("options") << "user assigned option qcfWhenMode" << std::endl;
  
}
template <> options::quantDynamicSplit__option_t::type runHandlerAndPredicates(
    options::quantDynamicSplit__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::quantDynamicSplit__option_t::type retval = stringToQuantDSplitMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::quantDynamicSplit__option_t,
    std::string option,
    std::string value)
{
  d_holder->quantDynamicSplit =
    runHandlerAndPredicates(options::quantDynamicSplit, option, value, d_handler);
  d_holder->quantDynamicSplit__setByUser__ = true;
  Trace("options") << "user assigned option quantDynamicSplit" << std::endl;
  
}
template <> void Options::assignBool(
    options::quantEpr__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::quantEpr, option, value, d_handler);
  d_holder->quantEpr = value;
  d_holder->quantEpr__setByUser__ = true;
  Trace("options") << "user assigned option quantEpr" << std::endl;
  
}
template <> void Options::assignBool(
    options::quantEprMatching__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::quantEprMatching, option, value, d_handler);
  d_holder->quantEprMatching = value;
  d_holder->quantEprMatching__setByUser__ = true;
  Trace("options") << "user assigned option quantEprMatching" << std::endl;
  
}
template <> void Options::assignBool(
    options::quantFunWellDefined__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::quantFunWellDefined, option, value, d_handler);
  d_holder->quantFunWellDefined = value;
  d_holder->quantFunWellDefined__setByUser__ = true;
  Trace("options") << "user assigned option quantFunWellDefined" << std::endl;
  
}
template <> void Options::assignBool(
    options::quantInduction__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::quantInduction, option, value, d_handler);
  d_holder->quantInduction = value;
  d_holder->quantInduction__setByUser__ = true;
  Trace("options") << "user assigned option quantInduction" << std::endl;
  
}
template <> void Options::assignBool(
    options::quantModelEe__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::quantModelEe, option, value, d_handler);
  d_holder->quantModelEe = value;
  d_holder->quantModelEe__setByUser__ = true;
  Trace("options") << "user assigned option quantModelEe" << std::endl;
  
}
template <> options::quantRepMode__option_t::type runHandlerAndPredicates(
    options::quantRepMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::quantRepMode__option_t::type retval = stringToQuantRepMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::quantRepMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->quantRepMode =
    runHandlerAndPredicates(options::quantRepMode, option, value, d_handler);
  d_holder->quantRepMode__setByUser__ = true;
  Trace("options") << "user assigned option quantRepMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::quantSplit__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::quantSplit, option, value, d_handler);
  d_holder->quantSplit = value;
  d_holder->quantSplit__setByUser__ = true;
  Trace("options") << "user assigned option quantSplit" << std::endl;
  
}
template <> void Options::assignBool(
    options::registerQuantBodyTerms__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::registerQuantBodyTerms, option, value, d_handler);
  d_holder->registerQuantBodyTerms = value;
  d_holder->registerQuantBodyTerms__setByUser__ = true;
  Trace("options") << "user assigned option registerQuantBodyTerms" << std::endl;
  
}
template <> void Options::assignBool(
    options::relationalTriggers__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::relationalTriggers, option, value, d_handler);
  d_holder->relationalTriggers = value;
  d_holder->relationalTriggers__setByUser__ = true;
  Trace("options") << "user assigned option relationalTriggers" << std::endl;
  
}
template <> void Options::assignBool(
    options::relevantTriggers__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::relevantTriggers, option, value, d_handler);
  d_holder->relevantTriggers = value;
  d_holder->relevantTriggers__setByUser__ = true;
  Trace("options") << "user assigned option relevantTriggers" << std::endl;
  
}
template <> void Options::assignBool(
    options::strictTriggers__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::strictTriggers, option, value, d_handler);
  d_holder->strictTriggers = value;
  d_holder->strictTriggers__setByUser__ = true;
  Trace("options") << "user assigned option strictTriggers" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygus__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygus, option, value, d_handler);
  d_holder->sygus = value;
  d_holder->sygus__setByUser__ = true;
  Trace("options") << "user assigned option sygus" << std::endl;
  
}
template <> options::sygusActiveGenEnumConsts__option_t::type runHandlerAndPredicates(
    options::sygusActiveGenEnumConsts__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusActiveGenEnumConsts__option_t::type retval = handleOption<unsigned long>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusActiveGenEnumConsts__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusActiveGenEnumConsts =
    runHandlerAndPredicates(options::sygusActiveGenEnumConsts, option, value, d_handler);
  d_holder->sygusActiveGenEnumConsts__setByUser__ = true;
  Trace("options") << "user assigned option sygusActiveGenEnumConsts" << std::endl;
  
}
template <> options::sygusActiveGenMode__option_t::type runHandlerAndPredicates(
    options::sygusActiveGenMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusActiveGenMode__option_t::type retval = stringToSygusActiveGenMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusActiveGenMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusActiveGenMode =
    runHandlerAndPredicates(options::sygusActiveGenMode, option, value, d_handler);
  d_holder->sygusActiveGenMode__setByUser__ = true;
  Trace("options") << "user assigned option sygusActiveGenMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusAddConstGrammar__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusAddConstGrammar, option, value, d_handler);
  d_holder->sygusAddConstGrammar = value;
  d_holder->sygusAddConstGrammar__setByUser__ = true;
  Trace("options") << "user assigned option sygusAddConstGrammar" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusArgRelevant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusArgRelevant, option, value, d_handler);
  d_holder->sygusArgRelevant = value;
  d_holder->sygusArgRelevant__setByUser__ = true;
  Trace("options") << "user assigned option sygusArgRelevant" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusInvAutoUnfold__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusInvAutoUnfold, option, value, d_handler);
  d_holder->sygusInvAutoUnfold = value;
  d_holder->sygusInvAutoUnfold__setByUser__ = true;
  Trace("options") << "user assigned option sygusInvAutoUnfold" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusBoolIteReturnConst__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusBoolIteReturnConst, option, value, d_handler);
  d_holder->sygusBoolIteReturnConst = value;
  d_holder->sygusBoolIteReturnConst__setByUser__ = true;
  Trace("options") << "user assigned option sygusBoolIteReturnConst" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusCoreConnective__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusCoreConnective, option, value, d_handler);
  d_holder->sygusCoreConnective = value;
  d_holder->sygusCoreConnective__setByUser__ = true;
  Trace("options") << "user assigned option sygusCoreConnective" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusConstRepairAbort__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusConstRepairAbort, option, value, d_handler);
  d_holder->sygusConstRepairAbort = value;
  d_holder->sygusConstRepairAbort__setByUser__ = true;
  Trace("options") << "user assigned option sygusConstRepairAbort" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusEvalOpt__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusEvalOpt, option, value, d_handler);
  d_holder->sygusEvalOpt = value;
  d_holder->sygusEvalOpt__setByUser__ = true;
  Trace("options") << "user assigned option sygusEvalOpt" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusEvalUnfold__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusEvalUnfold, option, value, d_handler);
  d_holder->sygusEvalUnfold = value;
  d_holder->sygusEvalUnfold__setByUser__ = true;
  Trace("options") << "user assigned option sygusEvalUnfold" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusEvalUnfoldBool__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusEvalUnfoldBool, option, value, d_handler);
  d_holder->sygusEvalUnfoldBool = value;
  d_holder->sygusEvalUnfoldBool__setByUser__ = true;
  Trace("options") << "user assigned option sygusEvalUnfoldBool" << std::endl;
  
}
template <> options::sygusExprMinerCheckTimeout__option_t::type runHandlerAndPredicates(
    options::sygusExprMinerCheckTimeout__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusExprMinerCheckTimeout__option_t::type retval = handleOption<unsigned long>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusExprMinerCheckTimeout__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusExprMinerCheckTimeout =
    runHandlerAndPredicates(options::sygusExprMinerCheckTimeout, option, value, d_handler);
  d_holder->sygusExprMinerCheckTimeout__setByUser__ = true;
  Trace("options") << "user assigned option sygusExprMinerCheckTimeout" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusExprMinerCheckUseExport__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusExprMinerCheckUseExport, option, value, d_handler);
  d_holder->sygusExprMinerCheckUseExport = value;
  d_holder->sygusExprMinerCheckUseExport__setByUser__ = true;
  Trace("options") << "user assigned option sygusExprMinerCheckUseExport" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusExtRew__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusExtRew, option, value, d_handler);
  d_holder->sygusExtRew = value;
  d_holder->sygusExtRew__setByUser__ = true;
  Trace("options") << "user assigned option sygusExtRew" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusFilterSolRevSubsume__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusFilterSolRevSubsume, option, value, d_handler);
  d_holder->sygusFilterSolRevSubsume = value;
  d_holder->sygusFilterSolRevSubsume__setByUser__ = true;
  Trace("options") << "user assigned option sygusFilterSolRevSubsume" << std::endl;
  
}
template <> options::sygusFilterSolMode__option_t::type runHandlerAndPredicates(
    options::sygusFilterSolMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusFilterSolMode__option_t::type retval = stringToSygusFilterSolMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusFilterSolMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusFilterSolMode =
    runHandlerAndPredicates(options::sygusFilterSolMode, option, value, d_handler);
  d_holder->sygusFilterSolMode__setByUser__ = true;
  Trace("options") << "user assigned option sygusFilterSolMode" << std::endl;
  
}
template <> options::sygusGrammarConsMode__option_t::type runHandlerAndPredicates(
    options::sygusGrammarConsMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusGrammarConsMode__option_t::type retval = stringToSygusGrammarConsMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusGrammarConsMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusGrammarConsMode =
    runHandlerAndPredicates(options::sygusGrammarConsMode, option, value, d_handler);
  d_holder->sygusGrammarConsMode__setByUser__ = true;
  Trace("options") << "user assigned option sygusGrammarConsMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusGrammarNorm__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusGrammarNorm, option, value, d_handler);
  d_holder->sygusGrammarNorm = value;
  d_holder->sygusGrammarNorm__setByUser__ = true;
  Trace("options") << "user assigned option sygusGrammarNorm" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusInference__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusInference, option, value, d_handler);
  d_holder->sygusInference = value;
  d_holder->sygusInference__setByUser__ = true;
  Trace("options") << "user assigned option sygusInference" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusInst__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusInst, option, value, d_handler);
  d_holder->sygusInst = value;
  d_holder->sygusInst__setByUser__ = true;
  Trace("options") << "user assigned option sygusInst" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusInvTemplWhenSyntax__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusInvTemplWhenSyntax, option, value, d_handler);
  d_holder->sygusInvTemplWhenSyntax = value;
  d_holder->sygusInvTemplWhenSyntax__setByUser__ = true;
  Trace("options") << "user assigned option sygusInvTemplWhenSyntax" << std::endl;
  
}
template <> options::sygusInvTemplMode__option_t::type runHandlerAndPredicates(
    options::sygusInvTemplMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusInvTemplMode__option_t::type retval = stringToSygusInvTemplMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusInvTemplMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusInvTemplMode =
    runHandlerAndPredicates(options::sygusInvTemplMode, option, value, d_handler);
  d_holder->sygusInvTemplMode__setByUser__ = true;
  Trace("options") << "user assigned option sygusInvTemplMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusMinGrammar__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusMinGrammar, option, value, d_handler);
  d_holder->sygusMinGrammar = value;
  d_holder->sygusMinGrammar__setByUser__ = true;
  Trace("options") << "user assigned option sygusMinGrammar" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusUnifPbe__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusUnifPbe, option, value, d_handler);
  d_holder->sygusUnifPbe = value;
  d_holder->sygusUnifPbe__setByUser__ = true;
  Trace("options") << "user assigned option sygusUnifPbe" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusPbeMultiFair__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusPbeMultiFair, option, value, d_handler);
  d_holder->sygusPbeMultiFair = value;
  d_holder->sygusPbeMultiFair__setByUser__ = true;
  Trace("options") << "user assigned option sygusPbeMultiFair" << std::endl;
  
}
template <> options::sygusPbeMultiFairDiff__option_t::type runHandlerAndPredicates(
    options::sygusPbeMultiFairDiff__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusPbeMultiFairDiff__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusPbeMultiFairDiff__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusPbeMultiFairDiff =
    runHandlerAndPredicates(options::sygusPbeMultiFairDiff, option, value, d_handler);
  d_holder->sygusPbeMultiFairDiff__setByUser__ = true;
  Trace("options") << "user assigned option sygusPbeMultiFairDiff" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusQePreproc__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusQePreproc, option, value, d_handler);
  d_holder->sygusQePreproc = value;
  d_holder->sygusQePreproc__setByUser__ = true;
  Trace("options") << "user assigned option sygusQePreproc" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusQueryGen__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusQueryGen, option, value, d_handler);
  d_holder->sygusQueryGen = value;
  d_holder->sygusQueryGen__setByUser__ = true;
  Trace("options") << "user assigned option sygusQueryGen" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusQueryGenCheck__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusQueryGenCheck, option, value, d_handler);
  d_holder->sygusQueryGenCheck = value;
  d_holder->sygusQueryGenCheck__setByUser__ = true;
  Trace("options") << "user assigned option sygusQueryGenCheck" << std::endl;
  
}
template <> options::sygusQueryGenDumpFiles__option_t::type runHandlerAndPredicates(
    options::sygusQueryGenDumpFiles__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusQueryGenDumpFiles__option_t::type retval = stringToSygusQueryDumpFilesMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusQueryGenDumpFiles__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusQueryGenDumpFiles =
    runHandlerAndPredicates(options::sygusQueryGenDumpFiles, option, value, d_handler);
  d_holder->sygusQueryGenDumpFiles__setByUser__ = true;
  Trace("options") << "user assigned option sygusQueryGenDumpFiles" << std::endl;
  
}
template <> options::sygusQueryGenThresh__option_t::type runHandlerAndPredicates(
    options::sygusQueryGenThresh__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusQueryGenThresh__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusQueryGenThresh__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusQueryGenThresh =
    runHandlerAndPredicates(options::sygusQueryGenThresh, option, value, d_handler);
  d_holder->sygusQueryGenThresh__setByUser__ = true;
  Trace("options") << "user assigned option sygusQueryGenThresh" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRecFun__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRecFun, option, value, d_handler);
  d_holder->sygusRecFun = value;
  d_holder->sygusRecFun__setByUser__ = true;
  Trace("options") << "user assigned option sygusRecFun" << std::endl;
  
}
template <> options::sygusRecFunEvalLimit__option_t::type runHandlerAndPredicates(
    options::sygusRecFunEvalLimit__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusRecFunEvalLimit__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusRecFunEvalLimit__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusRecFunEvalLimit =
    runHandlerAndPredicates(options::sygusRecFunEvalLimit, option, value, d_handler);
  d_holder->sygusRecFunEvalLimit__setByUser__ = true;
  Trace("options") << "user assigned option sygusRecFunEvalLimit" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRepairConst__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRepairConst, option, value, d_handler);
  d_holder->sygusRepairConst = value;
  d_holder->sygusRepairConst__setByUser__ = true;
  Trace("options") << "user assigned option sygusRepairConst" << std::endl;
  
}
template <> options::sygusRepairConstTimeout__option_t::type runHandlerAndPredicates(
    options::sygusRepairConstTimeout__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusRepairConstTimeout__option_t::type retval = handleOption<unsigned long>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusRepairConstTimeout__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusRepairConstTimeout =
    runHandlerAndPredicates(options::sygusRepairConstTimeout, option, value, d_handler);
  d_holder->sygusRepairConstTimeout__setByUser__ = true;
  Trace("options") << "user assigned option sygusRepairConstTimeout" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRew__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRew, option, value, d_handler);
  d_holder->sygusRew = value;
  d_holder->sygusRew__setByUser__ = true;
  Trace("options") << "user assigned option sygusRew" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRewSynth__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRewSynth, option, value, d_handler);
  d_holder->sygusRewSynth = value;
  d_holder->sygusRewSynth__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewSynth" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRewSynthAccel__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRewSynthAccel, option, value, d_handler);
  d_holder->sygusRewSynthAccel = value;
  d_holder->sygusRewSynthAccel__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewSynthAccel" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRewSynthCheck__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRewSynthCheck, option, value, d_handler);
  d_holder->sygusRewSynthCheck = value;
  d_holder->sygusRewSynthCheck__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewSynthCheck" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRewSynthFilterCong__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRewSynthFilterCong, option, value, d_handler);
  d_holder->sygusRewSynthFilterCong = value;
  d_holder->sygusRewSynthFilterCong__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewSynthFilterCong" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRewSynthFilterMatch__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRewSynthFilterMatch, option, value, d_handler);
  d_holder->sygusRewSynthFilterMatch = value;
  d_holder->sygusRewSynthFilterMatch__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewSynthFilterMatch" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRewSynthFilterNonLinear__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRewSynthFilterNonLinear, option, value, d_handler);
  d_holder->sygusRewSynthFilterNonLinear = value;
  d_holder->sygusRewSynthFilterNonLinear__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewSynthFilterNonLinear" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRewSynthFilterOrder__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRewSynthFilterOrder, option, value, d_handler);
  d_holder->sygusRewSynthFilterOrder = value;
  d_holder->sygusRewSynthFilterOrder__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewSynthFilterOrder" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRewSynthInput__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRewSynthInput, option, value, d_handler);
  d_holder->sygusRewSynthInput = value;
  d_holder->sygusRewSynthInput__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewSynthInput" << std::endl;
  
}
template <> options::sygusRewSynthInputNVars__option_t::type runHandlerAndPredicates(
    options::sygusRewSynthInputNVars__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusRewSynthInputNVars__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusRewSynthInputNVars__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusRewSynthInputNVars =
    runHandlerAndPredicates(options::sygusRewSynthInputNVars, option, value, d_handler);
  d_holder->sygusRewSynthInputNVars__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewSynthInputNVars" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRewSynthInputUseBool__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRewSynthInputUseBool, option, value, d_handler);
  d_holder->sygusRewSynthInputUseBool = value;
  d_holder->sygusRewSynthInputUseBool__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewSynthInputUseBool" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRewSynthRec__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRewSynthRec, option, value, d_handler);
  d_holder->sygusRewSynthRec = value;
  d_holder->sygusRewSynthRec__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewSynthRec" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRewVerify__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRewVerify, option, value, d_handler);
  d_holder->sygusRewVerify = value;
  d_holder->sygusRewVerify__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewVerify" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusRewVerifyAbort__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusRewVerifyAbort, option, value, d_handler);
  d_holder->sygusRewVerifyAbort = value;
  d_holder->sygusRewVerifyAbort__setByUser__ = true;
  Trace("options") << "user assigned option sygusRewVerifyAbort" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusSampleFpUniform__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusSampleFpUniform, option, value, d_handler);
  d_holder->sygusSampleFpUniform = value;
  d_holder->sygusSampleFpUniform__setByUser__ = true;
  Trace("options") << "user assigned option sygusSampleFpUniform" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusSampleGrammar__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusSampleGrammar, option, value, d_handler);
  d_holder->sygusSampleGrammar = value;
  d_holder->sygusSampleGrammar__setByUser__ = true;
  Trace("options") << "user assigned option sygusSampleGrammar" << std::endl;
  
}
template <> options::sygusSamples__option_t::type runHandlerAndPredicates(
    options::sygusSamples__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusSamples__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusSamples__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusSamples =
    runHandlerAndPredicates(options::sygusSamples, option, value, d_handler);
  d_holder->sygusSamples__setByUser__ = true;
  Trace("options") << "user assigned option sygusSamples" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiSingleInvAbort__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiSingleInvAbort, option, value, d_handler);
  d_holder->cegqiSingleInvAbort = value;
  d_holder->cegqiSingleInvAbort__setByUser__ = true;
  Trace("options") << "user assigned option cegqiSingleInvAbort" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiSingleInvPartial__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiSingleInvPartial, option, value, d_handler);
  d_holder->cegqiSingleInvPartial = value;
  d_holder->cegqiSingleInvPartial__setByUser__ = true;
  Trace("options") << "user assigned option cegqiSingleInvPartial" << std::endl;
  
}
template <> options::cegqiSingleInvReconstructLimit__option_t::type runHandlerAndPredicates(
    options::cegqiSingleInvReconstructLimit__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::cegqiSingleInvReconstructLimit__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::cegqiSingleInvReconstructLimit__option_t,
    std::string option,
    std::string value)
{
  d_holder->cegqiSingleInvReconstructLimit =
    runHandlerAndPredicates(options::cegqiSingleInvReconstructLimit, option, value, d_handler);
  d_holder->cegqiSingleInvReconstructLimit__setByUser__ = true;
  Trace("options") << "user assigned option cegqiSingleInvReconstructLimit" << std::endl;
  
}
template <> options::cegqiSingleInvReconstruct__option_t::type runHandlerAndPredicates(
    options::cegqiSingleInvReconstruct__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::cegqiSingleInvReconstruct__option_t::type retval = stringToCegqiSingleInvRconsMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::cegqiSingleInvReconstruct__option_t,
    std::string option,
    std::string value)
{
  d_holder->cegqiSingleInvReconstruct =
    runHandlerAndPredicates(options::cegqiSingleInvReconstruct, option, value, d_handler);
  d_holder->cegqiSingleInvReconstruct__setByUser__ = true;
  Trace("options") << "user assigned option cegqiSingleInvReconstruct" << std::endl;
  
}
template <> void Options::assignBool(
    options::cegqiSingleInvReconstructConst__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cegqiSingleInvReconstructConst, option, value, d_handler);
  d_holder->cegqiSingleInvReconstructConst = value;
  d_holder->cegqiSingleInvReconstructConst__setByUser__ = true;
  Trace("options") << "user assigned option cegqiSingleInvReconstructConst" << std::endl;
  
}
template <> options::cegqiSingleInvMode__option_t::type runHandlerAndPredicates(
    options::cegqiSingleInvMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::cegqiSingleInvMode__option_t::type retval = stringToCegqiSingleInvMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::cegqiSingleInvMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->cegqiSingleInvMode =
    runHandlerAndPredicates(options::cegqiSingleInvMode, option, value, d_handler);
  d_holder->cegqiSingleInvMode__setByUser__ = true;
  Trace("options") << "user assigned option cegqiSingleInvMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusStream__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusStream, option, value, d_handler);
  d_holder->sygusStream = value;
  d_holder->sygusStream__setByUser__ = true;
  Trace("options") << "user assigned option sygusStream" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusTemplEmbedGrammar__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusTemplEmbedGrammar, option, value, d_handler);
  d_holder->sygusTemplEmbedGrammar = value;
  d_holder->sygusTemplEmbedGrammar__setByUser__ = true;
  Trace("options") << "user assigned option sygusTemplEmbedGrammar" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusUnifCondIndNoRepeatSol__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusUnifCondIndNoRepeatSol, option, value, d_handler);
  d_holder->sygusUnifCondIndNoRepeatSol = value;
  d_holder->sygusUnifCondIndNoRepeatSol__setByUser__ = true;
  Trace("options") << "user assigned option sygusUnifCondIndNoRepeatSol" << std::endl;
  
}
template <> options::sygusUnifPi__option_t::type runHandlerAndPredicates(
    options::sygusUnifPi__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusUnifPi__option_t::type retval = stringToSygusUnifPiMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusUnifPi__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusUnifPi =
    runHandlerAndPredicates(options::sygusUnifPi, option, value, d_handler);
  d_holder->sygusUnifPi__setByUser__ = true;
  Trace("options") << "user assigned option sygusUnifPi" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusUnifShuffleCond__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusUnifShuffleCond, option, value, d_handler);
  d_holder->sygusUnifShuffleCond = value;
  d_holder->sygusUnifShuffleCond__setByUser__ = true;
  Trace("options") << "user assigned option sygusUnifShuffleCond" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusVerifySubcall__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusVerifySubcall, option, value, d_handler);
  d_holder->sygusVerifySubcall = value;
  d_holder->sygusVerifySubcall__setByUser__ = true;
  Trace("options") << "user assigned option sygusVerifySubcall" << std::endl;
  
}
template <> options::termDbMode__option_t::type runHandlerAndPredicates(
    options::termDbMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::termDbMode__option_t::type retval = stringToTermDbMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::termDbMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->termDbMode =
    runHandlerAndPredicates(options::termDbMode, option, value, d_handler);
  d_holder->termDbMode__setByUser__ = true;
  Trace("options") << "user assigned option termDbMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::trackInstLemmas__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::trackInstLemmas, option, value, d_handler);
  d_holder->trackInstLemmas = value;
  d_holder->trackInstLemmas__setByUser__ = true;
  Trace("options") << "user assigned option trackInstLemmas" << std::endl;
  
}
template <> options::triggerActiveSelMode__option_t::type runHandlerAndPredicates(
    options::triggerActiveSelMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::triggerActiveSelMode__option_t::type retval = stringToTriggerActiveSelMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::triggerActiveSelMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->triggerActiveSelMode =
    runHandlerAndPredicates(options::triggerActiveSelMode, option, value, d_handler);
  d_holder->triggerActiveSelMode__setByUser__ = true;
  Trace("options") << "user assigned option triggerActiveSelMode" << std::endl;
  
}
template <> options::triggerSelMode__option_t::type runHandlerAndPredicates(
    options::triggerSelMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::triggerSelMode__option_t::type retval = stringToTriggerSelMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::triggerSelMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->triggerSelMode =
    runHandlerAndPredicates(options::triggerSelMode, option, value, d_handler);
  d_holder->triggerSelMode__setByUser__ = true;
  Trace("options") << "user assigned option triggerSelMode" << std::endl;
  
}
template <> options::userPatternsQuant__option_t::type runHandlerAndPredicates(
    options::userPatternsQuant__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::userPatternsQuant__option_t::type retval = stringToUserPatMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::userPatternsQuant__option_t,
    std::string option,
    std::string value)
{
  d_holder->userPatternsQuant =
    runHandlerAndPredicates(options::userPatternsQuant, option, value, d_handler);
  d_holder->userPatternsQuant__setByUser__ = true;
  Trace("options") << "user assigned option userPatternsQuant" << std::endl;
  
}
template <> void Options::assignBool(
    options::varElimQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::varElimQuant, option, value, d_handler);
  d_holder->varElimQuant = value;
  d_holder->varElimQuant__setByUser__ = true;
  Trace("options") << "user assigned option varElimQuant" << std::endl;
  
}
template <> void Options::assignBool(
    options::varIneqElimQuant__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::varIneqElimQuant, option, value, d_handler);
  d_holder->varIneqElimQuant = value;
  d_holder->varIneqElimQuant__setByUser__ = true;
  Trace("options") << "user assigned option varIneqElimQuant" << std::endl;
  
}
template <> void Options::assignBool(
    options::sepCheckNeg__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sepCheckNeg, option, value, d_handler);
  d_holder->sepCheckNeg = value;
  d_holder->sepCheckNeg__setByUser__ = true;
  Trace("options") << "user assigned option sepCheckNeg" << std::endl;
  
}
template <> void Options::assignBool(
    options::sepChildRefine__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sepChildRefine, option, value, d_handler);
  d_holder->sepChildRefine = value;
  d_holder->sepChildRefine__setByUser__ = true;
  Trace("options") << "user assigned option sepChildRefine" << std::endl;
  
}
template <> void Options::assignBool(
    options::sepDisequalC__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sepDisequalC, option, value, d_handler);
  d_holder->sepDisequalC = value;
  d_holder->sepDisequalC__setByUser__ = true;
  Trace("options") << "user assigned option sepDisequalC" << std::endl;
  
}
template <> void Options::assignBool(
    options::sepExp__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sepExp, option, value, d_handler);
  d_holder->sepExp = value;
  d_holder->sepExp__setByUser__ = true;
  Trace("options") << "user assigned option sepExp" << std::endl;
  
}
template <> void Options::assignBool(
    options::sepMinimalRefine__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sepMinimalRefine, option, value, d_handler);
  d_holder->sepMinimalRefine = value;
  d_holder->sepMinimalRefine__setByUser__ = true;
  Trace("options") << "user assigned option sepMinimalRefine" << std::endl;
  
}
template <> void Options::assignBool(
    options::sepPreSkolemEmp__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sepPreSkolemEmp, option, value, d_handler);
  d_holder->sepPreSkolemEmp = value;
  d_holder->sepPreSkolemEmp__setByUser__ = true;
  Trace("options") << "user assigned option sepPreSkolemEmp" << std::endl;
  
}
template <> void Options::assignBool(
    options::setsExt__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::setsExt, option, value, d_handler);
  d_holder->setsExt = value;
  d_holder->setsExt__setByUser__ = true;
  Trace("options") << "user assigned option setsExt" << std::endl;
  
}
template <> void Options::assignBool(
    options::setsInferAsLemmas__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::setsInferAsLemmas, option, value, d_handler);
  d_holder->setsInferAsLemmas = value;
  d_holder->setsInferAsLemmas__setByUser__ = true;
  Trace("options") << "user assigned option setsInferAsLemmas" << std::endl;
  
}
template <> void Options::assignBool(
    options::setsProxyLemmas__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::setsProxyLemmas, option, value, d_handler);
  d_holder->setsProxyLemmas = value;
  d_holder->setsProxyLemmas__setByUser__ = true;
  Trace("options") << "user assigned option setsProxyLemmas" << std::endl;
  
}
template <> void Options::assignBool(
    options::abstractValues__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::abstractValues, option, value, d_handler);
  d_holder->abstractValues = value;
  d_holder->abstractValues__setByUser__ = true;
  Trace("options") << "user assigned option abstractValues" << std::endl;
  
}
template <> void Options::assignBool(
    options::ackermann__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::ackermann, option, value, d_handler);
  d_holder->ackermann = value;
  d_holder->ackermann__setByUser__ = true;
  Trace("options") << "user assigned option ackermann" << std::endl;
  
}
template <> options::bitblastStep__option_t::type runHandlerAndPredicates(
    options::bitblastStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::bitblastStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::bitblastStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->bitblastStep =
    runHandlerAndPredicates(options::bitblastStep, option, value, d_handler);
  d_holder->bitblastStep__setByUser__ = true;
  Trace("options") << "user assigned option bitblastStep" << std::endl;
  
}
template <> options::blockModelsMode__option_t::type runHandlerAndPredicates(
    options::blockModelsMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::blockModelsMode__option_t::type retval = stringToBlockModelsMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::blockModelsMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->blockModelsMode =
    runHandlerAndPredicates(options::blockModelsMode, option, value, d_handler);
  d_holder->blockModelsMode__setByUser__ = true;
  Trace("options") << "user assigned option blockModelsMode" << std::endl;
  
}
template <> options::bvEagerAssertStep__option_t::type runHandlerAndPredicates(
    options::bvEagerAssertStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::bvEagerAssertStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::bvEagerAssertStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->bvEagerAssertStep =
    runHandlerAndPredicates(options::bvEagerAssertStep, option, value, d_handler);
  d_holder->bvEagerAssertStep__setByUser__ = true;
  Trace("options") << "user assigned option bvEagerAssertStep" << std::endl;
  
}
template <> options::bvPropagationStep__option_t::type runHandlerAndPredicates(
    options::bvPropagationStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::bvPropagationStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::bvPropagationStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->bvPropagationStep =
    runHandlerAndPredicates(options::bvPropagationStep, option, value, d_handler);
  d_holder->bvPropagationStep__setByUser__ = true;
  Trace("options") << "user assigned option bvPropagationStep" << std::endl;
  
}
template <> options::bvSatConflictStep__option_t::type runHandlerAndPredicates(
    options::bvSatConflictStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::bvSatConflictStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::bvSatConflictStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->bvSatConflictStep =
    runHandlerAndPredicates(options::bvSatConflictStep, option, value, d_handler);
  d_holder->bvSatConflictStep__setByUser__ = true;
  Trace("options") << "user assigned option bvSatConflictStep" << std::endl;
  
}
template <> void Options::assignBool(
    options::checkAbducts__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::checkAbducts, option, value, d_handler);
  d_holder->checkAbducts = value;
  d_holder->checkAbducts__setByUser__ = true;
  Trace("options") << "user assigned option checkAbducts" << std::endl;
  
}
template <> void Options::assignBool(
    options::checkModels__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::checkModels, option, value, d_handler);
  d_holder->checkModels = value;
  d_holder->checkModels__setByUser__ = true;
  Trace("options") << "user assigned option checkModels" << std::endl;
  d_handler->notifyBeforeSearch(option);
}
template <> void runBoolPredicates(
    options::checkProofs__option_t,
    std::string option,
    bool b,
    options::OptionsHandler* handler)
{
  handler->LFSCEnabledBuild(option, b);
}
template <> void Options::assignBool(
    options::checkProofs__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::checkProofs, option, value, d_handler);
  d_holder->checkProofs = value;
  d_holder->checkProofs__setByUser__ = true;
  Trace("options") << "user assigned option checkProofs" << std::endl;
  d_handler->notifyBeforeSearch(option);
}
template <> void Options::assignBool(
    options::checkSynthSol__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::checkSynthSol, option, value, d_handler);
  d_holder->checkSynthSol = value;
  d_holder->checkSynthSol__setByUser__ = true;
  Trace("options") << "user assigned option checkSynthSol" << std::endl;
  
}
template <> void Options::assignBool(
    options::checkUnsatCores__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::checkUnsatCores, option, value, d_handler);
  d_holder->checkUnsatCores = value;
  d_holder->checkUnsatCores__setByUser__ = true;
  Trace("options") << "user assigned option checkUnsatCores" << std::endl;
  
}
template <> options::cnfStep__option_t::type runHandlerAndPredicates(
    options::cnfStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::cnfStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::cnfStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->cnfStep =
    runHandlerAndPredicates(options::cnfStep, option, value, d_handler);
  d_holder->cnfStep__setByUser__ = true;
  Trace("options") << "user assigned option cnfStep" << std::endl;
  
}
template <> void Options::assignBool(
    options::cpuTime__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::cpuTime, option, value, d_handler);
  d_holder->cpuTime = value;
  d_holder->cpuTime__setByUser__ = true;
  Trace("options") << "user assigned option cpuTime" << std::endl;
  
}
template <> void Options::assignBool(
    options::debugCheckModels__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::debugCheckModels, option, value, d_handler);
  d_holder->debugCheckModels = value;
  d_holder->debugCheckModels__setByUser__ = true;
  Trace("options") << "user assigned option debugCheckModels" << std::endl;
  d_handler->notifyBeforeSearch(option);
}
template <> options::decisionStep__option_t::type runHandlerAndPredicates(
    options::decisionStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::decisionStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::decisionStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->decisionStep =
    runHandlerAndPredicates(options::decisionStep, option, value, d_handler);
  d_holder->decisionStep__setByUser__ = true;
  Trace("options") << "user assigned option decisionStep" << std::endl;
  
}
template <> void Options::assign(
    options::diagnosticChannelName__option_t,
    std::string option,
    std::string value)
{
  d_holder->diagnosticChannelName =
    runHandlerAndPredicates(options::diagnosticChannelName, option, value, d_handler);
  d_holder->diagnosticChannelName__setByUser__ = true;
  Trace("options") << "user assigned option diagnosticChannelName" << std::endl;
  d_handler->notifySetDiagnosticOutputChannel(option);
}
template <> void Options::assignBool(
    options::dumpInstantiations__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dumpInstantiations, option, value, d_handler);
  d_holder->dumpInstantiations = value;
  d_holder->dumpInstantiations__setByUser__ = true;
  Trace("options") << "user assigned option dumpInstantiations" << std::endl;
  
}
template <> void Options::assignBool(
    options::dumpModels__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dumpModels, option, value, d_handler);
  d_holder->dumpModels = value;
  d_holder->dumpModels__setByUser__ = true;
  Trace("options") << "user assigned option dumpModels" << std::endl;
  
}
template <> void Options::assignBool(
    options::dumpProofs__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dumpProofs, option, value, d_handler);
  d_holder->dumpProofs = value;
  d_holder->dumpProofs__setByUser__ = true;
  Trace("options") << "user assigned option dumpProofs" << std::endl;
  
}
template <> void Options::assignBool(
    options::dumpSynth__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dumpSynth, option, value, d_handler);
  d_holder->dumpSynth = value;
  d_holder->dumpSynth__setByUser__ = true;
  Trace("options") << "user assigned option dumpSynth" << std::endl;
  
}
template <> options::dumpToFileName__option_t::type runHandlerAndPredicates(
    options::dumpToFileName__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::dumpToFileName__option_t::type retval = handleOption<std::string>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::dumpToFileName__option_t,
    std::string option,
    std::string value)
{
  d_holder->dumpToFileName =
    runHandlerAndPredicates(options::dumpToFileName, option, value, d_handler);
  d_holder->dumpToFileName__setByUser__ = true;
  Trace("options") << "user assigned option dumpToFileName" << std::endl;
  d_handler->notifyDumpToFile(option);
}
template <> void Options::assignBool(
    options::dumpUnsatCores__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dumpUnsatCores, option, value, d_handler);
  d_holder->dumpUnsatCores = value;
  d_holder->dumpUnsatCores__setByUser__ = true;
  Trace("options") << "user assigned option dumpUnsatCores" << std::endl;
  d_handler->notifyBeforeSearch(option);
}
template <> void Options::assignBool(
    options::dumpUnsatCoresFull__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::dumpUnsatCoresFull, option, value, d_handler);
  d_holder->dumpUnsatCoresFull = value;
  d_holder->dumpUnsatCoresFull__setByUser__ = true;
  Trace("options") << "user assigned option dumpUnsatCoresFull" << std::endl;
  d_handler->notifyBeforeSearch(option);
}
template <> options::dumpModeString__option_t::type runHandlerAndPredicates(
    options::dumpModeString__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::dumpModeString__option_t::type retval = handleOption<std::string>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::dumpModeString__option_t,
    std::string option,
    std::string value)
{
  d_holder->dumpModeString =
    runHandlerAndPredicates(options::dumpModeString, option, value, d_handler);
  d_holder->dumpModeString__setByUser__ = true;
  Trace("options") << "user assigned option dumpModeString" << std::endl;
  d_handler->notifyDumpMode(option);
}
template <> void Options::assignBool(
    options::expandDefinitions__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::expandDefinitions, option, value, d_handler);
  d_holder->expandDefinitions = value;
  d_holder->expandDefinitions__setByUser__ = true;
  Trace("options") << "user assigned option expandDefinitions" << std::endl;
  
}
template <> void Options::assignBool(
    options::extRewPrep__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::extRewPrep, option, value, d_handler);
  d_holder->extRewPrep = value;
  d_holder->extRewPrep__setByUser__ = true;
  Trace("options") << "user assigned option extRewPrep" << std::endl;
  
}
template <> void Options::assignBool(
    options::extRewPrepAgg__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::extRewPrepAgg, option, value, d_handler);
  d_holder->extRewPrepAgg = value;
  d_holder->extRewPrepAgg__setByUser__ = true;
  Trace("options") << "user assigned option extRewPrepAgg" << std::endl;
  
}
template <> void Options::assignBool(
    options::forceNoLimitCpuWhileDump__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::forceNoLimitCpuWhileDump, option, value, d_handler);
  d_holder->forceNoLimitCpuWhileDump = value;
  d_holder->forceNoLimitCpuWhileDump__setByUser__ = true;
  Trace("options") << "user assigned option forceNoLimitCpuWhileDump" << std::endl;
  
}
template <> void Options::assignBool(
    options::hardLimit__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::hardLimit, option, value, d_handler);
  d_holder->hardLimit = value;
  d_holder->hardLimit__setByUser__ = true;
  Trace("options") << "user assigned option hardLimit" << std::endl;
  
}
template <> void Options::assignBool(
    options::incrementalSolving__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::incrementalSolving, option, value, d_handler);
  d_holder->incrementalSolving = value;
  d_holder->incrementalSolving__setByUser__ = true;
  Trace("options") << "user assigned option incrementalSolving" << std::endl;
  
}
template <> void runBoolPredicates(
    options::interactiveMode__option_t,
    std::string option,
    bool b,
    options::OptionsHandler* handler)
{
  handler->setProduceAssertions(option, b);
}
template <> void Options::assignBool(
    options::interactiveMode__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::interactiveMode, option, value, d_handler);
  d_holder->interactiveMode = value;
  d_holder->interactiveMode__setByUser__ = true;
  Trace("options") << "user assigned option interactiveMode" << std::endl;
  d_handler->notifyBeforeSearch(option);
}
template <> void Options::assignBool(
    options::doITESimp__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::doITESimp, option, value, d_handler);
  d_holder->doITESimp = value;
  d_holder->doITESimp__setByUser__ = true;
  Trace("options") << "user assigned option doITESimp" << std::endl;
  
}
template <> options::lemmaStep__option_t::type runHandlerAndPredicates(
    options::lemmaStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::lemmaStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::lemmaStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->lemmaStep =
    runHandlerAndPredicates(options::lemmaStep, option, value, d_handler);
  d_holder->lemmaStep__setByUser__ = true;
  Trace("options") << "user assigned option lemmaStep" << std::endl;
  
}
template <> options::modelCoresMode__option_t::type runHandlerAndPredicates(
    options::modelCoresMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::modelCoresMode__option_t::type retval = stringToModelCoresMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::modelCoresMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->modelCoresMode =
    runHandlerAndPredicates(options::modelCoresMode, option, value, d_handler);
  d_holder->modelCoresMode__setByUser__ = true;
  Trace("options") << "user assigned option modelCoresMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::modelUninterpDtEnum__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::modelUninterpDtEnum, option, value, d_handler);
  d_holder->modelUninterpDtEnum = value;
  d_holder->modelUninterpDtEnum__setByUser__ = true;
  Trace("options") << "user assigned option modelUninterpDtEnum" << std::endl;
  
}
template <> void Options::assignBool(
    options::modelWitnessValue__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::modelWitnessValue, option, value, d_handler);
  d_holder->modelWitnessValue = value;
  d_holder->modelWitnessValue__setByUser__ = true;
  Trace("options") << "user assigned option modelWitnessValue" << std::endl;
  
}
template <> void Options::assignBool(
    options::doITESimpOnRepeat__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::doITESimpOnRepeat, option, value, d_handler);
  d_holder->doITESimpOnRepeat = value;
  d_holder->doITESimpOnRepeat__setByUser__ = true;
  Trace("options") << "user assigned option doITESimpOnRepeat" << std::endl;
  
}
template <> options::parseStep__option_t::type runHandlerAndPredicates(
    options::parseStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::parseStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::parseStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->parseStep =
    runHandlerAndPredicates(options::parseStep, option, value, d_handler);
  d_holder->parseStep__setByUser__ = true;
  Trace("options") << "user assigned option parseStep" << std::endl;
  
}
template <> options::preprocessStep__option_t::type runHandlerAndPredicates(
    options::preprocessStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::preprocessStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::preprocessStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->preprocessStep =
    runHandlerAndPredicates(options::preprocessStep, option, value, d_handler);
  d_holder->preprocessStep__setByUser__ = true;
  Trace("options") << "user assigned option preprocessStep" << std::endl;
  
}
template <> void Options::assignBool(
    options::produceAbducts__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::produceAbducts, option, value, d_handler);
  d_holder->produceAbducts = value;
  d_holder->produceAbducts__setByUser__ = true;
  Trace("options") << "user assigned option produceAbducts" << std::endl;
  
}
template <> void runBoolPredicates(
    options::produceAssertions__option_t,
    std::string option,
    bool b,
    options::OptionsHandler* handler)
{
  handler->setProduceAssertions(option, b);
}
template <> void Options::assignBool(
    options::produceAssertions__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::produceAssertions, option, value, d_handler);
  d_holder->produceAssertions = value;
  d_holder->produceAssertions__setByUser__ = true;
  Trace("options") << "user assigned option produceAssertions" << std::endl;
  d_handler->notifyBeforeSearch(option);
}
template <> void Options::assignBool(
    options::produceAssignments__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::produceAssignments, option, value, d_handler);
  d_holder->produceAssignments = value;
  d_holder->produceAssignments__setByUser__ = true;
  Trace("options") << "user assigned option produceAssignments" << std::endl;
  d_handler->notifyBeforeSearch(option);
}
template <> void Options::assignBool(
    options::produceModels__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::produceModels, option, value, d_handler);
  d_holder->produceModels = value;
  d_holder->produceModels__setByUser__ = true;
  Trace("options") << "user assigned option produceModels" << std::endl;
  d_handler->notifyBeforeSearch(option);
}
template <> void runBoolPredicates(
    options::unsatAssumptions__option_t,
    std::string option,
    bool b,
    options::OptionsHandler* handler)
{
  handler->proofEnabledBuild(option, b);
}
template <> void Options::assignBool(
    options::unsatAssumptions__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::unsatAssumptions, option, value, d_handler);
  d_holder->unsatAssumptions = value;
  d_holder->unsatAssumptions__setByUser__ = true;
  Trace("options") << "user assigned option unsatAssumptions" << std::endl;
  d_handler->notifyBeforeSearch(option);
}
template <> void runBoolPredicates(
    options::unsatCores__option_t,
    std::string option,
    bool b,
    options::OptionsHandler* handler)
{
  handler->proofEnabledBuild(option, b);
}
template <> void Options::assignBool(
    options::unsatCores__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::unsatCores, option, value, d_handler);
  d_holder->unsatCores = value;
  d_holder->unsatCores__setByUser__ = true;
  Trace("options") << "user assigned option unsatCores" << std::endl;
  d_handler->notifyBeforeSearch(option);
}
template <> void runBoolPredicates(
    options::proof__option_t,
    std::string option,
    bool b,
    options::OptionsHandler* handler)
{
  handler->proofEnabledBuild(option, b);
}
template <> void Options::assignBool(
    options::proof__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::proof, option, value, d_handler);
  d_holder->proof = value;
  d_holder->proof__setByUser__ = true;
  Trace("options") << "user assigned option proof" << std::endl;
  d_handler->notifyBeforeSearch(option);
}
template <> options::quantifierStep__option_t::type runHandlerAndPredicates(
    options::quantifierStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::quantifierStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::quantifierStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->quantifierStep =
    runHandlerAndPredicates(options::quantifierStep, option, value, d_handler);
  d_holder->quantifierStep__setByUser__ = true;
  Trace("options") << "user assigned option quantifierStep" << std::endl;
  
}
template <> void Options::assign(
    options::regularChannelName__option_t,
    std::string option,
    std::string value)
{
  d_holder->regularChannelName =
    runHandlerAndPredicates(options::regularChannelName, option, value, d_handler);
  d_holder->regularChannelName__setByUser__ = true;
  Trace("options") << "user assigned option regularChannelName" << std::endl;
  d_handler->notifySetRegularOutputChannel(option);
}
template <> void Options::assignBool(
    options::repeatSimp__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::repeatSimp, option, value, d_handler);
  d_holder->repeatSimp = value;
  d_holder->repeatSimp__setByUser__ = true;
  Trace("options") << "user assigned option repeatSimp" << std::endl;
  
}
template <> options::restartStep__option_t::type runHandlerAndPredicates(
    options::restartStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::restartStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::restartStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->restartStep =
    runHandlerAndPredicates(options::restartStep, option, value, d_handler);
  d_holder->restartStep__setByUser__ = true;
  Trace("options") << "user assigned option restartStep" << std::endl;
  
}
template <> options::rewriteStep__option_t::type runHandlerAndPredicates(
    options::rewriteStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::rewriteStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::rewriteStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->rewriteStep =
    runHandlerAndPredicates(options::rewriteStep, option, value, d_handler);
  d_holder->rewriteStep__setByUser__ = true;
  Trace("options") << "user assigned option rewriteStep" << std::endl;
  
}
template <> options::perCallResourceLimit__option_t::type runHandlerAndPredicates(
    options::perCallResourceLimit__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::perCallResourceLimit__option_t::type retval = handler->limitHandler(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::perCallResourceLimit__option_t,
    std::string option,
    std::string value)
{
  d_holder->perCallResourceLimit =
    runHandlerAndPredicates(options::perCallResourceLimit, option, value, d_handler);
  d_holder->perCallResourceLimit__setByUser__ = true;
  Trace("options") << "user assigned option perCallResourceLimit" << std::endl;
  d_handler->notifyRlimitPer(option);
}
template <> options::cumulativeResourceLimit__option_t::type runHandlerAndPredicates(
    options::cumulativeResourceLimit__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::cumulativeResourceLimit__option_t::type retval = handler->limitHandler(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::cumulativeResourceLimit__option_t,
    std::string option,
    std::string value)
{
  d_holder->cumulativeResourceLimit =
    runHandlerAndPredicates(options::cumulativeResourceLimit, option, value, d_handler);
  d_holder->cumulativeResourceLimit__setByUser__ = true;
  Trace("options") << "user assigned option cumulativeResourceLimit" << std::endl;
  d_handler->notifyRlimit(option);
}
template <> options::satConflictStep__option_t::type runHandlerAndPredicates(
    options::satConflictStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::satConflictStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::satConflictStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->satConflictStep =
    runHandlerAndPredicates(options::satConflictStep, option, value, d_handler);
  d_holder->satConflictStep__setByUser__ = true;
  Trace("options") << "user assigned option satConflictStep" << std::endl;
  
}
template <> void Options::assignBool(
    options::compressItes__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::compressItes, option, value, d_handler);
  d_holder->compressItes = value;
  d_holder->compressItes__setByUser__ = true;
  Trace("options") << "user assigned option compressItes" << std::endl;
  
}
template <> options::zombieHuntThreshold__option_t::type runHandlerAndPredicates(
    options::zombieHuntThreshold__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::zombieHuntThreshold__option_t::type retval = handleOption<uint32_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::zombieHuntThreshold__option_t,
    std::string option,
    std::string value)
{
  d_holder->zombieHuntThreshold =
    runHandlerAndPredicates(options::zombieHuntThreshold, option, value, d_handler);
  d_holder->zombieHuntThreshold__setByUser__ = true;
  Trace("options") << "user assigned option zombieHuntThreshold" << std::endl;
  
}
template <> void Options::assignBool(
    options::simplifyWithCareEnabled__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::simplifyWithCareEnabled, option, value, d_handler);
  d_holder->simplifyWithCareEnabled = value;
  d_holder->simplifyWithCareEnabled__setByUser__ = true;
  Trace("options") << "user assigned option simplifyWithCareEnabled" << std::endl;
  
}
template <> options::simplificationMode__option_t::type runHandlerAndPredicates(
    options::simplificationMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::simplificationMode__option_t::type retval = stringToSimplificationMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::simplificationMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->simplificationMode =
    runHandlerAndPredicates(options::simplificationMode, option, value, d_handler);
  d_holder->simplificationMode__setByUser__ = true;
  Trace("options") << "user assigned option simplificationMode" << std::endl;
  
}
template <> options::solveBVAsInt__option_t::type runHandlerAndPredicates(
    options::solveBVAsInt__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::solveBVAsInt__option_t::type retval = handleOption<uint32_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::solveBVAsInt__option_t,
    std::string option,
    std::string value)
{
  d_holder->solveBVAsInt =
    runHandlerAndPredicates(options::solveBVAsInt, option, value, d_handler);
  d_holder->solveBVAsInt__setByUser__ = true;
  Trace("options") << "user assigned option solveBVAsInt" << std::endl;
  
}
template <> options::solveIntAsBV__option_t::type runHandlerAndPredicates(
    options::solveIntAsBV__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::solveIntAsBV__option_t::type retval = handleOption<uint32_t>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::solveIntAsBV__option_t,
    std::string option,
    std::string value)
{
  d_holder->solveIntAsBV =
    runHandlerAndPredicates(options::solveIntAsBV, option, value, d_handler);
  d_holder->solveIntAsBV__setByUser__ = true;
  Trace("options") << "user assigned option solveIntAsBV" << std::endl;
  
}
template <> void Options::assignBool(
    options::solveRealAsInt__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::solveRealAsInt, option, value, d_handler);
  d_holder->solveRealAsInt = value;
  d_holder->solveRealAsInt__setByUser__ = true;
  Trace("options") << "user assigned option solveRealAsInt" << std::endl;
  
}
template <> void Options::assignBool(
    options::sortInference__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sortInference, option, value, d_handler);
  d_holder->sortInference = value;
  d_holder->sortInference__setByUser__ = true;
  Trace("options") << "user assigned option sortInference" << std::endl;
  
}
template <> void Options::assignBool(
    options::doStaticLearning__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::doStaticLearning, option, value, d_handler);
  d_holder->doStaticLearning = value;
  d_holder->doStaticLearning__setByUser__ = true;
  Trace("options") << "user assigned option doStaticLearning" << std::endl;
  
}
template <> options::sygusOut__option_t::type runHandlerAndPredicates(
    options::sygusOut__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::sygusOut__option_t::type retval = stringToSygusSolutionOutMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::sygusOut__option_t,
    std::string option,
    std::string value)
{
  d_holder->sygusOut =
    runHandlerAndPredicates(options::sygusOut, option, value, d_handler);
  d_holder->sygusOut__setByUser__ = true;
  Trace("options") << "user assigned option sygusOut" << std::endl;
  
}
template <> void Options::assignBool(
    options::sygusPrintCallbacks__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::sygusPrintCallbacks, option, value, d_handler);
  d_holder->sygusPrintCallbacks = value;
  d_holder->sygusPrintCallbacks__setByUser__ = true;
  Trace("options") << "user assigned option sygusPrintCallbacks" << std::endl;
  
}
template <> options::theoryCheckStep__option_t::type runHandlerAndPredicates(
    options::theoryCheckStep__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::theoryCheckStep__option_t::type retval = handleOption<unsigned>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::theoryCheckStep__option_t,
    std::string option,
    std::string value)
{
  d_holder->theoryCheckStep =
    runHandlerAndPredicates(options::theoryCheckStep, option, value, d_handler);
  d_holder->theoryCheckStep__setByUser__ = true;
  Trace("options") << "user assigned option theoryCheckStep" << std::endl;
  
}
template <> options::perCallMillisecondLimit__option_t::type runHandlerAndPredicates(
    options::perCallMillisecondLimit__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::perCallMillisecondLimit__option_t::type retval = handler->limitHandler(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::perCallMillisecondLimit__option_t,
    std::string option,
    std::string value)
{
  d_holder->perCallMillisecondLimit =
    runHandlerAndPredicates(options::perCallMillisecondLimit, option, value, d_handler);
  d_holder->perCallMillisecondLimit__setByUser__ = true;
  Trace("options") << "user assigned option perCallMillisecondLimit" << std::endl;
  d_handler->notifyTlimitPer(option);
}
template <> options::cumulativeMillisecondLimit__option_t::type runHandlerAndPredicates(
    options::cumulativeMillisecondLimit__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::cumulativeMillisecondLimit__option_t::type retval = handler->limitHandler(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::cumulativeMillisecondLimit__option_t,
    std::string option,
    std::string value)
{
  d_holder->cumulativeMillisecondLimit =
    runHandlerAndPredicates(options::cumulativeMillisecondLimit, option, value, d_handler);
  d_holder->cumulativeMillisecondLimit__setByUser__ = true;
  Trace("options") << "user assigned option cumulativeMillisecondLimit" << std::endl;
  d_handler->notifyTlimit(option);
}
template <> void Options::assignBool(
    options::unconstrainedSimp__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::unconstrainedSimp, option, value, d_handler);
  d_holder->unconstrainedSimp = value;
  d_holder->unconstrainedSimp__setByUser__ = true;
  Trace("options") << "user assigned option unconstrainedSimp" << std::endl;
  
}
template <> void Options::assignBool(
    options::regExpElim__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::regExpElim, option, value, d_handler);
  d_holder->regExpElim = value;
  d_holder->regExpElim__setByUser__ = true;
  Trace("options") << "user assigned option regExpElim" << std::endl;
  
}
template <> void Options::assignBool(
    options::regExpElimAgg__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::regExpElimAgg, option, value, d_handler);
  d_holder->regExpElimAgg = value;
  d_holder->regExpElimAgg__setByUser__ = true;
  Trace("options") << "user assigned option regExpElimAgg" << std::endl;
  
}
template <> options::stringRegExpInterMode__option_t::type runHandlerAndPredicates(
    options::stringRegExpInterMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::stringRegExpInterMode__option_t::type retval = stringToRegExpInterMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::stringRegExpInterMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->stringRegExpInterMode =
    runHandlerAndPredicates(options::stringRegExpInterMode, option, value, d_handler);
  d_holder->stringRegExpInterMode__setByUser__ = true;
  Trace("options") << "user assigned option stringRegExpInterMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringCheckEntailLen__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringCheckEntailLen, option, value, d_handler);
  d_holder->stringCheckEntailLen = value;
  d_holder->stringCheckEntailLen__setByUser__ = true;
  Trace("options") << "user assigned option stringCheckEntailLen" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringEager__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringEager, option, value, d_handler);
  d_holder->stringEager = value;
  d_holder->stringEager__setByUser__ = true;
  Trace("options") << "user assigned option stringEager" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringEagerLen__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringEagerLen, option, value, d_handler);
  d_holder->stringEagerLen = value;
  d_holder->stringEagerLen__setByUser__ = true;
  Trace("options") << "user assigned option stringEagerLen" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringExp__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringExp, option, value, d_handler);
  d_holder->stringExp = value;
  d_holder->stringExp__setByUser__ = true;
  Trace("options") << "user assigned option stringExp" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringFlatForms__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringFlatForms, option, value, d_handler);
  d_holder->stringFlatForms = value;
  d_holder->stringFlatForms__setByUser__ = true;
  Trace("options") << "user assigned option stringFlatForms" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringFMF__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringFMF, option, value, d_handler);
  d_holder->stringFMF = value;
  d_holder->stringFMF__setByUser__ = true;
  Trace("options") << "user assigned option stringFMF" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringGuessModel__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringGuessModel, option, value, d_handler);
  d_holder->stringGuessModel = value;
  d_holder->stringGuessModel__setByUser__ = true;
  Trace("options") << "user assigned option stringGuessModel" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringInferAsLemmas__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringInferAsLemmas, option, value, d_handler);
  d_holder->stringInferAsLemmas = value;
  d_holder->stringInferAsLemmas__setByUser__ = true;
  Trace("options") << "user assigned option stringInferAsLemmas" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringInferSym__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringInferSym, option, value, d_handler);
  d_holder->stringInferSym = value;
  d_holder->stringInferSym__setByUser__ = true;
  Trace("options") << "user assigned option stringInferSym" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringIgnNegMembership__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringIgnNegMembership, option, value, d_handler);
  d_holder->stringIgnNegMembership = value;
  d_holder->stringIgnNegMembership__setByUser__ = true;
  Trace("options") << "user assigned option stringIgnNegMembership" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringLazyPreproc__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringLazyPreproc, option, value, d_handler);
  d_holder->stringLazyPreproc = value;
  d_holder->stringLazyPreproc__setByUser__ = true;
  Trace("options") << "user assigned option stringLazyPreproc" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringLenNorm__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringLenNorm, option, value, d_handler);
  d_holder->stringLenNorm = value;
  d_holder->stringLenNorm__setByUser__ = true;
  Trace("options") << "user assigned option stringLenNorm" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringLenPropCsp__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringLenPropCsp, option, value, d_handler);
  d_holder->stringLenPropCsp = value;
  d_holder->stringLenPropCsp__setByUser__ = true;
  Trace("options") << "user assigned option stringLenPropCsp" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringMinPrefixExplain__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringMinPrefixExplain, option, value, d_handler);
  d_holder->stringMinPrefixExplain = value;
  d_holder->stringMinPrefixExplain__setByUser__ = true;
  Trace("options") << "user assigned option stringMinPrefixExplain" << std::endl;
  
}
template <> void Options::assignBool(
    options::stdPrintASCII__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stdPrintASCII, option, value, d_handler);
  d_holder->stdPrintASCII = value;
  d_holder->stdPrintASCII__setByUser__ = true;
  Trace("options") << "user assigned option stdPrintASCII" << std::endl;
  
}
template <> options::stringProcessLoopMode__option_t::type runHandlerAndPredicates(
    options::stringProcessLoopMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::stringProcessLoopMode__option_t::type retval = stringToProcessLoopMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::stringProcessLoopMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->stringProcessLoopMode =
    runHandlerAndPredicates(options::stringProcessLoopMode, option, value, d_handler);
  d_holder->stringProcessLoopMode__setByUser__ = true;
  Trace("options") << "user assigned option stringProcessLoopMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::stringRExplainLemmas__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::stringRExplainLemmas, option, value, d_handler);
  d_holder->stringRExplainLemmas = value;
  d_holder->stringRExplainLemmas__setByUser__ = true;
  Trace("options") << "user assigned option stringRExplainLemmas" << std::endl;
  
}
template <> void Options::assignBool(
    options::assignFunctionValues__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::assignFunctionValues, option, value, d_handler);
  d_holder->assignFunctionValues = value;
  d_holder->assignFunctionValues__setByUser__ = true;
  Trace("options") << "user assigned option assignFunctionValues" << std::endl;
  
}
template <> void Options::assignBool(
    options::condenseFunctionValues__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::condenseFunctionValues, option, value, d_handler);
  d_holder->condenseFunctionValues = value;
  d_holder->condenseFunctionValues__setByUser__ = true;
  Trace("options") << "user assigned option condenseFunctionValues" << std::endl;
  
}
template <> options::theoryOfMode__option_t::type runHandlerAndPredicates(
    options::theoryOfMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::theoryOfMode__option_t::type retval = stringToTheoryOfMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::theoryOfMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->theoryOfMode =
    runHandlerAndPredicates(options::theoryOfMode, option, value, d_handler);
  d_holder->theoryOfMode__setByUser__ = true;
  Trace("options") << "user assigned option theoryOfMode" << std::endl;
  
}
template <> void Options::assignBool(
    options::ufSymmetryBreaker__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::ufSymmetryBreaker, option, value, d_handler);
  d_holder->ufSymmetryBreaker = value;
  d_holder->ufSymmetryBreaker__setByUser__ = true;
  Trace("options") << "user assigned option ufSymmetryBreaker" << std::endl;
  
}
template <> void Options::assignBool(
    options::ufHo__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::ufHo, option, value, d_handler);
  d_holder->ufHo = value;
  d_holder->ufHo__setByUser__ = true;
  Trace("options") << "user assigned option ufHo" << std::endl;
  
}
template <> void Options::assignBool(
    options::ufHoExt__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::ufHoExt, option, value, d_handler);
  d_holder->ufHoExt = value;
  d_holder->ufHoExt__setByUser__ = true;
  Trace("options") << "user assigned option ufHoExt" << std::endl;
  
}
template <> options::ufssAbortCardinality__option_t::type runHandlerAndPredicates(
    options::ufssAbortCardinality__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::ufssAbortCardinality__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::ufssAbortCardinality__option_t,
    std::string option,
    std::string value)
{
  d_holder->ufssAbortCardinality =
    runHandlerAndPredicates(options::ufssAbortCardinality, option, value, d_handler);
  d_holder->ufssAbortCardinality__setByUser__ = true;
  Trace("options") << "user assigned option ufssAbortCardinality" << std::endl;
  
}
template <> void Options::assignBool(
    options::ufssFairness__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::ufssFairness, option, value, d_handler);
  d_holder->ufssFairness = value;
  d_holder->ufssFairness__setByUser__ = true;
  Trace("options") << "user assigned option ufssFairness" << std::endl;
  
}
template <> void Options::assignBool(
    options::ufssFairnessMonotone__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::ufssFairnessMonotone, option, value, d_handler);
  d_holder->ufssFairnessMonotone = value;
  d_holder->ufssFairnessMonotone__setByUser__ = true;
  Trace("options") << "user assigned option ufssFairnessMonotone" << std::endl;
  
}
template <> void Options::assignBool(
    options::ufssTotality__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::ufssTotality, option, value, d_handler);
  d_holder->ufssTotality = value;
  d_holder->ufssTotality__setByUser__ = true;
  Trace("options") << "user assigned option ufssTotality" << std::endl;
  
}
template <> options::ufssTotalityLimited__option_t::type runHandlerAndPredicates(
    options::ufssTotalityLimited__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::ufssTotalityLimited__option_t::type retval = handleOption<int>(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::ufssTotalityLimited__option_t,
    std::string option,
    std::string value)
{
  d_holder->ufssTotalityLimited =
    runHandlerAndPredicates(options::ufssTotalityLimited, option, value, d_handler);
  d_holder->ufssTotalityLimited__setByUser__ = true;
  Trace("options") << "user assigned option ufssTotalityLimited" << std::endl;
  
}
template <> void Options::assignBool(
    options::ufssTotalitySymBreak__option_t,
    std::string option,
    bool value)
{
  runBoolPredicates(options::ufssTotalitySymBreak, option, value, d_handler);
  d_holder->ufssTotalitySymBreak = value;
  d_holder->ufssTotalitySymBreak__setByUser__ = true;
  Trace("options") << "user assigned option ufssTotalitySymBreak" << std::endl;
  
}
template <> options::ufssMode__option_t::type runHandlerAndPredicates(
    options::ufssMode__option_t,
    std::string option,
    std::string optionarg,
    options::OptionsHandler* handler)
{
  options::ufssMode__option_t::type retval = stringToUfssMode(option, optionarg);
  
  return retval;
}
template <> void Options::assign(
    options::ufssMode__option_t,
    std::string option,
    std::string value)
{
  d_holder->ufssMode =
    runHandlerAndPredicates(options::ufssMode, option, value, d_handler);
  d_holder->ufssMode__setByUser__ = true;
  Trace("options") << "user assigned option ufssMode" << std::endl;
  
}


#ifdef CVC4_DEBUG
#  define USE_EARLY_TYPE_CHECKING_BY_DEFAULT true
#else /* CVC4_DEBUG */
#  define USE_EARLY_TYPE_CHECKING_BY_DEFAULT false
#endif /* CVC4_DEBUG */

#if defined(CVC4_MUZZLED) || defined(CVC4_COMPETITION_MODE)
#  define DO_SEMANTIC_CHECKS_BY_DEFAULT false
#else /* CVC4_MUZZLED || CVC4_COMPETITION_MODE */
#  define DO_SEMANTIC_CHECKS_BY_DEFAULT true
#endif /* CVC4_MUZZLED || CVC4_COMPETITION_MODE */

options::OptionsHolder::OptionsHolder() :
  maxApproxDepth(200),
  maxApproxDepth__setByUser__(false),
  brabTest(true),
  brabTest__setByUser__(false),
  arithNoPartialFun(false),
  arithNoPartialFun__setByUser__(false),
  arithPropAsLemmaLength(8),
  arithPropAsLemmaLength__setByUser__(false),
  arithPropagationMode(ArithPropagationMode::BOTH_PROP),
  arithPropagationMode__setByUser__(false),
  arithRewriteEq(false),
  arithRewriteEq__setByUser__(false),
  collectPivots(false),
  collectPivots__setByUser__(false),
  doCutAllBounded(false),
  doCutAllBounded__setByUser__(false),
  exportDioDecompositions(false),
  exportDioDecompositions__setByUser__(false),
  dioRepeat(false),
  dioRepeat__setByUser__(false),
  arithDioSolver(true),
  arithDioSolver__setByUser__(false),
  dioSolverTurns(10),
  dioSolverTurns__setByUser__(false),
  arithErrorSelectionRule(ErrorSelectionRule::MINIMUM_AMOUNT),
  arithErrorSelectionRule__setByUser__(false),
  havePenalties(false),
  havePenalties__setByUser__(false),
  arithHeuristicPivots(0),
  arithHeuristicPivots__setByUser__(false),
  replayFailureLemma(false),
  replayFailureLemma__setByUser__(false),
  maxCutsInContext(65535),
  maxCutsInContext__setByUser__(false),
  arithMLTrick(false),
  arithMLTrick__setByUser__(false),
  arithMLTrickSubstitutions(1),
  arithMLTrickSubstitutions__setByUser__(false),
  newProp(true),
  newProp__setByUser__(false),
  nlExt(true),
  nlExt__setByUser__(false),
  nlExtEntailConflicts(false),
  nlExtEntailConflicts__setByUser__(false),
  nlExtFactor(true),
  nlExtFactor__setByUser__(false),
  nlExtIncPrecision(true),
  nlExtIncPrecision__setByUser__(false),
  nlExtPurify(false),
  nlExtPurify__setByUser__(false),
  nlExtResBound(false),
  nlExtResBound__setByUser__(false),
  nlExtRewrites(true),
  nlExtRewrites__setByUser__(false),
  nlExtSplitZero(false),
  nlExtSplitZero__setByUser__(false),
  nlExtTfTaylorDegree(4),
  nlExtTfTaylorDegree__setByUser__(false),
  nlExtTfTangentPlanes(true),
  nlExtTfTangentPlanes__setByUser__(false),
  nlExtTangentPlanes(false),
  nlExtTangentPlanes__setByUser__(false),
  nlExtTangentPlanesInterleave(false),
  nlExtTangentPlanesInterleave__setByUser__(false),
  pbRewrites(false),
  pbRewrites__setByUser__(false),
  arithPivotThreshold(2),
  arithPivotThreshold__setByUser__(false),
  ppAssertMaxSubSize(2),
  ppAssertMaxSubSize__setByUser__(false),
  arithPropagateMaxLength(16),
  arithPropagateMaxLength__setByUser__(false),
  replayEarlyCloseDepths(1),
  replayEarlyCloseDepths__setByUser__(false),
  replayFailurePenalty(100),
  replayFailurePenalty__setByUser__(false),
  lemmaRejectCutSize(25500),
  lemmaRejectCutSize__setByUser__(false),
  replayNumericFailurePenalty(4194304),
  replayNumericFailurePenalty__setByUser__(false),
  replayRejectCutSize(25500),
  replayRejectCutSize__setByUser__(false),
  soiApproxMajorFailurePen(50),
  soiApproxMajorFailurePen__setByUser__(false),
  soiApproxMajorFailure(.01),
  soiApproxMajorFailure__setByUser__(false),
  soiApproxMinorFailurePen(10),
  soiApproxMinorFailurePen__setByUser__(false),
  soiApproxMinorFailure(.0001),
  soiApproxMinorFailure__setByUser__(false),
  restrictedPivots(true),
  restrictedPivots__setByUser__(false),
  revertArithModels(false),
  revertArithModels__setByUser__(false),
  rrTurns(3),
  rrTurns__setByUser__(false),
  trySolveIntStandardEffort(false),
  trySolveIntStandardEffort__setByUser__(false),
  arithSimplexCheckPeriod(200),
  arithSimplexCheckPeriod__setByUser__(false),
  soiQuickExplain(false),
  soiQuickExplain__setByUser__(false),
  arithStandardCheckVarOrderPivots(-1),
  arithStandardCheckVarOrderPivots__setByUser__(false),
  arithUnateLemmaMode(ArithUnateLemmaMode::ALL),
  arithUnateLemmaMode__setByUser__(false),
  useApprox(false),
  useApprox__setByUser__(false),
  useFC(false),
  useFC__setByUser__(false),
  useSOI(false),
  useSOI__setByUser__(false),
  arraysConfig(0),
  arraysConfig__setByUser__(false),
  arraysEagerIndexSplitting(true),
  arraysEagerIndexSplitting__setByUser__(false),
  arraysEagerLemmas(false),
  arraysEagerLemmas__setByUser__(false),
  arraysLazyRIntro1(true),
  arraysLazyRIntro1__setByUser__(false),
  arraysModelBased(false),
  arraysModelBased__setByUser__(false),
  arraysOptimizeLinear(true),
  arraysOptimizeLinear__setByUser__(false),
  arraysPropagate(2),
  arraysPropagate__setByUser__(false),
  arraysReduceSharing(false),
  arraysReduceSharing__setByUser__(false),
  arraysWeakEquivalence(false),
  arraysWeakEquivalence__setByUser__(false),
  binary_name(),
  binary_name__setByUser__(false),
  err(&std::cerr),
  err__setByUser__(false),
  in(&std::cin),
  in__setByUser__(false),
  inputLanguage(language::input::LANG_AUTO),
  inputLanguage__setByUser__(false),
  languageHelp(),
  languageHelp__setByUser__(false),
  out(&std::cout),
  out__setByUser__(false),
  outputLanguage(language::output::LANG_AUTO),
  outputLanguage__setByUser__(false),
  parseOnly(),
  parseOnly__setByUser__(false),
  preprocessOnly(),
  preprocessOnly__setByUser__(false),
  printSuccess(),
  printSuccess__setByUser__(false),
  statistics(),
  statistics__setByUser__(false),
  statsEveryQuery(false),
  statsEveryQuery__setByUser__(false),
  statsHideZeros(false),
  statsHideZeros__setByUser__(false),
  verbosity(0),
  verbosity__setByUser__(false),
  bitvectorAig(false),
  bitvectorAig__setByUser__(false),
  bitblastMode(BitblastMode::LAZY),
  bitblastMode__setByUser__(false),
  bitwiseEq(true),
  bitwiseEq__setByUser__(false),
  boolToBitvector(BoolToBVMode::OFF),
  boolToBitvector__setByUser__(false),
  bvAbstraction(false),
  bvAbstraction__setByUser__(false),
  bitvectorAigSimplifications(),
  bitvectorAigSimplifications__setByUser__(false),
  bvAlgExtf(true),
  bvAlgExtf__setByUser__(false),
  bitvectorAlgebraicBudget(1500),
  bitvectorAlgebraicBudget__setByUser__(false),
  bitvectorAlgebraicSolver(true),
  bitvectorAlgebraicSolver__setByUser__(false),
  bitvectorDivByZeroConst(false),
  bitvectorDivByZeroConst__setByUser__(false),
  bvEagerExplanations(false),
  bvEagerExplanations__setByUser__(false),
  bitvectorEqualitySlicer(BvSlicerMode::OFF),
  bitvectorEqualitySlicer__setByUser__(false),
  bitvectorEqualitySolver(true),
  bitvectorEqualitySolver__setByUser__(false),
  bvExtractArithRewrite(false),
  bvExtractArithRewrite__setByUser__(false),
  bvGaussElim(false),
  bvGaussElim__setByUser__(false),
  bitvectorInequalitySolver(true),
  bitvectorInequalitySolver__setByUser__(false),
  bvIntroducePow2(false),
  bvIntroducePow2__setByUser__(false),
  bvLazyReduceExtf(false),
  bvLazyReduceExtf__setByUser__(false),
  bvLazyRewriteExtf(true),
  bvLazyRewriteExtf__setByUser__(false),
  bvNumFunc(1),
  bvNumFunc__setByUser__(false),
  bvOptimizeSatProof(BvOptimizeSatProof::FORMULA),
  bvOptimizeSatProof__setByUser__(false),
  bvPrintConstsAsIndexedSymbols(false),
  bvPrintConstsAsIndexedSymbols__setByUser__(false),
  bvProofFormat(BvProofFormat::ER),
  bvProofFormat__setByUser__(false),
  bitvectorPropagate(true),
  bitvectorPropagate__setByUser__(false),
  bitvectorQuickXplain(false),
  bitvectorQuickXplain__setByUser__(false),
  bvSatSolver(SatSolverMode::MINISAT),
  bvSatSolver__setByUser__(false),
  skolemizeArguments(false),
  skolemizeArguments__setByUser__(false),
  bitvectorToBool(false),
  bitvectorToBool__setByUser__(false),
  cdtBisimilar(true),
  cdtBisimilar__setByUser__(false),
  dtBinarySplit(false),
  dtBinarySplit__setByUser__(false),
  dtBlastSplits(false),
  dtBlastSplits__setByUser__(false),
  dtCyclic(true),
  dtCyclic__setByUser__(false),
  dtForceAssignment(false),
  dtForceAssignment__setByUser__(false),
  dtInferAsLemmas(false),
  dtInferAsLemmas__setByUser__(false),
  dtNestedRec(false),
  dtNestedRec__setByUser__(false),
  dtRewriteErrorSel(false),
  dtRewriteErrorSel__setByUser__(false),
  dtSharedSelectors(true),
  dtSharedSelectors__setByUser__(false),
  sygusAbortSize(-1),
  sygusAbortSize__setByUser__(false),
  sygusFairMax(true),
  sygusFairMax__setByUser__(false),
  sygusFair(SygusFairMode::DT_SIZE),
  sygusFair__setByUser__(false),
  sygusSymBreak(true),
  sygusSymBreak__setByUser__(false),
  sygusSymBreakAgg(true),
  sygusSymBreakAgg__setByUser__(false),
  sygusSymBreakDynamic(true),
  sygusSymBreakDynamic__setByUser__(false),
  sygusSymBreakLazy(true),
  sygusSymBreakLazy__setByUser__(false),
  sygusSymBreakPbe(true),
  sygusSymBreakPbe__setByUser__(false),
  sygusSymBreakRlv(true),
  sygusSymBreakRlv__setByUser__(false),
  decisionRandomWeight(0),
  decisionRandomWeight__setByUser__(false),
  decisionThreshold(0),
  decisionThreshold__setByUser__(false),
  decisionUseWeight(false),
  decisionUseWeight__setByUser__(false),
  decisionWeightInternal(DecisionWeightInternal::OFF),
  decisionWeightInternal__setByUser__(false),
  decisionMode(DecisionMode::INTERNAL),
  decisionMode__setByUser__(false),
  decisionStopOnly(),
  decisionStopOnly__setByUser__(false),
  defaultDagThresh(1),
  defaultDagThresh__setByUser__(false),
  defaultExprDepth(0),
  defaultExprDepth__setByUser__(false),
  printExprTypes(false),
  printExprTypes__setByUser__(false),
  typeChecking(DO_SEMANTIC_CHECKS_BY_DEFAULT),
  typeChecking__setByUser__(false),
  fpExp(false),
  fpExp__setByUser__(false),
  earlyExit(true),
  earlyExit__setByUser__(false),
  help(),
  help__setByUser__(false),
  interactive(),
  interactive__setByUser__(false),
  interactivePrompt(true),
  interactivePrompt__setByUser__(false),
  seed(0),
  seed__setByUser__(false),
  segvSpin(false),
  segvSpin__setByUser__(false),
  tearDownIncremental(0),
  tearDownIncremental__setByUser__(false),
  version(),
  version__setByUser__(false),
  filesystemAccess(true),
  filesystemAccess__setByUser__(false),
  forceLogicString(),
  forceLogicString__setByUser__(false),
  globalDeclarations(false),
  globalDeclarations__setByUser__(false),
  memoryMap(),
  memoryMap__setByUser__(false),
  semanticChecks(DO_SEMANTIC_CHECKS_BY_DEFAULT),
  semanticChecks__setByUser__(false),
  strictParsing(),
  strictParsing__setByUser__(false),
  flattenHOChains(false),
  flattenHOChains__setByUser__(false),
  instFormatMode(InstFormatMode::DEFAULT),
  instFormatMode__setByUser__(false),
  modelFormatMode(ModelFormatMode::DEFAULT),
  modelFormatMode__setByUser__(false),
  aggressiveCoreMin(false),
  aggressiveCoreMin__setByUser__(false),
  allowEmptyDependencies(false),
  allowEmptyDependencies__setByUser__(false),
  fewerPreprocessingHoles(false),
  fewerPreprocessingHoles__setByUser__(false),
  lfscLetification(true),
  lfscLetification__setByUser__(false),
  minisatDumpDimacs(false),
  minisatDumpDimacs__setByUser__(false),
  minisatUseElim(true),
  minisatUseElim__setByUser__(false),
  satRandomFreq(0.0),
  satRandomFreq__setByUser__(false),
  satRandomSeed(0),
  satRandomSeed__setByUser__(false),
  sat_refine_conflicts(false),
  sat_refine_conflicts__setByUser__(false),
  satRestartFirst(25),
  satRestartFirst__setByUser__(false),
  satRestartInc(3.0),
  satRestartInc__setByUser__(false),
  satClauseDecay(0.999),
  satClauseDecay__setByUser__(false),
  satVarDecay(0.95),
  satVarDecay__setByUser__(false),
  aggressiveMiniscopeQuant(false),
  aggressiveMiniscopeQuant__setByUser__(false),
  cegisSample(CegisSampleMode::NONE),
  cegisSample__setByUser__(false),
  cegqi(false),
  cegqi__setByUser__(false),
  cegqiAll(false),
  cegqiAll__setByUser__(false),
  cegqiBv(true),
  cegqiBv__setByUser__(false),
  cegqiBvConcInv(true),
  cegqiBvConcInv__setByUser__(false),
  cegqiBvIneqMode(CegqiBvIneqMode::EQ_BOUNDARY),
  cegqiBvIneqMode__setByUser__(false),
  cegqiBvInterleaveValue(false),
  cegqiBvInterleaveValue__setByUser__(false),
  cegqiBvLinearize(true),
  cegqiBvLinearize__setByUser__(false),
  cegqiBvRmExtract(true),
  cegqiBvRmExtract__setByUser__(false),
  cegqiBvSolveNl(false),
  cegqiBvSolveNl__setByUser__(false),
  cegqiFullEffort(false),
  cegqiFullEffort__setByUser__(false),
  cegqiInnermost(true),
  cegqiInnermost__setByUser__(false),
  cegqiMidpoint(false),
  cegqiMidpoint__setByUser__(false),
  cegqiMinBounds(false),
  cegqiMinBounds__setByUser__(false),
  cegqiModel(true),
  cegqiModel__setByUser__(false),
  cegqiMultiInst(false),
  cegqiMultiInst__setByUser__(false),
  cegqiNestedQE(false),
  cegqiNestedQE__setByUser__(false),
  cegqiNopt(true),
  cegqiNopt__setByUser__(false),
  cegqiPreRegInst(false),
  cegqiPreRegInst__setByUser__(false),
  cegqiRepeatLit(false),
  cegqiRepeatLit__setByUser__(false),
  cegqiRoundUpLowerLia(false),
  cegqiRoundUpLowerLia__setByUser__(false),
  cegqiSat(true),
  cegqiSat__setByUser__(false),
  cegqiUseInfInt(false),
  cegqiUseInfInt__setByUser__(false),
  cegqiUseInfReal(false),
  cegqiUseInfReal__setByUser__(false),
  condVarSplitQuantAgg(false),
  condVarSplitQuantAgg__setByUser__(false),
  condVarSplitQuant(true),
  condVarSplitQuant__setByUser__(false),
  conjectureFilterActiveTerms(true),
  conjectureFilterActiveTerms__setByUser__(false),
  conjectureFilterCanonical(true),
  conjectureFilterCanonical__setByUser__(false),
  conjectureFilterModel(true),
  conjectureFilterModel__setByUser__(false),
  conjectureGen(false),
  conjectureGen__setByUser__(false),
  conjectureGenGtEnum(50),
  conjectureGenGtEnum__setByUser__(false),
  conjectureGenMaxDepth(3),
  conjectureGenMaxDepth__setByUser__(false),
  conjectureGenPerRound(1),
  conjectureGenPerRound__setByUser__(false),
  conjectureUeeIntro(false),
  conjectureUeeIntro__setByUser__(false),
  conjectureNoFilter(false),
  conjectureNoFilter__setByUser__(false),
  dtStcInduction(false),
  dtStcInduction__setByUser__(false),
  dtVarExpandQuant(true),
  dtVarExpandQuant__setByUser__(false),
  eMatching(true),
  eMatching__setByUser__(false),
  elimExtArithQuant(true),
  elimExtArithQuant__setByUser__(false),
  elimTautQuant(true),
  elimTautQuant__setByUser__(false),
  extRewriteQuant(false),
  extRewriteQuant__setByUser__(false),
  finiteModelFind(false),
  finiteModelFind__setByUser__(false),
  fmfBound(false),
  fmfBound__setByUser__(false),
  fmfBoundInt(false),
  fmfBoundInt__setByUser__(false),
  fmfBoundLazy(false),
  fmfBoundLazy__setByUser__(false),
  fmfFmcSimple(true),
  fmfFmcSimple__setByUser__(false),
  fmfFreshDistConst(false),
  fmfFreshDistConst__setByUser__(false),
  fmfFunWellDefined(false),
  fmfFunWellDefined__setByUser__(false),
  fmfFunWellDefinedRelevant(false),
  fmfFunWellDefinedRelevant__setByUser__(false),
  fmfInstEngine(false),
  fmfInstEngine__setByUser__(false),
  fmfTypeCompletionThresh(1000),
  fmfTypeCompletionThresh__setByUser__(false),
  fullSaturateInterleave(false),
  fullSaturateInterleave__setByUser__(false),
  fullSaturateStratify(false),
  fullSaturateStratify__setByUser__(false),
  fullSaturateQuant(false),
  fullSaturateQuant__setByUser__(false),
  fullSaturateLimit(-1),
  fullSaturateLimit__setByUser__(false),
  fullSaturateQuantRd(true),
  fullSaturateQuantRd__setByUser__(false),
  globalNegate(false),
  globalNegate__setByUser__(false),
  hoElim(false),
  hoElim__setByUser__(false),
  hoElimStoreAx(true),
  hoElimStoreAx__setByUser__(false),
  hoMatching(true),
  hoMatching__setByUser__(false),
  hoMatchingVarArgPriority(true),
  hoMatchingVarArgPriority__setByUser__(false),
  hoMergeTermDb(true),
  hoMergeTermDb__setByUser__(false),
  incrementTriggers(true),
  incrementTriggers__setByUser__(false),
  instLevelInputOnly(true),
  instLevelInputOnly__setByUser__(false),
  instMaxLevel(-1),
  instMaxLevel__setByUser__(false),
  instNoEntail(true),
  instNoEntail__setByUser__(false),
  instNoModelTrue(false),
  instNoModelTrue__setByUser__(false),
  instWhenPhase(2),
  instWhenPhase__setByUser__(false),
  instWhenStrictInterleave(true),
  instWhenStrictInterleave__setByUser__(false),
  instWhenTcFirst(true),
  instWhenTcFirst__setByUser__(false),
  instWhenMode(InstWhenMode::FULL_LAST_CALL),
  instWhenMode__setByUser__(false),
  intWfInduction(false),
  intWfInduction__setByUser__(false),
  iteDtTesterSplitQuant(false),
  iteDtTesterSplitQuant__setByUser__(false),
  iteLiftQuant(IteLiftQuantMode::SIMPLE),
  iteLiftQuant__setByUser__(false),
  literalMatchMode(LiteralMatchMode::USE),
  literalMatchMode__setByUser__(false),
  lteRestrictInstClosure(false),
  lteRestrictInstClosure__setByUser__(false),
  macrosQuant(false),
  macrosQuant__setByUser__(false),
  macrosQuantMode(MacrosQuantMode::GROUND_UF),
  macrosQuantMode__setByUser__(false),
  mbqiInterleave(false),
  mbqiInterleave__setByUser__(false),
  fmfOneInstPerRound(false),
  fmfOneInstPerRound__setByUser__(false),
  mbqiMode(MbqiMode::FMC),
  mbqiMode__setByUser__(false),
  miniscopeQuant(true),
  miniscopeQuant__setByUser__(false),
  miniscopeQuantFreeVar(true),
  miniscopeQuantFreeVar__setByUser__(false),
  multiTriggerCache(false),
  multiTriggerCache__setByUser__(false),
  multiTriggerLinear(true),
  multiTriggerLinear__setByUser__(false),
  multiTriggerPriority(false),
  multiTriggerPriority__setByUser__(false),
  multiTriggerWhenSingle(false),
  multiTriggerWhenSingle__setByUser__(false),
  partialTriggers(false),
  partialTriggers__setByUser__(false),
  preSkolemQuant(false),
  preSkolemQuant__setByUser__(false),
  preSkolemQuantAgg(true),
  preSkolemQuantAgg__setByUser__(false),
  preSkolemQuantNested(true),
  preSkolemQuantNested__setByUser__(false),
  prenexQuantUser(false),
  prenexQuantUser__setByUser__(false),
  prenexQuant(PrenexQuantMode::SIMPLE),
  prenexQuant__setByUser__(false),
  pureThTriggers(false),
  pureThTriggers__setByUser__(false),
  purifyDtTriggers(false),
  purifyDtTriggers__setByUser__(false),
  purifyTriggers(false),
  purifyTriggers__setByUser__(false),
  qcfAllConflict(false),
  qcfAllConflict__setByUser__(false),
  qcfEagerCheckRd(true),
  qcfEagerCheckRd__setByUser__(false),
  qcfEagerTest(true),
  qcfEagerTest__setByUser__(false),
  qcfNestedConflict(false),
  qcfNestedConflict__setByUser__(false),
  qcfSkipRd(false),
  qcfSkipRd__setByUser__(false),
  qcfTConstraint(false),
  qcfTConstraint__setByUser__(false),
  qcfVoExp(false),
  qcfVoExp__setByUser__(false),
  quantAlphaEquiv(true),
  quantAlphaEquiv__setByUser__(false),
  quantAntiSkolem(false),
  quantAntiSkolem__setByUser__(false),
  quantConflictFind(true),
  quantConflictFind__setByUser__(false),
  qcfMode(QcfMode::PROP_EQ),
  qcfMode__setByUser__(false),
  qcfWhenMode(QcfWhenMode::DEFAULT),
  qcfWhenMode__setByUser__(false),
  quantDynamicSplit(QuantDSplitMode::DEFAULT),
  quantDynamicSplit__setByUser__(false),
  quantEpr(false),
  quantEpr__setByUser__(false),
  quantEprMatching(true),
  quantEprMatching__setByUser__(false),
  quantFunWellDefined(false),
  quantFunWellDefined__setByUser__(false),
  quantInduction(false),
  quantInduction__setByUser__(false),
  quantModelEe(false),
  quantModelEe__setByUser__(false),
  quantRepMode(QuantRepMode::FIRST),
  quantRepMode__setByUser__(false),
  quantSplit(true),
  quantSplit__setByUser__(false),
  registerQuantBodyTerms(false),
  registerQuantBodyTerms__setByUser__(false),
  relationalTriggers(false),
  relationalTriggers__setByUser__(false),
  relevantTriggers(false),
  relevantTriggers__setByUser__(false),
  strictTriggers(false),
  strictTriggers__setByUser__(false),
  sygus(false),
  sygus__setByUser__(false),
  sygusActiveGenEnumConsts(5),
  sygusActiveGenEnumConsts__setByUser__(false),
  sygusActiveGenMode(SygusActiveGenMode::AUTO),
  sygusActiveGenMode__setByUser__(false),
  sygusAddConstGrammar(false),
  sygusAddConstGrammar__setByUser__(false),
  sygusArgRelevant(false),
  sygusArgRelevant__setByUser__(false),
  sygusInvAutoUnfold(true),
  sygusInvAutoUnfold__setByUser__(false),
  sygusBoolIteReturnConst(true),
  sygusBoolIteReturnConst__setByUser__(false),
  sygusCoreConnective(false),
  sygusCoreConnective__setByUser__(false),
  sygusConstRepairAbort(false),
  sygusConstRepairAbort__setByUser__(false),
  sygusEvalOpt(true),
  sygusEvalOpt__setByUser__(false),
  sygusEvalUnfold(true),
  sygusEvalUnfold__setByUser__(false),
  sygusEvalUnfoldBool(true),
  sygusEvalUnfoldBool__setByUser__(false),
  sygusExprMinerCheckTimeout(),
  sygusExprMinerCheckTimeout__setByUser__(false),
  sygusExprMinerCheckUseExport(true),
  sygusExprMinerCheckUseExport__setByUser__(false),
  sygusExtRew(true),
  sygusExtRew__setByUser__(false),
  sygusFilterSolRevSubsume(false),
  sygusFilterSolRevSubsume__setByUser__(false),
  sygusFilterSolMode(SygusFilterSolMode::NONE),
  sygusFilterSolMode__setByUser__(false),
  sygusGrammarConsMode(SygusGrammarConsMode::SIMPLE),
  sygusGrammarConsMode__setByUser__(false),
  sygusGrammarNorm(false),
  sygusGrammarNorm__setByUser__(false),
  sygusInference(false),
  sygusInference__setByUser__(false),
  sygusInst(false),
  sygusInst__setByUser__(false),
  sygusInvTemplWhenSyntax(false),
  sygusInvTemplWhenSyntax__setByUser__(false),
  sygusInvTemplMode(SygusInvTemplMode::POST),
  sygusInvTemplMode__setByUser__(false),
  sygusMinGrammar(true),
  sygusMinGrammar__setByUser__(false),
  sygusUnifPbe(true),
  sygusUnifPbe__setByUser__(false),
  sygusPbeMultiFair(false),
  sygusPbeMultiFair__setByUser__(false),
  sygusPbeMultiFairDiff(0),
  sygusPbeMultiFairDiff__setByUser__(false),
  sygusQePreproc(false),
  sygusQePreproc__setByUser__(false),
  sygusQueryGen(false),
  sygusQueryGen__setByUser__(false),
  sygusQueryGenCheck(true),
  sygusQueryGenCheck__setByUser__(false),
  sygusQueryGenDumpFiles(SygusQueryDumpFilesMode::NONE),
  sygusQueryGenDumpFiles__setByUser__(false),
  sygusQueryGenThresh(5),
  sygusQueryGenThresh__setByUser__(false),
  sygusRecFun(true),
  sygusRecFun__setByUser__(false),
  sygusRecFunEvalLimit(1000),
  sygusRecFunEvalLimit__setByUser__(false),
  sygusRepairConst(false),
  sygusRepairConst__setByUser__(false),
  sygusRepairConstTimeout(),
  sygusRepairConstTimeout__setByUser__(false),
  sygusRew(false),
  sygusRew__setByUser__(false),
  sygusRewSynth(false),
  sygusRewSynth__setByUser__(false),
  sygusRewSynthAccel(false),
  sygusRewSynthAccel__setByUser__(false),
  sygusRewSynthCheck(false),
  sygusRewSynthCheck__setByUser__(false),
  sygusRewSynthFilterCong(true),
  sygusRewSynthFilterCong__setByUser__(false),
  sygusRewSynthFilterMatch(true),
  sygusRewSynthFilterMatch__setByUser__(false),
  sygusRewSynthFilterNonLinear(false),
  sygusRewSynthFilterNonLinear__setByUser__(false),
  sygusRewSynthFilterOrder(true),
  sygusRewSynthFilterOrder__setByUser__(false),
  sygusRewSynthInput(false),
  sygusRewSynthInput__setByUser__(false),
  sygusRewSynthInputNVars(3),
  sygusRewSynthInputNVars__setByUser__(false),
  sygusRewSynthInputUseBool(false),
  sygusRewSynthInputUseBool__setByUser__(false),
  sygusRewSynthRec(false),
  sygusRewSynthRec__setByUser__(false),
  sygusRewVerify(false),
  sygusRewVerify__setByUser__(false),
  sygusRewVerifyAbort(true),
  sygusRewVerifyAbort__setByUser__(false),
  sygusSampleFpUniform(false),
  sygusSampleFpUniform__setByUser__(false),
  sygusSampleGrammar(true),
  sygusSampleGrammar__setByUser__(false),
  sygusSamples(1000),
  sygusSamples__setByUser__(false),
  cegqiSingleInvAbort(false),
  cegqiSingleInvAbort__setByUser__(false),
  cegqiSingleInvPartial(false),
  cegqiSingleInvPartial__setByUser__(false),
  cegqiSingleInvReconstructLimit(10000),
  cegqiSingleInvReconstructLimit__setByUser__(false),
  cegqiSingleInvReconstruct(CegqiSingleInvRconsMode::ALL_LIMIT),
  cegqiSingleInvReconstruct__setByUser__(false),
  cegqiSingleInvReconstructConst(true),
  cegqiSingleInvReconstructConst__setByUser__(false),
  cegqiSingleInvMode(CegqiSingleInvMode::NONE),
  cegqiSingleInvMode__setByUser__(false),
  sygusStream(false),
  sygusStream__setByUser__(false),
  sygusTemplEmbedGrammar(false),
  sygusTemplEmbedGrammar__setByUser__(false),
  sygusUnifCondIndNoRepeatSol(true),
  sygusUnifCondIndNoRepeatSol__setByUser__(false),
  sygusUnifPi(SygusUnifPiMode::NONE),
  sygusUnifPi__setByUser__(false),
  sygusUnifShuffleCond(false),
  sygusUnifShuffleCond__setByUser__(false),
  sygusVerifySubcall(true),
  sygusVerifySubcall__setByUser__(false),
  termDbMode(TermDbMode::ALL),
  termDbMode__setByUser__(false),
  trackInstLemmas(false),
  trackInstLemmas__setByUser__(false),
  triggerActiveSelMode(TriggerActiveSelMode::ALL),
  triggerActiveSelMode__setByUser__(false),
  triggerSelMode(TriggerSelMode::MIN),
  triggerSelMode__setByUser__(false),
  userPatternsQuant(UserPatMode::TRUST),
  userPatternsQuant__setByUser__(false),
  varElimQuant(true),
  varElimQuant__setByUser__(false),
  varIneqElimQuant(true),
  varIneqElimQuant__setByUser__(false),
  sepCheckNeg(true),
  sepCheckNeg__setByUser__(false),
  sepChildRefine(false),
  sepChildRefine__setByUser__(false),
  sepDisequalC(true),
  sepDisequalC__setByUser__(false),
  sepExp(false),
  sepExp__setByUser__(false),
  sepMinimalRefine(false),
  sepMinimalRefine__setByUser__(false),
  sepPreSkolemEmp(false),
  sepPreSkolemEmp__setByUser__(false),
  setsExt(false),
  setsExt__setByUser__(false),
  setsInferAsLemmas(true),
  setsInferAsLemmas__setByUser__(false),
  setsProxyLemmas(false),
  setsProxyLemmas__setByUser__(false),
  abstractValues(false),
  abstractValues__setByUser__(false),
  ackermann(false),
  ackermann__setByUser__(false),
  bitblastStep(1),
  bitblastStep__setByUser__(false),
  blockModelsMode(BlockModelsMode::NONE),
  blockModelsMode__setByUser__(false),
  bvEagerAssertStep(1),
  bvEagerAssertStep__setByUser__(false),
  bvPropagationStep(1),
  bvPropagationStep__setByUser__(false),
  bvSatConflictStep(1),
  bvSatConflictStep__setByUser__(false),
  checkAbducts(false),
  checkAbducts__setByUser__(false),
  checkModels(),
  checkModels__setByUser__(false),
  checkProofs(),
  checkProofs__setByUser__(false),
  checkSynthSol(false),
  checkSynthSol__setByUser__(false),
  checkUnsatCores(),
  checkUnsatCores__setByUser__(false),
  cnfStep(1),
  cnfStep__setByUser__(false),
  cpuTime(false),
  cpuTime__setByUser__(false),
  debugCheckModels(),
  debugCheckModels__setByUser__(false),
  decisionStep(1),
  decisionStep__setByUser__(false),
  diagnosticChannelName(),
  diagnosticChannelName__setByUser__(false),
  dumpInstantiations(false),
  dumpInstantiations__setByUser__(false),
  dumpModels(false),
  dumpModels__setByUser__(false),
  dumpProofs(false),
  dumpProofs__setByUser__(false),
  dumpSynth(false),
  dumpSynth__setByUser__(false),
  dumpToFileName(),
  dumpToFileName__setByUser__(false),
  dumpUnsatCores(false),
  dumpUnsatCores__setByUser__(false),
  dumpUnsatCoresFull(false),
  dumpUnsatCoresFull__setByUser__(false),
  dumpModeString(),
  dumpModeString__setByUser__(false),
  expandDefinitions(false),
  expandDefinitions__setByUser__(false),
  extRewPrep(false),
  extRewPrep__setByUser__(false),
  extRewPrepAgg(false),
  extRewPrepAgg__setByUser__(false),
  forceNoLimitCpuWhileDump(false),
  forceNoLimitCpuWhileDump__setByUser__(false),
  hardLimit(false),
  hardLimit__setByUser__(false),
  incrementalSolving(true),
  incrementalSolving__setByUser__(false),
  interactiveMode(),
  interactiveMode__setByUser__(false),
  doITESimp(),
  doITESimp__setByUser__(false),
  lemmaStep(1),
  lemmaStep__setByUser__(false),
  modelCoresMode(ModelCoresMode::NONE),
  modelCoresMode__setByUser__(false),
  modelUninterpDtEnum(false),
  modelUninterpDtEnum__setByUser__(false),
  modelWitnessValue(false),
  modelWitnessValue__setByUser__(false),
  doITESimpOnRepeat(false),
  doITESimpOnRepeat__setByUser__(false),
  parseStep(1),
  parseStep__setByUser__(false),
  preprocessStep(1),
  preprocessStep__setByUser__(false),
  produceAbducts(false),
  produceAbducts__setByUser__(false),
  produceAssertions(),
  produceAssertions__setByUser__(false),
  produceAssignments(false),
  produceAssignments__setByUser__(false),
  produceModels(false),
  produceModels__setByUser__(false),
  unsatAssumptions(false),
  unsatAssumptions__setByUser__(false),
  unsatCores(),
  unsatCores__setByUser__(false),
  proof(false),
  proof__setByUser__(false),
  quantifierStep(1),
  quantifierStep__setByUser__(false),
  regularChannelName(),
  regularChannelName__setByUser__(false),
  repeatSimp(),
  repeatSimp__setByUser__(false),
  restartStep(1),
  restartStep__setByUser__(false),
  rewriteStep(1),
  rewriteStep__setByUser__(false),
  perCallResourceLimit(),
  perCallResourceLimit__setByUser__(false),
  cumulativeResourceLimit(),
  cumulativeResourceLimit__setByUser__(false),
  satConflictStep(1),
  satConflictStep__setByUser__(false),
  compressItes(false),
  compressItes__setByUser__(false),
  zombieHuntThreshold(524288),
  zombieHuntThreshold__setByUser__(false),
  simplifyWithCareEnabled(false),
  simplifyWithCareEnabled__setByUser__(false),
  simplificationMode(SimplificationMode::BATCH),
  simplificationMode__setByUser__(false),
  solveBVAsInt(0),
  solveBVAsInt__setByUser__(false),
  solveIntAsBV(0),
  solveIntAsBV__setByUser__(false),
  solveRealAsInt(false),
  solveRealAsInt__setByUser__(false),
  sortInference(false),
  sortInference__setByUser__(false),
  doStaticLearning(true),
  doStaticLearning__setByUser__(false),
  sygusOut(SygusSolutionOutMode::STATUS_AND_DEF),
  sygusOut__setByUser__(false),
  sygusPrintCallbacks(true),
  sygusPrintCallbacks__setByUser__(false),
  theoryCheckStep(1),
  theoryCheckStep__setByUser__(false),
  perCallMillisecondLimit(),
  perCallMillisecondLimit__setByUser__(false),
  cumulativeMillisecondLimit(),
  cumulativeMillisecondLimit__setByUser__(false),
  unconstrainedSimp(false),
  unconstrainedSimp__setByUser__(false),
  regExpElim(false),
  regExpElim__setByUser__(false),
  regExpElimAgg(false),
  regExpElimAgg__setByUser__(false),
  stringRegExpInterMode(RegExpInterMode::CONSTANT),
  stringRegExpInterMode__setByUser__(false),
  stringCheckEntailLen(true),
  stringCheckEntailLen__setByUser__(false),
  stringEager(false),
  stringEager__setByUser__(false),
  stringEagerLen(true),
  stringEagerLen__setByUser__(false),
  stringExp(false),
  stringExp__setByUser__(false),
  stringFlatForms(true),
  stringFlatForms__setByUser__(false),
  stringFMF(false),
  stringFMF__setByUser__(false),
  stringGuessModel(false),
  stringGuessModel__setByUser__(false),
  stringInferAsLemmas(false),
  stringInferAsLemmas__setByUser__(false),
  stringInferSym(true),
  stringInferSym__setByUser__(false),
  stringIgnNegMembership(false),
  stringIgnNegMembership__setByUser__(false),
  stringLazyPreproc(true),
  stringLazyPreproc__setByUser__(false),
  stringLenNorm(true),
  stringLenNorm__setByUser__(false),
  stringLenPropCsp(false),
  stringLenPropCsp__setByUser__(false),
  stringMinPrefixExplain(true),
  stringMinPrefixExplain__setByUser__(false),
  stdPrintASCII(false),
  stdPrintASCII__setByUser__(false),
  stringProcessLoopMode(ProcessLoopMode::FULL),
  stringProcessLoopMode__setByUser__(false),
  stringRExplainLemmas(true),
  stringRExplainLemmas__setByUser__(false),
  assignFunctionValues(true),
  assignFunctionValues__setByUser__(false),
  condenseFunctionValues(true),
  condenseFunctionValues__setByUser__(false),
  theoryOfMode(TheoryOfMode::THEORY_OF_TYPE_BASED),
  theoryOfMode__setByUser__(false),
  ufSymmetryBreaker(true),
  ufSymmetryBreaker__setByUser__(false),
  ufHo(false),
  ufHo__setByUser__(false),
  ufHoExt(true),
  ufHoExt__setByUser__(false),
  ufssAbortCardinality(-1),
  ufssAbortCardinality__setByUser__(false),
  ufssFairness(true),
  ufssFairness__setByUser__(false),
  ufssFairnessMonotone(false),
  ufssFairnessMonotone__setByUser__(false),
  ufssTotality(false),
  ufssTotality__setByUser__(false),
  ufssTotalityLimited(-1),
  ufssTotalityLimited__setByUser__(false),
  ufssTotalitySymBreak(false),
  ufssTotalitySymBreak__setByUser__(false),
  ufssMode(UfssMode::FULL),
  ufssMode__setByUser__(false)
{
}


static const std::string mostCommonOptionsDescription = "\
Most commonly-used CVC4 options:\n"
"  --lang=LANG | -L LANG  force input language (default is \"auto\"; see --lang\n"
"                         help)\n"
"  --output-lang=LANG     force output language (default is \"auto\"; see\n"
"                         --output-lang help)\n"
"  --quiet | -q           decrease verbosity (may be repeated)\n"
"  --stats                give statistics on exit [*]\n"
"  --verbose | -v         increase verbosity (may be repeated)\n"
"  --copyright            show CVC4 copyright information\n"
"  --help | -h            full command line reference\n"
"  --seed=N | -s N        seed for random number generator\n"
"  --show-config          show CVC4 static configuration\n"
"  --version | -V         identify this CVC4 binary\n"
"  --strict-parsing       be less tolerant of non-conforming inputs [*]\n"
"  --cpu-time             measures CPU time if set to true and wall time if false\n"
"                         (default false) [*]\n"
"  --dump-to=FILE         all dumping goes to FILE (instead of stdout)\n"
"  --dump=MODE            dump preprocessed assertions, etc., see --dump=help\n"
"  --hard-limit           the resource limit is hard potentially leaving the\n"
"                         smtEngine in an unsafe state (should be destroyed and\n"
"                         rebuild after resourcing out) [*]\n"
"  --incremental | -i     enable incremental solving [*]\n"
"  --produce-assertions   keep an assertions list (enables get-assertions\n"
"                         command) [*]\n"
"  --produce-models | -m  support the get-value and get-model commands [*]\n"
"  --rlimit-per=N         enable resource limiting per query\n"
"  --rlimit=N             enable resource limiting (currently, roughly the number\n"
"                         of SAT conflicts)\n"
"  --tlimit-per=MS        enable time limiting per query (give milliseconds)\n"
"  --tlimit=MS            enable time limiting (give milliseconds)\n";


static const std::string optionsDescription = mostCommonOptionsDescription + "\n\
\n\
Additional CVC4 options:\n"
"\nFrom the Arithmetic theory module:\n"
"  --approx-branch-depth=N\n"
"                         maximum branch depth the approximate solver is allowed\n"
"                         to take\n"
"  --arith-brab           whether to use simple rounding, similar to a unit-cube\n"
"                         test, for integers [*]\n"
"  --arith-no-partial-fun do not use partial function semantics for arithmetic\n"
"                         (not SMT LIB compliant) [*]\n"
"  --arith-prop-clauses=N rows shorter than this are propagated as clauses\n"
"  --arith-prop=MODE      turns on arithmetic propagation (default is 'old', see\n"
"                         --arith-prop=help)\n"
"  --arith-rewrite-equalities\n"
"                         turns on the preprocessing rewrite turning equalities\n"
"                         into a conjunction of inequalities [*]\n"
"  --collect-pivot-stats  collect the pivot history [*]\n"
"  --cut-all-bounded      turns on the integer solving step of periodically\n"
"                         cutting all integer variables that have both upper and\n"
"                         lower bounds [*]\n"
"  --dio-decomps          let skolem variables for integer divisibility\n"
"                         constraints leak from the dio solver [*]\n"
"  --dio-repeat           handle dio solver constraints in mass or one at a time\n"
"                         [*]\n"
"  --dio-solver           turns on Linear Diophantine Equation solver (Griggio,\n"
"                         JSAT 2012) [*]\n"
"  --dio-turns=N          turns in a row dio solver cutting gets\n"
"  --error-selection-rule=RULE\n"
"                         change the pivot rule for the basic variable (default\n"
"                         is 'min', see --pivot-rule help)\n"
"  --fc-penalties         turns on degenerate pivot penalties [*]\n"
"  --heuristic-pivots=N   the number of times to apply the heuristic pivot rule;\n"
"                         if N < 0, this defaults to the number of variables; if\n"
"                         this is unset, this is tuned by the logic selection\n"
"  --lemmas-on-replay-failure\n"
"                         attempt to use external lemmas if approximate solve\n"
"                         integer failed [*]\n"
"  --maxCutsInContext=N   maximum cuts in a given context before signalling a\n"
"                         restart\n"
"  --miplib-trick         turns on the preprocessing step of attempting to infer\n"
"                         bounds on miplib problems [*]\n"
"  --miplib-trick-subs=N  do substitution for miplib 'tmp' vars if defined in <=\n"
"                         N eliminated vars\n"
"  --new-prop             use the new row propagation system [*]\n"
"  --nl-ext               extended approach to non-linear [*]\n"
"  --nl-ext-ent-conf      check for entailed conflicts in non-linear solver [*]\n"
"  --nl-ext-factor        use factoring inference in non-linear solver [*]\n"
"  --nl-ext-inc-prec      whether to increment the precision for irrational\n"
"                         function constraints [*]\n"
"  --nl-ext-purify        purify non-linear terms at preprocess [*]\n"
"  --nl-ext-rbound        use resolution-style inference for inferring new bounds\n"
"                         [*]\n"
"  --nl-ext-rewrite       do rewrites in non-linear solver [*]\n"
"  --nl-ext-split-zero    initial splits on zero for all variables [*]\n"
"  --nl-ext-tf-taylor-deg=N\n"
"                         initial degree of polynomials for Taylor approximation\n"
"  --nl-ext-tf-tplanes    use non-terminating tangent plane strategy for\n"
"                         transcendental functions for non-linear [*]\n"
"  --nl-ext-tplanes       use non-terminating tangent plane strategy for\n"
"                         non-linear [*]\n"
"  --nl-ext-tplanes-interleave\n"
"                         interleave tangent plane strategy for non-linear [*]\n"
"  --pb-rewrites          apply pseudo boolean rewrites [*]\n"
"  --pivot-threshold=N    sets the number of pivots using --pivot-rule per basic\n"
"                         variable per simplex instance before using variable\n"
"                         order\n"
"  --pp-assert-max-sub-size=N\n"
"                         threshold for substituting an equality in ppAssert\n"
"  --prop-row-length=N    sets the maximum row length to be used in propagation\n"
"  --replay-early-close-depth=N\n"
"                         multiples of the depths to try to close the approx log\n"
"                         eagerly\n"
"  --replay-failure-penalty=N\n"
"                         number of solve integer attempts to skips after a\n"
"                         numeric failure\n"
"  --replay-lemma-reject-cut=N\n"
"                         maximum complexity of any coefficient while outputting\n"
"                         replaying cut lemmas\n"
"  --replay-num-err-penalty=N\n"
"                         number of solve integer attempts to skips after a\n"
"                         numeric failure\n"
"  --replay-reject-cut=N  maximum complexity of any coefficient while replaying\n"
"                         cuts\n"
"  --replay-soi-major-threshold-pen=N\n"
"                         threshold for a major tolerance failure by the\n"
"                         approximate solver\n"
"  --replay-soi-major-threshold=T\n"
"                         threshold for a major tolerance failure by the\n"
"                         approximate solver\n"
"  --replay-soi-minor-threshold-pen=N\n"
"                         threshold for a minor tolerance failure by the\n"
"                         approximate solver\n"
"  --replay-soi-minor-threshold=T\n"
"                         threshold for a minor tolerance failure by the\n"
"                         approximate solver\n"
"  --restrict-pivots      have a pivot cap for simplex at effort levels below\n"
"                         fullEffort [*]\n"
"  --revert-arith-models-on-unsat\n"
"                         revert the arithmetic model to a known safe model on\n"
"                         unsat if one is cached [*]\n"
"  --rr-turns=N           round robin turn\n"
"  --se-solve-int         attempt to use the approximate solve integer method on\n"
"                         standard effort [*]\n"
"  --simplex-check-period=N\n"
"                         the number of pivots to do in simplex before rechecking\n"
"                         for a conflict on all variables\n"
"  --soi-qe               use quick explain to minimize the sum of infeasibility\n"
"                         conflicts [*]\n"
"  --standard-effort-variable-order-pivots=N\n"
"                         limits the number of pivots in a single invocation of\n"
"                         check() at a non-full effort level using Bland's pivot\n"
"                         rule (EXPERTS only)\n"
"  --unate-lemmas=MODE    determines which lemmas to add before solving (default\n"
"                         is 'all', see --unate-lemmas=help)\n"
"  --use-approx           attempt to use an approximate solver [*]\n"
"  --use-fcsimplex        use focusing and converging simplex (FMCAD 2013\n"
"                         submission) [*]\n"
"  --use-soi              use sum of infeasibility simplex (FMCAD 2013\n"
"                         submission) [*]\n"
"\nFrom the Arrays theory module:\n"
"  --arrays-config=N      set different array option configurations - for\n"
"                         developers only\n"
"  --arrays-eager-index   turn on eager index splitting for generated array\n"
"                         lemmas [*]\n"
"  --arrays-eager-lemmas  turn on eager lemma generation for arrays [*]\n"
"  --arrays-lazy-rintro1  turn on optimization to only perform RIntro1 rule\n"
"                         lazily (see Jovanovic/Barrett 2012: Being Careful with\n"
"                         Theory Combination) [*]\n"
"  --arrays-model-based   turn on model-based array solver [*]\n"
"  --arrays-optimize-linear\n"
"                         turn on optimization for linear array terms (see de\n"
"                         Moura FMCAD 09 arrays paper) [*]\n"
"  --arrays-prop=N        propagation effort for arrays: 0 is none, 1 is some, 2\n"
"                         is full\n"
"  --arrays-reduce-sharing\n"
"                         use model information to reduce size of care graph for\n"
"                         arrays [*]\n"
"  --arrays-weak-equiv    use algorithm from Christ/Hoenicke (SMT 2014) [*]\n"
"\nFrom the Base module:\n"
"  --debug=TAG | -d TAG   debug something (e.g. -d arith), can repeat\n"
"  --parse-only           exit after parsing input [*]\n"
"  --preprocess-only      exit after preprocessing input [*]\n"
"  --print-success        print the \"success\" output required of SMT-LIBv2 [*]\n"
"  --stats-every-query    in incremental mode, print stats after every\n"
"                         satisfiability or validity query [*]\n"
"  --stats-hide-zeros     hide statistics which are zero [*]\n"
"  --trace=TAG | -t TAG   trace something (e.g. -t pushpop), can repeat\n"
"\nFrom the Bitvector theory module:\n"
"  --bitblast-aig         bitblast by first converting to AIG (implies\n"
"                         --bitblast=eager) [*]\n"
"  --bitblast=MODE        choose bitblasting mode, see --bitblast=help\n"
"  --bitwise-eq           lift equivalence with one-bit bit-vectors to be boolean\n"
"                         operations [*]\n"
"  --bool-to-bv=MODE      convert booleans to bit-vectors of size 1 at various\n"
"                         levels of aggressiveness, see --bool-to-bv=help\n"
"  --bv-aig-simp=COMMAND  abc command to run AIG simplifications (implies\n"
"                         --bitblast-aig, default is \"balance;drw\") (EXPERTS\n"
"                         only)\n"
"  --bv-alg-extf          algebraic inferences for extended functions [*]\n"
"  --bv-algebraic-budget=N\n"
"                         the budget allowed for the algebraic solver in number\n"
"                         of SAT conflicts (EXPERTS only)\n"
"  --bv-algebraic-solver  turn on the algebraic solver for the bit-vector theory\n"
"                         (only if --bitblast=lazy) [*]\n"
"  --bv-div-zero-const    always return -1 on division by zero [*]\n"
"  --bv-eager-explanations\n"
"                         compute bit-blasting propagation explanations eagerly\n"
"                         (EXPERTS only) [*]\n"
"  --bv-eq-slicer=MODE    turn on the slicing equality solver for the bit-vector\n"
"                         theory (only if --bitblast=lazy)\n"
"  --bv-eq-solver         use the equality engine for the bit-vector theory (only\n"
"                         if --bitblast=lazy) [*]\n"
"  --bv-extract-arith     enable rewrite pushing extract [i:0] over arithmetic\n"
"                         operations (can blow up) (EXPERTS only) [*]\n"
"  --bv-gauss-elim        simplify formula via Gaussian Elimination if applicable\n"
"                         (EXPERTS only) [*]\n"
"  --bv-inequality-solver turn on the inequality solver for the bit-vector theory\n"
"                         (only if --bitblast=lazy) [*]\n"
"  --bv-intro-pow2        introduce bitvector powers of two as a preprocessing\n"
"                         pass (EXPERTS only) [*]\n"
"  --bv-lazy-reduce-extf  reduce extended functions like bv2nat and int2bv at\n"
"                         last call instead of full effort [*]\n"
"  --bv-lazy-rewrite-extf lazily rewrite extended functions like bv2nat and\n"
"                         int2bv [*]\n"
"  --bv-num-func=N        number of function symbols in conflicts that are\n"
"                         generalized (EXPERTS only)\n"
"  --bv-optimize-sat-proof=MODE\n"
"                         enable SAT proof optimizations, see\n"
"                         --bv-optimize-sat-proof=help (EXPERTS only)\n"
"  --bv-print-consts-as-indexed-symbols\n"
"                         print bit-vector constants in binary (e.g. #b0001)\n"
"                         instead of decimal (e.g. (_ bv1 4)), applies to SMT-LIB\n"
"                         2.x [*]\n"
"  --bv-proof-format=MODE choose which UNSAT proof format to use, see\n"
"                         --bv-sat-solver=help (EXPERTS only)\n"
"  --bv-propagate         use bit-vector propagation in the bit-blaster [*]\n"
"  --bv-quick-xplain      minimize bv conflicts using the QuickXplain algorithm\n"
"                         (EXPERTS only) [*]\n"
"  --bv-sat-solver=MODE   choose which sat solver to use, see\n"
"                         --bv-sat-solver=help (EXPERTS only)\n"
"  --bv-to-bool           lift bit-vectors of size 1 to booleans when possible\n"
"                         [*]\n"
"\nFrom the Datatypes theory module:\n"
"  --cdt-bisimilar        do bisimilarity check for co-datatypes [*]\n"
"  --dt-binary-split      do binary splits for datatype constructor types [*]\n"
"  --dt-blast-splits      when applicable, blast splitting lemmas for all\n"
"                         variables at once [*]\n"
"  --dt-cyclic            do cyclicity check for datatypes [*]\n"
"  --dt-force-assignment  force the datatypes solver to give specific values to\n"
"                         all datatypes terms before answering sat [*]\n"
"  --dt-infer-as-lemmas   always send lemmas out instead of making internal\n"
"                         inferences [*]\n"
"  --dt-nested-rec        allow nested recursion in datatype definitions [*]\n"
"  --dt-rewrite-error-sel rewrite incorrectly applied selectors to arbitrary\n"
"                         ground term (EXPERTS only) [*]\n"
"  --dt-share-sel         internally use shared selectors across multiple\n"
"                         constructors [*]\n"
"  --sygus-abort-size=N   tells enumerative sygus to only consider solutions up\n"
"                         to term size N (-1 == no limit, default)\n"
"  --sygus-fair-max       use max instead of sum for multi-function sygus\n"
"                         conjectures [*]\n"
"  --sygus-fair=MODE      if and how to apply fairness for sygus\n"
"  --sygus-sym-break      simple sygus symmetry breaking lemmas [*]\n"
"  --sygus-sym-break-agg  use aggressive checks for simple sygus symmetry\n"
"                         breaking lemmas [*]\n"
"  --sygus-sym-break-dynamic\n"
"                         dynamic sygus symmetry breaking lemmas [*]\n"
"  --sygus-sym-break-lazy lazily add symmetry breaking lemmas for terms [*]\n"
"  --sygus-sym-break-pbe  sygus symmetry breaking lemmas based on pbe conjectures\n"
"                         [*]\n"
"  --sygus-sym-break-rlv  add relevancy conditions to symmetry breaking lemmas\n"
"                         [*]\n"
"\nFrom the Decision heuristics module:\n"
"  --decision-random-weight=N\n"
"                         assign random weights to nodes between 0 and N-1 (0:\n"
"                         disable) (EXPERTS only)\n"
"  --decision-threshold=N ignore all nodes greater than threshold in first\n"
"                         attempt to pick decision (EXPERTS only)\n"
"  --decision-use-weight  use the weight nodes (locally, by looking at children)\n"
"                         to direct recursive search (EXPERTS only) [*]\n"
"  --decision-weight-internal=HOW\n"
"                         compute weights of internal nodes using children: off,\n"
"                         max, sum, usr1 (EXPERTS only)\n"
"  --decision=MODE        choose decision mode, see --decision=help\n"
"\nFrom the Expression package module:\n"
"  --dag-thresh=N         dagify common subexprs appearing > N times (1 ==\n"
"                         default, 0 == don't dagify)\n"
"  --expr-depth=N         print exprs to depth N (0 == default, -1 == no limit)\n"
"  --print-expr-types     print types with variables when printing exprs [*]\n"
"  --type-checking        never type check expressions [*]\n"
"\nFrom the Fp module:\n"
"  --fp-exp               Allow floating-point sorts of all sizes, rather than\n"
"                         only Float32 (8/24) or Float64 (11/53) (experimental)\n"
"                         [*]\n"
"\nFrom the Driver module:\n"
"  --early-exit           do not run destructors at exit; default on except in\n"
"                         debug builds (EXPERTS only) [*]\n"
"  --interactive          force interactive/non-interactive mode [*]\n"
"  --segv-spin            spin on segfault/other crash waiting for gdb [*]\n"
"  --show-debug-tags      show all available tags for debugging\n"
"  --show-trace-tags      show all available tags for tracing\n"
"  --tear-down-incremental=N\n"
"                         implement PUSH/POP/multi-query by destroying and\n"
"                         recreating SmtEngine every N queries (EXPERTS only)\n"
"\nFrom the Parser module:\n"
"  --force-logic=LOGIC    set the logic, and override all further user attempts\n"
"                         to change it (EXPERTS only)\n"
"  --mmap                 memory map file input [*]\n"
"\nFrom the Printing module:\n"
"  --flatten-ho-chains    print (binary) application chains in a flattened way,\n"
"                         e.g. (a b c) rather than ((a b) c) [*]\n"
"  --inst-format=MODE     print format mode for instantiations, see\n"
"                         --inst-format=help\n"
"  --model-format=MODE    print format mode for models, see --model-format=help\n"
"\nFrom the Proof module:\n"
"  --aggressive-core-min  turns on aggressive unsat core minimization\n"
"                         (experimental) [*]\n"
"  --allow-empty-dependencies\n"
"                         if unable to track the dependencies of a\n"
"                         rewritten/preprocessed assertion, fail silently [*]\n"
"  --fewer-preprocessing-holes\n"
"                         try to eliminate preprocessing holes in proofs [*]\n"
"  --lfsc-letification    turns on global letification in LFSC proofs [*]\n"
"\nFrom the SAT layer module:\n"
"  --minisat-dump-dimacs  instead of solving minisat dumps the asserted clauses\n"
"                         in Dimacs format [*]\n"
"  --minisat-elimination  use Minisat elimination [*]\n"
"  --random-freq=P        sets the frequency of random decisions in the sat\n"
"                         solver (P=0.0 by default)\n"
"  --random-seed=S        sets the random seed for the sat solver\n"
"  --refine-conflicts     refine theory conflict clauses (default false) [*]\n"
"  --restart-int-base=N   sets the base restart interval for the sat solver (N=25\n"
"                         by default)\n"
"  --restart-int-inc=F    sets the restart interval increase factor for the sat\n"
"                         solver (F=3.0 by default)\n"
"\nFrom the Quantifiers module:\n"
"  --ag-miniscope-quant   perform aggressive miniscoping for quantifiers [*]\n"
"  --cegis-sample=MODE    mode for using samples in the counterexample-guided\n"
"                         inductive synthesis loop\n"
"  --cegqi                turns on counterexample-based quantifier instantiation\n"
"                         [*]\n"
"  --cegqi-all            apply counterexample-based instantiation to all\n"
"                         quantified formulas [*]\n"
"  --cegqi-bv             use word-level inversion approach for\n"
"                         counterexample-guided quantifier instantiation for\n"
"                         bit-vectors [*]\n"
"  --cegqi-bv-concat-inv  compute inverse for concat over equalities rather than\n"
"                         producing an invertibility condition [*]\n"
"  --cegqi-bv-ineq=MODE   choose mode for handling bit-vector inequalities with\n"
"                         counterexample-guided instantiation\n"
"  --cegqi-bv-interleave-value\n"
"                         interleave model value instantiation with word-level\n"
"                         inversion approach [*]\n"
"  --cegqi-bv-linear      linearize adder chains for variables [*]\n"
"  --cegqi-bv-rm-extract  replaces extract terms with variables for\n"
"                         counterexample-guided instantiation for bit-vectors [*]\n"
"  --cegqi-bv-solve-nl    try to solve non-linear bv literals using model value\n"
"                         projections [*]\n"
"  --cegqi-full           turns on full effort counterexample-based quantifier\n"
"                         instantiation, which may resort to model-value\n"
"                         instantiation [*]\n"
"  --cegqi-innermost      only process innermost quantified formulas in\n"
"                         counterexample-based quantifier instantiation [*]\n"
"  --cegqi-midpoint       choose substitutions based on midpoints of lower and\n"
"                         upper bounds for counterexample-based quantifier\n"
"                         instantiation [*]\n"
"  --cegqi-min-bounds     use minimally constrained lower/upper bound for\n"
"                         counterexample-based quantifier instantiation [*]\n"
"  --cegqi-model          guide instantiations by model values for\n"
"                         counterexample-based quantifier instantiation [*]\n"
"  --cegqi-multi-inst     when applicable, do multi instantiations per quantifier\n"
"                         per round in counterexample-based quantifier\n"
"                         instantiation [*]\n"
"  --cegqi-nested-qe      process nested quantified formulas with quantifier\n"
"                         elimination in counterexample-based quantifier\n"
"                         instantiation [*]\n"
"  --cegqi-nopt           non-optimal bounds for counterexample-based quantifier\n"
"                         instantiation [*]\n"
"  --cegqi-prereg-inst    preregister ground instantiations in\n"
"                         counterexample-based quantifier instantiation [*]\n"
"  --cegqi-repeat-lit     solve literals more than once in counterexample-based\n"
"                         quantifier instantiation [*]\n"
"  --cegqi-round-up-lia   round up integer lower bounds in substitutions for\n"
"                         counterexample-based quantifier instantiation [*]\n"
"  --cegqi-sat            answer sat when quantifiers are asserted with\n"
"                         counterexample-based quantifier instantiation [*]\n"
"  --cegqi-use-inf-int    use integer infinity for vts in counterexample-based\n"
"                         quantifier instantiation [*]\n"
"  --cegqi-use-inf-real   use real infinity for vts in counterexample-based\n"
"                         quantifier instantiation [*]\n"
"  --cond-var-split-agg-quant\n"
"                         aggressive split quantified formulas that lead to\n"
"                         variable eliminations [*]\n"
"  --cond-var-split-quant split quantified formulas that lead to variable\n"
"                         eliminations [*]\n"
"  --conjecture-filter-active-terms\n"
"                         filter based on active terms [*]\n"
"  --conjecture-filter-canonical\n"
"                         filter based on canonicity [*]\n"
"  --conjecture-filter-model\n"
"                         filter based on model [*]\n"
"  --conjecture-gen       generate candidate conjectures for inductive proofs [*]\n"
"  --conjecture-gen-gt-enum=N\n"
"                         number of ground terms to generate for model filtering\n"
"  --conjecture-gen-max-depth=N\n"
"                         maximum depth of terms to consider for conjectures\n"
"  --conjecture-gen-per-round=N\n"
"                         number of conjectures to generate per instantiation\n"
"                         round\n"
"  --conjecture-gen-uee-intro\n"
"                         more aggressive merging for universal equality engine,\n"
"                         introduces terms [*]\n"
"  --conjecture-no-filter do not filter conjectures [*]\n"
"  --dt-stc-ind           apply strengthening for existential quantification over\n"
"                         datatypes based on structural induction [*]\n"
"  --dt-var-exp-quant     expand datatype variables bound to one constructor in\n"
"                         quantifiers [*]\n"
"  --e-matching           whether to do heuristic E-matching [*]\n"
"  --elim-ext-arith-quant eliminate extended arithmetic symbols in quantified\n"
"                         formulas [*]\n"
"  --elim-taut-quant      eliminate tautological disjuncts of quantified formulas\n"
"                         [*]\n"
"  --ext-rewrite-quant    apply extended rewriting to bodies of quantified\n"
"                         formulas [*]\n"
"  --finite-model-find    use finite model finding heuristic for quantifier\n"
"                         instantiation [*]\n"
"  --fmf-bound            finite model finding on bounded quantification [*]\n"
"  --fmf-bound-int        finite model finding on bounded integer quantification\n"
"                         [*]\n"
"  --fmf-bound-lazy       enforce bounds for bounded quantification lazily via\n"
"                         use of proxy variables [*]\n"
"  --fmf-fmc-simple       simple models in full model check for finite model\n"
"                         finding [*]\n"
"  --fmf-fresh-dc         use fresh distinguished representative when applying\n"
"                         Inst-Gen techniques [*]\n"
"  --fmf-fun              find models for recursively defined functions, assumes\n"
"                         functions are admissible [*]\n"
"  --fmf-fun-rlv          find models for recursively defined functions, assumes\n"
"                         functions are admissible, allows empty type when\n"
"                         function is irrelevant [*]\n"
"  --fmf-inst-engine      use instantiation engine in conjunction with finite\n"
"                         model finding [*]\n"
"  --fmf-type-completion-thresh=N\n"
"                         the maximum cardinality of an interpreted type for\n"
"                         which exhaustive enumeration in finite model finding is\n"
"                         attempted\n"
"  --fs-interleave        interleave enumerative instantiation with other\n"
"                         techniques [*]\n"
"  --fs-stratify          stratify effort levels in enumerative instantiation,\n"
"                         which favors speed over fairness [*]\n"
"  --full-saturate-quant  enumerative instantiation: instantiate with ground\n"
"                         terms from relevant domain, then arbitrary ground terms\n"
"                         before answering unknown [*]\n"
"  --full-saturate-quant-limit=N\n"
"                         maximum number of rounds of enumerative instantiation\n"
"                         to apply (-1 means no limit)\n"
"  --full-saturate-quant-rd\n"
"                         whether to use relevant domain first for enumerative\n"
"                         instantiation strategy [*]\n"
"  --global-negate        do global negation of input formula [*]\n"
"  --ho-elim              eagerly eliminate higher-order constraints [*]\n"
"  --ho-elim-store-ax     use store axiom during ho-elim [*]\n"
"  --ho-matching          do higher-order matching algorithm for triggers with\n"
"                         variable operators [*]\n"
"  --ho-matching-var-priority\n"
"                         give priority to variable arguments over constant\n"
"                         arguments [*]\n"
"  --ho-merge-term-db     merge term indices modulo equality [*]\n"
"  --increment-triggers   generate additional triggers as needed during search\n"
"                         [*]\n"
"  --inst-level-input-only\n"
"                         only input terms are assigned instantiation level zero\n"
"                         [*]\n"
"  --inst-max-level=N     maximum inst level of terms used to instantiate\n"
"                         quantified formulas with (-1 == no limit, default)\n"
"  --inst-no-entail       do not consider instances of quantified formulas that\n"
"                         are currently entailed [*]\n"
"  --inst-no-model-true   do not consider instances of quantified formulas that\n"
"                         are currently true in model, if it is available [*]\n"
"  --inst-when-phase=N    instantiation rounds quantifiers takes (>=1) before\n"
"                         allowing theory combination to happen\n"
"  --inst-when-strict-interleave\n"
"                         ensure theory combination and standard quantifier\n"
"                         effort strategies take turns [*]\n"
"  --inst-when-tc-first   allow theory combination to happen once initially,\n"
"                         before quantifier strategies are run [*]\n"
"  --inst-when=MODE       when to apply instantiation\n"
"  --int-wf-ind           apply strengthening for integers based on well-founded\n"
"                         induction [*]\n"
"  --ite-dtt-split-quant  split ites with dt testers as conditions [*]\n"
"  --ite-lift-quant=MODE  ite lifting mode for quantified formulas\n"
"  --literal-matching=MODE\n"
"                         choose literal matching mode\n"
"  --lte-restrict-inst-closure\n"
"                         treat arguments of inst closure as restricted terms for\n"
"                         instantiation [*]\n"
"  --macros-quant         perform quantifiers macro expansion [*]\n"
"  --macros-quant-mode=MODE\n"
"                         mode for quantifiers macro expansion\n"
"  --mbqi-interleave      interleave model-based quantifier instantiation with\n"
"                         other techniques [*]\n"
"  --mbqi-one-inst-per-round\n"
"                         only add one instantiation per quantifier per round for\n"
"                         mbqi [*]\n"
"  --mbqi=MODE            choose mode for model-based quantifier instantiation\n"
"  --miniscope-quant      miniscope quantifiers [*]\n"
"  --miniscope-quant-fv   miniscope quantifiers for ground subformulas [*]\n"
"  --multi-trigger-cache  caching version of multi triggers [*]\n"
"  --multi-trigger-linear implementation of multi triggers where maximum number\n"
"                         of instantiations is linear wrt number of ground terms\n"
"                         [*]\n"
"  --multi-trigger-priority\n"
"                         only try multi triggers if single triggers give no\n"
"                         instantiations [*]\n"
"  --multi-trigger-when-single\n"
"                         select multi triggers when single triggers exist [*]\n"
"  --partial-triggers     use triggers that do not contain all free variables [*]\n"
"  --pre-skolem-quant     apply skolemization eagerly to bodies of quantified\n"
"                         formulas [*]\n"
"  --pre-skolem-quant-agg apply skolemization to quantified formulas aggressively\n"
"                         [*]\n"
"  --pre-skolem-quant-nested\n"
"                         apply skolemization to nested quantified formulas [*]\n"
"  --prenex-quant-user    prenex quantified formulas with user patterns [*]\n"
"  --prenex-quant=MODE    prenex mode for quantified formulas\n"
"  --pure-th-triggers     use pure theory terms as single triggers [*]\n"
"  --purify-dt-triggers   purify dt triggers, match all constructors of correct\n"
"                         form instead of selectors [*]\n"
"  --purify-triggers      purify triggers, e.g. f( x+1 ) becomes f( y ), x mapsto\n"
"                         y-1 [*]\n"
"  --qcf-all-conflict     add all available conflicting instances during\n"
"                         conflict-based instantiation [*]\n"
"  --qcf-eager-check-rd   optimization, eagerly check relevant domain of matched\n"
"                         position [*]\n"
"  --qcf-eager-test       optimization, test qcf instances eagerly [*]\n"
"  --qcf-nested-conflict  consider conflicts for nested quantifiers [*]\n"
"  --qcf-skip-rd          optimization, skip instances based on possibly\n"
"                         irrelevant portions of quantified formulas [*]\n"
"  --qcf-tconstraint      enable entailment checks for t-constraints in qcf\n"
"                         algorithm [*]\n"
"  --qcf-vo-exp           qcf experimental variable ordering [*]\n"
"  --quant-alpha-equiv    infer alpha equivalence between quantified formulas [*]\n"
"  --quant-anti-skolem    perform anti-skolemization for quantified formulas [*]\n"
"  --quant-cf             enable conflict find mechanism for quantifiers [*]\n"
"  --quant-cf-mode=MODE   what effort to apply conflict find mechanism\n"
"  --quant-cf-when=MODE   when to invoke conflict find mechanism for quantifiers\n"
"  --quant-dsplit-mode=MODE\n"
"                         mode for dynamic quantifiers splitting\n"
"  --quant-epr            infer whether in effectively propositional fragment,\n"
"                         use for cegqi [*]\n"
"  --quant-epr-match      use matching heuristics for EPR instantiation [*]\n"
"  --quant-fun-wd         assume that function defined by quantifiers are well\n"
"                         defined [*]\n"
"  --quant-ind            use all available techniques for inductive reasoning\n"
"                         [*]\n"
"  --quant-model-ee       use equality engine of model for last call effort [*]\n"
"  --quant-rep-mode=MODE  selection mode for representatives in quantifiers\n"
"                         engine\n"
"  --quant-split          apply splitting to quantified formulas based on\n"
"                         variable disjoint disjuncts [*]\n"
"  --register-quant-body-terms\n"
"                         consider ground terms within bodies of quantified\n"
"                         formulas for matching [*]\n"
"  --relational-triggers  choose relational triggers such as x = f(y), x >= f(y)\n"
"                         [*]\n"
"  --relevant-triggers    prefer triggers that are more relevant based on SInE\n"
"                         style analysis [*]\n"
"  --strict-triggers      only instantiate quantifiers with user patterns based\n"
"                         on triggers [*]\n"
"  --sygus                use sygus solver (default is true for sygus inputs) [*]\n"
"  --sygus-active-gen-cfactor=N\n"
"                         the branching factor for the number of interpreted\n"
"                         constants to consider for each size when using\n"
"                         --sygus-active-gen=enum\n"
"  --sygus-active-gen=MODE\n"
"                         mode for actively-generated sygus enumerators\n"
"  --sygus-add-const-grammar\n"
"                         statically add constants appearing in conjecture to\n"
"                         grammars [*]\n"
"  --sygus-arg-relevant   static inference techniques for computing whether\n"
"                         arguments of functions-to-synthesize are relevant [*]\n"
"  --sygus-auto-unfold    enable approach which automatically unfolds transition\n"
"                         systems for directly solving invariant synthesis\n"
"                         problems [*]\n"
"  --sygus-bool-ite-return-const\n"
"                         Only use Boolean constants for return values in\n"
"                         unification-based function synthesis [*]\n"
"  --sygus-core-connective\n"
"                         use unsat core analysis to construct Boolean connective\n"
"                         to sygus conjectures [*]\n"
"  --sygus-crepair-abort  abort if constant repair techniques are not applicable\n"
"                         [*]\n"
"  --sygus-eval-opt       use optimized approach for evaluation in sygus [*]\n"
"  --sygus-eval-unfold    do unfolding of sygus evaluation functions [*]\n"
"  --sygus-eval-unfold-bool\n"
"                         do unfolding of Boolean evaluation functions that\n"
"                         appear in refinement lemmas [*]\n"
"  --sygus-expr-miner-check-timeout=N\n"
"                         timeout (in milliseconds) for satisfiability checks in\n"
"                         expression miners\n"
"  --sygus-expr-miner-check-use-export\n"
"                         export expressions to a different ExprManager with\n"
"                         different options for satisfiability checks in\n"
"                         expression miners (EXPERTS only) [*]\n"
"  --sygus-ext-rew        use extended rewriter for sygus [*]\n"
"  --sygus-filter-sol-rev compute backwards filtering to compute whether previous\n"
"                         solutions are filtered based on later ones (EXPERTS\n"
"                         only) [*]\n"
"  --sygus-filter-sol=MODE\n"
"                         mode for filtering sygus solutions\n"
"  --sygus-grammar-cons=MODE\n"
"                         mode for SyGuS grammar construction\n"
"  --sygus-grammar-norm   statically normalize sygus grammars based on flattening\n"
"                         (linearization) [*]\n"
"  --sygus-inference      attempt to preprocess arbitrary inputs to sygus\n"
"                         conjectures [*]\n"
"  --sygus-inst           Enable SyGuS instantiation quantifiers module [*]\n"
"  --sygus-inv-templ-when-sg\n"
"                         use invariant templates (with solution reconstruction)\n"
"                         for syntax guided problems [*]\n"
"  --sygus-inv-templ=MODE template mode for sygus invariant synthesis (weaken\n"
"                         pre-condition, strengthen post-condition, or none)\n"
"  --sygus-min-grammar    statically minimize sygus grammars [*]\n"
"  --sygus-pbe            enable approach which unifies conditional solutions,\n"
"                         specialized for programming-by-examples (pbe)\n"
"                         conjectures [*]\n"
"  --sygus-pbe-multi-fair when using multiple enumerators, ensure that we only\n"
"                         register value of minimial term size [*]\n"
"  --sygus-pbe-multi-fair-diff=N\n"
"                         when using multiple enumerators, ensure that we only\n"
"                         register values of minimial term size plus this value\n"
"                         (default 0)\n"
"  --sygus-qe-preproc     use quantifier elimination as a preprocessing step for\n"
"                         sygus [*]\n"
"  --sygus-query-gen      use sygus to enumerate interesting satisfiability\n"
"                         queries [*]\n"
"  --sygus-query-gen-check\n"
"                         use interesting satisfiability queries to check\n"
"                         soundness of CVC4 [*]\n"
"  --sygus-query-gen-dump-files=MODE\n"
"                         mode for dumping external files corresponding to\n"
"                         interesting satisfiability queries with sygus-query-gen\n"
"  --sygus-query-gen-thresh=N\n"
"                         number of points that we allow to be equal for\n"
"                         enumerating satisfiable queries with sygus-query-gen\n"
"  --sygus-rec-fun        enable efficient support for recursive functions in\n"
"                         sygus grammars [*]\n"
"  --sygus-rec-fun-eval-limit=N\n"
"                         use a hard limit for how many times in a given\n"
"                         evaluator call a recursive function can be evaluated\n"
"                         (so infinite loops can be avoided)\n"
"  --sygus-repair-const   use approach to repair constants in sygus candidate\n"
"                         solutions [*]\n"
"  --sygus-repair-const-timeout=N\n"
"                         timeout (in milliseconds) for the satisfiability check\n"
"                         to repair constants in sygus candidate solutions\n"
"  --sygus-rr             use sygus to enumerate and verify correctness of\n"
"                         rewrite rules [*]\n"
"  --sygus-rr-synth       use sygus to enumerate candidate rewrite rules [*]\n"
"  --sygus-rr-synth-accel add dynamic symmetry breaking clauses based on\n"
"                         candidate rewrites [*]\n"
"  --sygus-rr-synth-check use satisfiability check to verify correctness of\n"
"                         candidate rewrites [*]\n"
"  --sygus-rr-synth-filter-cong\n"
"                         filter candidate rewrites based on congruence [*]\n"
"  --sygus-rr-synth-filter-match\n"
"                         filter candidate rewrites based on matching [*]\n"
"  --sygus-rr-synth-filter-nl\n"
"                         filter non-linear candidate rewrites [*]\n"
"  --sygus-rr-synth-filter-order\n"
"                         filter candidate rewrites based on variable ordering\n"
"                         [*]\n"
"  --sygus-rr-synth-input synthesize rewrite rules based on the input formula [*]\n"
"  --sygus-rr-synth-input-nvars=N\n"
"                         the maximum number of variables per type that appear in\n"
"                         rewrites from sygus-rr-synth-input\n"
"  --sygus-rr-synth-input-use-bool\n"
"                         synthesize Boolean rewrite rules based on the input\n"
"                         formula [*]\n"
"  --sygus-rr-synth-rec   synthesize rewrite rules over all sygus grammar types\n"
"                         recursively [*]\n"
"  --sygus-rr-verify      use sygus to verify the correctness of rewrite rules\n"
"                         via sampling [*]\n"
"  --sygus-rr-verify-abort\n"
"                         abort when sygus-rr-verify finds an instance of\n"
"                         unsoundness [*]\n"
"  --sygus-sample-fp-uniform\n"
"                         sample floating-point values uniformly instead of in a\n"
"                         biased fashion [*]\n"
"  --sygus-sample-grammar when applicable, use grammar for choosing sample points\n"
"                         [*]\n"
"  --sygus-samples=N      number of points to consider when doing sygus rewriter\n"
"                         sample testing\n"
"  --sygus-si-abort       abort if synthesis conjecture is not single invocation\n"
"                         [*]\n"
"  --sygus-si-partial     combined techniques for synthesis conjectures that are\n"
"                         partially single invocation [*]\n"
"  --sygus-si-rcons-limit=N\n"
"                         number of rounds of enumeration to use during solution\n"
"                         reconstruction (negative means unlimited)\n"
"  --sygus-si-rcons=MODE  policy for reconstructing solutions for single\n"
"                         invocation conjectures\n"
"  --sygus-si-reconstruct-const\n"
"                         include constants when reconstruct solutions for single\n"
"                         invocation conjectures in original grammar [*]\n"
"  --sygus-si=MODE        mode for processing single invocation synthesis\n"
"                         conjectures\n"
"  --sygus-stream         enumerate a stream of solutions instead of terminating\n"
"                         after the first one [*]\n"
"  --sygus-templ-embed-grammar\n"
"                         embed sygus templates into grammars [*]\n"
"  --sygus-unif-cond-independent-no-repeat-sol\n"
"                         Do not try repeated solutions when using independent\n"
"                         synthesis of conditions in unification-based function\n"
"                         synthesis [*]\n"
"  --sygus-unif-pi=MODE   mode for synthesis via piecewise-indepedent unification\n"
"  --sygus-unif-shuffle-cond\n"
"                         Shuffle condition pool when building solutions (may\n"
"                         change solutions sizes) [*]\n"
"  --sygus-verify-subcall use separate copy of the SMT solver for verification\n"
"                         lemmas in sygus [*]\n"
"  --term-db-mode=MODE    which ground terms to consider for instantiation\n"
"  --track-inst-lemmas    track instantiation lemmas (for proofs, unsat cores, qe\n"
"                         and synthesis minimization) [*]\n"
"  --trigger-active-sel=MODE\n"
"                         selection mode to activate triggers\n"
"  --trigger-sel=MODE     selection mode for triggers\n"
"  --user-pat=MODE        policy for handling user-provided patterns for\n"
"                         quantifier instantiation\n"
"  --var-elim-quant       enable simple variable elimination for quantified\n"
"                         formulas [*]\n"
"  --var-ineq-elim-quant  enable variable elimination based on infinite\n"
"                         projection of unbound arithmetic variables [*]\n"
"\nFrom the Sep module:\n"
"  --sep-check-neg        check negated spatial assertions [*]\n"
"  --sep-child-refine     child-specific refinements of negated star, positive\n"
"                         wand [*]\n"
"  --sep-deq-c            assume cardinality elements are distinct [*]\n"
"  --sep-exp              experimental flag for sep [*]\n"
"  --sep-min-refine       only add refinement lemmas for minimal (innermost)\n"
"                         assertions [*]\n"
"  --sep-pre-skolem-emp   eliminate emp constraint at preprocess time [*]\n"
"\nFrom the Sets module:\n"
"  --sets-ext             enable extended symbols such as complement and universe\n"
"                         in theory of sets [*]\n"
"  --sets-infer-as-lemmas send inferences as lemmas [*]\n"
"  --sets-proxy-lemmas    introduce proxy variables eagerly to shorten lemmas [*]\n"
"\nFrom the SMT layer module:\n"
"  --abstract-values      in models, output arrays (and in future, maybe others)\n"
"                         using abstract values, as required by the SMT-LIB\n"
"                         standard [*]\n"
"  --ackermann            eliminate functions by ackermannization [*]\n"
"  --bitblast-step=N      amount of resources spent for each bitblast step\n"
"                         (EXPERTS only)\n"
"  --block-models=MODE    mode for producing several models\n"
"  --bv-eager-assert-step=N\n"
"                         amount of resources spent for each eager BV assert step\n"
"                         (EXPERTS only)\n"
"  --bv-propagation-step=N\n"
"                         amount of resources spent for each BV propagation step\n"
"                         (EXPERTS only)\n"
"  --bv-sat-conflict-step=N\n"
"                         amount of resources spent for each sat conflict\n"
"                         (bitvectors) (EXPERTS only)\n"
"  --check-abducts        checks whether produced solutions to get-abduct are\n"
"                         correct [*]\n"
"  --check-models         after SAT/INVALID/UNKNOWN, check that the generated\n"
"                         model satisfies user assertions [*]\n"
"  --check-proofs         after UNSAT/VALID, machine-check the generated proof\n"
"                         [*]\n"
"  --check-synth-sol      checks whether produced solutions to\n"
"                         functions-to-synthesize satisfy the conjecture [*]\n"
"  --check-unsat-cores    after UNSAT/VALID, produce and check an unsat core\n"
"                         (expensive) [*]\n"
"  --cnf-step=N           amount of resources spent for each call to cnf\n"
"                         conversion (EXPERTS only)\n"
"  --debug-check-models   after SAT/INVALID/UNKNOWN, check that the generated\n"
"                         model satisfies user and internal assertions [*]\n"
"  --decision-step=N      amount of getNext decision calls in the decision engine\n"
"                         (EXPERTS only)\n"
"  --dump-instantiations  output instantiations of quantified formulas after\n"
"                         every UNSAT/VALID response [*]\n"
"  --dump-models          output models after every SAT/INVALID/UNKNOWN response\n"
"                         [*]\n"
"  --dump-proofs          output proofs after every UNSAT/VALID response [*]\n"
"  --dump-synth           output solution for synthesis conjectures after every\n"
"                         UNSAT/VALID response [*]\n"
"  --dump-unsat-cores     output unsat cores after every UNSAT/VALID response [*]\n"
"  --dump-unsat-cores-full\n"
"                         dump the full unsat core, including unlabeled\n"
"                         assertions [*]\n"
"  --ext-rew-prep         use extended rewriter as a preprocessing pass [*]\n"
"  --ext-rew-prep-agg     use aggressive extended rewriter as a preprocessing\n"
"                         pass [*]\n"
"  --force-no-limit-cpu-while-dump\n"
"                         Force no CPU limit when dumping models and proofs [*]\n"
"  --ite-simp             turn on ite simplification (Kim (and Somenzi) et al.,\n"
"                         SAT 2009) [*]\n"
"  --lemma-step=N         amount of resources spent when adding lemmas (EXPERTS\n"
"                         only)\n"
"  --model-cores=MODE     mode for producing model cores\n"
"  --model-u-dt-enum      in models, output uninterpreted sorts as datatype\n"
"                         enumerations [*]\n"
"  --model-witness-value  in models, use a witness constant for choice functions\n"
"                         [*]\n"
"  --on-repeat-ite-simp   do the ite simplification pass again if repeating\n"
"                         simplification [*]\n"
"  --parse-step=N         amount of resources spent for each command/expression\n"
"                         parsing (EXPERTS only)\n"
"  --preprocess-step=N    amount of resources spent for each preprocessing step\n"
"                         in SmtEngine (EXPERTS only)\n"
"  --produce-assignments  support the get-assignment command [*]\n"
"  --produce-unsat-assumptions\n"
"                         turn on unsat assumptions generation [*]\n"
"  --produce-unsat-cores  turn on unsat core generation [*]\n"
"  --proof                turn on proof generation [*]\n"
"  --quantifier-step=N    amount of resources spent for quantifier instantiations\n"
"                         (EXPERTS only)\n"
"  --repeat-simp          make multiple passes with nonclausal simplifier [*]\n"
"  --restart-step=N       amount of resources spent for each theory restart\n"
"                         (EXPERTS only)\n"
"  --rewrite-step=N       amount of resources spent for each rewrite step\n"
"                         (EXPERTS only)\n"
"  --sat-conflict-step=N  amount of resources spent for each sat conflict (main\n"
"                         sat solver) (EXPERTS only)\n"
"  --simp-ite-compress    enables compressing ites after ite simplification [*]\n"
"  --simp-ite-hunt-zombies=N\n"
"                         post ite compression enables zombie removal while the\n"
"                         number of nodes is above this threshold\n"
"  --simp-with-care       enables simplifyWithCare in ite simplificiation [*]\n"
"  --simplification=MODE  choose simplification mode, see --simplification=help\n"
"  --sort-inference       calculate sort inference of input problem, convert the\n"
"                         input based on monotonic sorts [*]\n"
"  --static-learning      use static learning (on by default) [*]\n"
"  --sygus-out=MODE       output mode for sygus\n"
"  --sygus-print-callbacks\n"
"                         use sygus print callbacks to print sygus terms in the\n"
"                         user-provided form (disable for debugging) [*]\n"
"  --theory-check-step=N  amount of resources spent for each theory check call\n"
"                         (EXPERTS only)\n"
"  --unconstrained-simp   turn on unconstrained simplification (see\n"
"                         Bruttomesso/Brummayer PhD thesis). Fully supported only\n"
"                         in (subsets of) the logic QF_ABV. [*]\n"
"\nFrom the Strings theory module:\n"
"  --re-elim              elimination techniques for regular expressions [*]\n"
"  --re-elim-agg          aggressive elimination techniques for regular\n"
"                         expressions [*]\n"
"  --re-inter-mode=MODE   determines which regular expressions intersections to\n"
"                         compute (EXPERTS only)\n"
"  --strings-check-entail-len\n"
"                         check entailment between length terms to reduce\n"
"                         splitting [*]\n"
"  --strings-eager        strings eager check [*]\n"
"  --strings-eager-len    strings eager length lemmas [*]\n"
"  --strings-exp          experimental features in the theory of strings [*]\n"
"  --strings-ff           do flat form inferences [*]\n"
"  --strings-fmf          the finite model finding used by the theory of strings\n"
"                         [*]\n"
"  --strings-guess-model  use model guessing to avoid string extended function\n"
"                         reductions [*]\n"
"  --strings-infer-as-lemmas\n"
"                         always send lemmas out instead of making internal\n"
"                         inferences [*]\n"
"  --strings-infer-sym    strings split on empty string [*]\n"
"  --strings-inm          internal for strings: ignore negative membership\n"
"                         constraints (fragment checking is needed, left to users\n"
"                         for now) [*]\n"
"  --strings-lazy-pp      perform string preprocessing lazily [*]\n"
"  --strings-len-norm     strings length normalization lemma [*]\n"
"  --strings-lprop-csp    do length propagation based on constant splits [*]\n"
"  --strings-min-prefix-explain\n"
"                         minimize explanations for prefix of normal forms in\n"
"                         strings [*]\n"
"  --strings-print-ascii  the alphabet contains only printable characters from\n"
"                         the standard extended ASCII [*]\n"
"  --strings-process-loop-mode=MODE\n"
"                         determines how to process looping string equations\n"
"                         (EXPERTS only)\n"
"  --strings-rexplain-lemmas\n"
"                         regression explanations for string lemmas [*]\n"
"\nFrom the Theory layer module:\n"
"  --assign-function-values\n"
"                         assign values for uninterpreted functions in models [*]\n"
"  --condense-function-values\n"
"                         condense values for functions in models rather than\n"
"                         explicitly representing them [*]\n"
"  --theoryof-mode=MODE   mode for Theory::theoryof() (EXPERTS only)\n"
"\nFrom the Uninterpreted functions theory module:\n"
"  --symmetry-breaker     use UF symmetry breaker (Deharbe et al., CADE 2011) [*]\n"
"  --uf-ho                enable support for higher-order reasoning [*]\n"
"  --uf-ho-ext            apply extensionality on function symbols [*]\n"
"  --uf-ss-abort-card=N   tells the uf with cardinality to only consider models\n"
"                         that interpret uninterpreted sorts of cardinality at\n"
"                         most N (-1 == no limit, default)\n"
"  --uf-ss-fair           use fair strategy for finite model finding multiple\n"
"                         sorts [*]\n"
"  --uf-ss-fair-monotone  group monotone sorts when enforcing fairness for finite\n"
"                         model finding [*]\n"
"  --uf-ss-totality       always use totality axioms for enforcing cardinality\n"
"                         constraints [*]\n"
"  --uf-ss-totality-limited=N\n"
"                         apply totality axioms, but only up to cardinality N (-1\n"
"                         == do not apply totality axioms, default)\n"
"  --uf-ss-totality-sym-break\n"
"                         apply symmetry breaking for totality axioms [*]\n"
"  --uf-ss=MODE           mode of operation for uf with cardinality solver.\n";


static const std::string optionsFootnote = "\n\
[*] Each of these options has a --no-OPTIONNAME variant, which reverses the\n\
    sense of the option.\n\
";

static const std::string languageDescription =
    "\
Languages currently supported as arguments to the -L / --lang option:\n\
  auto                           attempt to automatically determine language\n\
  cvc4 | presentation | pl       CVC4 presentation language\n\
  smt | smtlib | smt2 |\n\
  smt2.0 | smtlib2 | smtlib2.0   SMT-LIB format 2.0\n\
  smt2.5 | smtlib2.5             SMT-LIB format 2.5\n\
  smt2.6 | smtlib2.6             SMT-LIB format 2.6 with support for the strings standard\n\
  tptp                           TPTP format (cnf, fof and tff)\n\
  sygus1                         SyGuS version 1.0 \n\
  sygus | sygus2                 SyGuS version 2.0\n\
\n\
Languages currently supported as arguments to the --output-lang option:\n\
  auto                           match output language to input language\n\
  cvc4 | presentation | pl       CVC4 presentation language\n\
  cvc3                           CVC3 presentation language\n\
  smt | smtlib | smt2 |\n\
  smt2.0 | smtlib2.0 | smtlib2   SMT-LIB format 2.0\n\
  smt2.5 | smtlib2.5             SMT-LIB format 2.5\n\
  smt2.6 | smtlib2.6             SMT-LIB format 2.6 with support for the strings standard\n\
  tptp                           TPTP format\n\
  z3str                          SMT-LIB 2.0 with Z3-str string constraints\n\
  ast                            internal format (simple syntax trees)\n\
";

std::string Options::getDescription() const {
  return optionsDescription;
}

void Options::printUsage(const std::string msg, std::ostream& out) {
  out << msg << optionsDescription << std::endl
      << optionsFootnote << std::endl << std::flush;
}

void Options::printShortUsage(const std::string msg, std::ostream& out) {
  out << msg << mostCommonOptionsDescription << std::endl
      << optionsFootnote << std::endl
      << "For full usage, please use --help."
      << std::endl << std::endl << std::flush;
}

void Options::printLanguageHelp(std::ostream& out) {
  out << languageDescription << std::flush;
}

/**
 * This is a table of long options.  By policy, each short option
 * should have an equivalent long option (but the reverse isn't the
 * case), so this table should thus contain all command-line options.
 *
 * Each option in this array has four elements:
 *
 * 1. the long option string
 * 2. argument behavior for the option:
 *    no_argument - no argument permitted
 *    required_argument - an argument is expected
 *    optional_argument - an argument is permitted but not required
 * 3. this is a pointer to an int which is set to the 4th entry of the
 *    array if the option is present; or NULL, in which case
 *    getopt_long() returns the 4th entry
 * 4. the return value for getopt_long() when this long option (or the
 *    value to set the 3rd entry to; see #3)
 *
 * If you add something here, you should add it in src/main/usage.h
 * also, to document it.
 *
 * If you add something that has a short option equivalent, you should
 * add it to the getopt_long() call in parseOptions().
 */
static struct option cmdlineOptions[] = {
  { "approx-branch-depth", required_argument, nullptr, 256 },
  { "arith-brab", no_argument, nullptr, 257 },
  { "no-arith-brab", no_argument, nullptr, 258 },
  { "arith-no-partial-fun", no_argument, nullptr, 259 },
  { "no-arith-no-partial-fun", no_argument, nullptr, 260 },
  { "arith-prop-clauses", required_argument, nullptr, 261 },
  { "arith-prop", required_argument, nullptr, 262 },
  { "arith-rewrite-equalities", no_argument, nullptr, 263 },
  { "no-arith-rewrite-equalities", no_argument, nullptr, 264 },
  { "collect-pivot-stats", no_argument, nullptr, 265 },
  { "no-collect-pivot-stats", no_argument, nullptr, 266 },
  { "cut-all-bounded", no_argument, nullptr, 267 },
  { "no-cut-all-bounded", no_argument, nullptr, 268 },
  { "dio-decomps", no_argument, nullptr, 269 },
  { "no-dio-decomps", no_argument, nullptr, 270 },
  { "dio-repeat", no_argument, nullptr, 271 },
  { "no-dio-repeat", no_argument, nullptr, 272 },
  { "dio-solver", no_argument, nullptr, 273 },
  { "no-dio-solver", no_argument, nullptr, 274 },
  { "dio-turns", required_argument, nullptr, 275 },
  { "error-selection-rule", required_argument, nullptr, 276 },
  { "fc-penalties", no_argument, nullptr, 277 },
  { "no-fc-penalties", no_argument, nullptr, 278 },
  { "heuristic-pivots", required_argument, nullptr, 279 },
  { "lemmas-on-replay-failure", no_argument, nullptr, 280 },
  { "no-lemmas-on-replay-failure", no_argument, nullptr, 281 },
  { "maxCutsInContext", required_argument, nullptr, 282 },
  { "miplib-trick", no_argument, nullptr, 283 },
  { "no-miplib-trick", no_argument, nullptr, 284 },
  { "miplib-trick-subs", required_argument, nullptr, 285 },
  { "new-prop", no_argument, nullptr, 286 },
  { "no-new-prop", no_argument, nullptr, 287 },
  { "nl-ext", no_argument, nullptr, 288 },
  { "no-nl-ext", no_argument, nullptr, 289 },
  { "nl-ext-ent-conf", no_argument, nullptr, 290 },
  { "no-nl-ext-ent-conf", no_argument, nullptr, 291 },
  { "nl-ext-factor", no_argument, nullptr, 292 },
  { "no-nl-ext-factor", no_argument, nullptr, 293 },
  { "nl-ext-inc-prec", no_argument, nullptr, 294 },
  { "no-nl-ext-inc-prec", no_argument, nullptr, 295 },
  { "nl-ext-purify", no_argument, nullptr, 296 },
  { "no-nl-ext-purify", no_argument, nullptr, 297 },
  { "nl-ext-rbound", no_argument, nullptr, 298 },
  { "no-nl-ext-rbound", no_argument, nullptr, 299 },
  { "nl-ext-rewrite", no_argument, nullptr, 300 },
  { "no-nl-ext-rewrite", no_argument, nullptr, 301 },
  { "nl-ext-split-zero", no_argument, nullptr, 302 },
  { "no-nl-ext-split-zero", no_argument, nullptr, 303 },
  { "nl-ext-tf-taylor-deg", required_argument, nullptr, 304 },
  { "nl-ext-tf-tplanes", no_argument, nullptr, 305 },
  { "no-nl-ext-tf-tplanes", no_argument, nullptr, 306 },
  { "nl-ext-tplanes", no_argument, nullptr, 307 },
  { "no-nl-ext-tplanes", no_argument, nullptr, 308 },
  { "nl-ext-tplanes-interleave", no_argument, nullptr, 309 },
  { "no-nl-ext-tplanes-interleave", no_argument, nullptr, 310 },
  { "pb-rewrites", no_argument, nullptr, 311 },
  { "no-pb-rewrites", no_argument, nullptr, 312 },
  { "pivot-threshold", required_argument, nullptr, 313 },
  { "pp-assert-max-sub-size", required_argument, nullptr, 314 },
  { "prop-row-length", required_argument, nullptr, 315 },
  { "replay-early-close-depth", required_argument, nullptr, 316 },
  { "replay-failure-penalty", required_argument, nullptr, 317 },
  { "replay-lemma-reject-cut", required_argument, nullptr, 318 },
  { "replay-num-err-penalty", required_argument, nullptr, 319 },
  { "replay-reject-cut", required_argument, nullptr, 320 },
  { "replay-soi-major-threshold-pen", required_argument, nullptr, 321 },
  { "replay-soi-major-threshold", required_argument, nullptr, 322 },
  { "replay-soi-minor-threshold-pen", required_argument, nullptr, 323 },
  { "replay-soi-minor-threshold", required_argument, nullptr, 324 },
  { "restrict-pivots", no_argument, nullptr, 325 },
  { "no-restrict-pivots", no_argument, nullptr, 326 },
  { "revert-arith-models-on-unsat", no_argument, nullptr, 327 },
  { "no-revert-arith-models-on-unsat", no_argument, nullptr, 328 },
  { "rr-turns", required_argument, nullptr, 329 },
  { "se-solve-int", no_argument, nullptr, 330 },
  { "no-se-solve-int", no_argument, nullptr, 331 },
  { "simplex-check-period", required_argument, nullptr, 332 },
  { "soi-qe", no_argument, nullptr, 333 },
  { "no-soi-qe", no_argument, nullptr, 334 },
  { "standard-effort-variable-order-pivots", required_argument, nullptr, 335 },
  { "unate-lemmas", required_argument, nullptr, 336 },
  { "use-approx", no_argument, nullptr, 337 },
  { "no-use-approx", no_argument, nullptr, 338 },
  { "use-fcsimplex", no_argument, nullptr, 339 },
  { "no-use-fcsimplex", no_argument, nullptr, 340 },
  { "use-soi", no_argument, nullptr, 341 },
  { "no-use-soi", no_argument, nullptr, 342 },
  { "arrays-config", required_argument, nullptr, 343 },
  { "arrays-eager-index", no_argument, nullptr, 344 },
  { "no-arrays-eager-index", no_argument, nullptr, 345 },
  { "arrays-eager-lemmas", no_argument, nullptr, 346 },
  { "no-arrays-eager-lemmas", no_argument, nullptr, 347 },
  { "arrays-lazy-rintro1", no_argument, nullptr, 348 },
  { "no-arrays-lazy-rintro1", no_argument, nullptr, 349 },
  { "arrays-model-based", no_argument, nullptr, 350 },
  { "no-arrays-model-based", no_argument, nullptr, 351 },
  { "arrays-optimize-linear", no_argument, nullptr, 352 },
  { "no-arrays-optimize-linear", no_argument, nullptr, 353 },
  { "arrays-prop", required_argument, nullptr, 354 },
  { "arrays-reduce-sharing", no_argument, nullptr, 355 },
  { "no-arrays-reduce-sharing", no_argument, nullptr, 356 },
  { "arrays-weak-equiv", no_argument, nullptr, 357 },
  { "no-arrays-weak-equiv", no_argument, nullptr, 358 },
  { "debug", required_argument, nullptr, 359 },
  { "lang", required_argument, nullptr, 360 },
  { "output-lang", required_argument, nullptr, 361 },
  { "parse-only", no_argument, nullptr, 362 },
  { "no-parse-only", no_argument, nullptr, 363 },
  { "preprocess-only", no_argument, nullptr, 364 },
  { "no-preprocess-only", no_argument, nullptr, 365 },
  { "print-success", no_argument, nullptr, 366 },
  { "no-print-success", no_argument, nullptr, 367 },
  { "quiet", no_argument, nullptr, 368 },
  { "stats", no_argument, nullptr, 369 },
  { "no-stats", no_argument, nullptr, 370 },
  { "stats-every-query", no_argument, nullptr, 371 },
  { "no-stats-every-query", no_argument, nullptr, 372 },
  { "stats-hide-zeros", no_argument, nullptr, 373 },
  { "no-stats-hide-zeros", no_argument, nullptr, 374 },
  { "trace", required_argument, nullptr, 375 },
  { "verbose", no_argument, nullptr, 376 },
  { "bitblast-aig", no_argument, nullptr, 377 },
  { "no-bitblast-aig", no_argument, nullptr, 378 },
  { "bitblast", required_argument, nullptr, 379 },
  { "bitwise-eq", no_argument, nullptr, 380 },
  { "no-bitwise-eq", no_argument, nullptr, 381 },
  { "bool-to-bv", required_argument, nullptr, 382 },
  { "bv-abstraction", no_argument, nullptr, 383 },
  { "no-bv-abstraction", no_argument, nullptr, 384 },
  { "bv-aig-simp", required_argument, nullptr, 385 },
  { "bv-alg-extf", no_argument, nullptr, 386 },
  { "no-bv-alg-extf", no_argument, nullptr, 387 },
  { "bv-algebraic-budget", required_argument, nullptr, 388 },
  { "bv-algebraic-solver", no_argument, nullptr, 389 },
  { "no-bv-algebraic-solver", no_argument, nullptr, 390 },
  { "bv-div-zero-const", no_argument, nullptr, 391 },
  { "no-bv-div-zero-const", no_argument, nullptr, 392 },
  { "bv-eager-explanations", no_argument, nullptr, 393 },
  { "no-bv-eager-explanations", no_argument, nullptr, 394 },
  { "bv-eq-slicer", required_argument, nullptr, 395 },
  { "bv-eq-solver", no_argument, nullptr, 396 },
  { "no-bv-eq-solver", no_argument, nullptr, 397 },
  { "bv-extract-arith", no_argument, nullptr, 398 },
  { "no-bv-extract-arith", no_argument, nullptr, 399 },
  { "bv-gauss-elim", no_argument, nullptr, 400 },
  { "no-bv-gauss-elim", no_argument, nullptr, 401 },
  { "bv-inequality-solver", no_argument, nullptr, 402 },
  { "no-bv-inequality-solver", no_argument, nullptr, 403 },
  { "bv-intro-pow2", no_argument, nullptr, 404 },
  { "no-bv-intro-pow2", no_argument, nullptr, 405 },
  { "bv-lazy-reduce-extf", no_argument, nullptr, 406 },
  { "no-bv-lazy-reduce-extf", no_argument, nullptr, 407 },
  { "bv-lazy-rewrite-extf", no_argument, nullptr, 408 },
  { "no-bv-lazy-rewrite-extf", no_argument, nullptr, 409 },
  { "bv-num-func", required_argument, nullptr, 410 },
  { "bv-optimize-sat-proof", required_argument, nullptr, 411 },
  { "bv-print-consts-as-indexed-symbols", no_argument, nullptr, 412 },
  { "no-bv-print-consts-as-indexed-symbols", no_argument, nullptr, 413 },
  { "bv-proof-format", required_argument, nullptr, 414 },
  { "bv-propagate", no_argument, nullptr, 415 },
  { "no-bv-propagate", no_argument, nullptr, 416 },
  { "bv-quick-xplain", no_argument, nullptr, 417 },
  { "no-bv-quick-xplain", no_argument, nullptr, 418 },
  { "bv-sat-solver", required_argument, nullptr, 419 },
  { "bv-skolemize", no_argument, nullptr, 420 },
  { "no-bv-skolemize", no_argument, nullptr, 421 },
  { "bv-to-bool", no_argument, nullptr, 422 },
  { "no-bv-to-bool", no_argument, nullptr, 423 },
  { "cdt-bisimilar", no_argument, nullptr, 424 },
  { "no-cdt-bisimilar", no_argument, nullptr, 425 },
  { "dt-binary-split", no_argument, nullptr, 426 },
  { "no-dt-binary-split", no_argument, nullptr, 427 },
  { "dt-blast-splits", no_argument, nullptr, 428 },
  { "no-dt-blast-splits", no_argument, nullptr, 429 },
  { "dt-cyclic", no_argument, nullptr, 430 },
  { "no-dt-cyclic", no_argument, nullptr, 431 },
  { "dt-force-assignment", no_argument, nullptr, 432 },
  { "no-dt-force-assignment", no_argument, nullptr, 433 },
  { "dt-infer-as-lemmas", no_argument, nullptr, 434 },
  { "no-dt-infer-as-lemmas", no_argument, nullptr, 435 },
  { "dt-nested-rec", no_argument, nullptr, 436 },
  { "no-dt-nested-rec", no_argument, nullptr, 437 },
  { "dt-rewrite-error-sel", no_argument, nullptr, 438 },
  { "no-dt-rewrite-error-sel", no_argument, nullptr, 439 },
  { "dt-share-sel", no_argument, nullptr, 440 },
  { "no-dt-share-sel", no_argument, nullptr, 441 },
  { "sygus-abort-size", required_argument, nullptr, 442 },
  { "sygus-fair-max", no_argument, nullptr, 443 },
  { "no-sygus-fair-max", no_argument, nullptr, 444 },
  { "sygus-fair", required_argument, nullptr, 445 },
  { "sygus-sym-break", no_argument, nullptr, 446 },
  { "no-sygus-sym-break", no_argument, nullptr, 447 },
  { "sygus-sym-break-agg", no_argument, nullptr, 448 },
  { "no-sygus-sym-break-agg", no_argument, nullptr, 449 },
  { "sygus-sym-break-dynamic", no_argument, nullptr, 450 },
  { "no-sygus-sym-break-dynamic", no_argument, nullptr, 451 },
  { "sygus-sym-break-lazy", no_argument, nullptr, 452 },
  { "no-sygus-sym-break-lazy", no_argument, nullptr, 453 },
  { "sygus-sym-break-pbe", no_argument, nullptr, 454 },
  { "no-sygus-sym-break-pbe", no_argument, nullptr, 455 },
  { "sygus-sym-break-rlv", no_argument, nullptr, 456 },
  { "no-sygus-sym-break-rlv", no_argument, nullptr, 457 },
  { "decision-random-weight", required_argument, nullptr, 458 },
  { "decision-threshold", required_argument, nullptr, 459 },
  { "decision-use-weight", no_argument, nullptr, 460 },
  { "no-decision-use-weight", no_argument, nullptr, 461 },
  { "decision-weight-internal", required_argument, nullptr, 462 },
  { "decision", required_argument, nullptr, 463 },
  { "dag-thresh", required_argument, nullptr, 464 },
  { "expr-depth", required_argument, nullptr, 465 },
  { "print-expr-types", no_argument, nullptr, 466 },
  { "no-print-expr-types", no_argument, nullptr, 467 },
  { "type-checking", no_argument, nullptr, 468 },
  { "no-type-checking", no_argument, nullptr, 469 },
  { "fp-exp", no_argument, nullptr, 470 },
  { "no-fp-exp", no_argument, nullptr, 471 },
  { "copyright", no_argument, nullptr, 472 },
  { "early-exit", no_argument, nullptr, 473 },
  { "no-early-exit", no_argument, nullptr, 474 },
  { "help", no_argument, nullptr, 475 },
  { "interactive", no_argument, nullptr, 476 },
  { "no-interactive", no_argument, nullptr, 477 },
  { "interactive-prompt", no_argument, nullptr, 478 },
  { "no-interactive-prompt", no_argument, nullptr, 479 },
  { "seed", required_argument, nullptr, 480 },
  { "segv-spin", no_argument, nullptr, 481 },
  { "no-segv-spin", no_argument, nullptr, 482 },
  { "show-config", no_argument, nullptr, 483 },
  { "show-debug-tags", no_argument, nullptr, 484 },
  { "show-trace-tags", no_argument, nullptr, 485 },
  { "tear-down-incremental", required_argument, nullptr, 486 },
  { "version", no_argument, nullptr, 487 },
  { "filesystem-access", no_argument, nullptr, 488 },
  { "no-filesystem-access", no_argument, nullptr, 489 },
  { "force-logic", required_argument, nullptr, 490 },
  { "mmap", no_argument, nullptr, 491 },
  { "no-mmap", no_argument, nullptr, 492 },
  { "strict-parsing", no_argument, nullptr, 493 },
  { "no-strict-parsing", no_argument, nullptr, 494 },
  { "flatten-ho-chains", no_argument, nullptr, 495 },
  { "no-flatten-ho-chains", no_argument, nullptr, 496 },
  { "inst-format", required_argument, nullptr, 497 },
  { "model-format", required_argument, nullptr, 498 },
  { "aggressive-core-min", no_argument, nullptr, 499 },
  { "no-aggressive-core-min", no_argument, nullptr, 500 },
  { "allow-empty-dependencies", no_argument, nullptr, 501 },
  { "no-allow-empty-dependencies", no_argument, nullptr, 502 },
  { "fewer-preprocessing-holes", no_argument, nullptr, 503 },
  { "no-fewer-preprocessing-holes", no_argument, nullptr, 504 },
  { "lfsc-letification", no_argument, nullptr, 505 },
  { "no-lfsc-letification", no_argument, nullptr, 506 },
  { "minisat-dump-dimacs", no_argument, nullptr, 507 },
  { "no-minisat-dump-dimacs", no_argument, nullptr, 508 },
  { "minisat-elimination", no_argument, nullptr, 509 },
  { "no-minisat-elimination", no_argument, nullptr, 510 },
  { "random-freq", required_argument, nullptr, 511 },
  { "random-seed", required_argument, nullptr, 512 },
  { "refine-conflicts", no_argument, nullptr, 513 },
  { "no-refine-conflicts", no_argument, nullptr, 514 },
  { "restart-int-base", required_argument, nullptr, 515 },
  { "restart-int-inc", required_argument, nullptr, 516 },
  { "ag-miniscope-quant", no_argument, nullptr, 517 },
  { "no-ag-miniscope-quant", no_argument, nullptr, 518 },
  { "cegis-sample", required_argument, nullptr, 519 },
  { "cegqi", no_argument, nullptr, 520 },
  { "no-cegqi", no_argument, nullptr, 521 },
  { "cegqi-all", no_argument, nullptr, 522 },
  { "no-cegqi-all", no_argument, nullptr, 523 },
  { "cegqi-bv", no_argument, nullptr, 524 },
  { "no-cegqi-bv", no_argument, nullptr, 525 },
  { "cegqi-bv-concat-inv", no_argument, nullptr, 526 },
  { "no-cegqi-bv-concat-inv", no_argument, nullptr, 527 },
  { "cegqi-bv-ineq", required_argument, nullptr, 528 },
  { "cegqi-bv-interleave-value", no_argument, nullptr, 529 },
  { "no-cegqi-bv-interleave-value", no_argument, nullptr, 530 },
  { "cegqi-bv-linear", no_argument, nullptr, 531 },
  { "no-cegqi-bv-linear", no_argument, nullptr, 532 },
  { "cegqi-bv-rm-extract", no_argument, nullptr, 533 },
  { "no-cegqi-bv-rm-extract", no_argument, nullptr, 534 },
  { "cegqi-bv-solve-nl", no_argument, nullptr, 535 },
  { "no-cegqi-bv-solve-nl", no_argument, nullptr, 536 },
  { "cegqi-full", no_argument, nullptr, 537 },
  { "no-cegqi-full", no_argument, nullptr, 538 },
  { "cegqi-innermost", no_argument, nullptr, 539 },
  { "no-cegqi-innermost", no_argument, nullptr, 540 },
  { "cegqi-midpoint", no_argument, nullptr, 541 },
  { "no-cegqi-midpoint", no_argument, nullptr, 542 },
  { "cegqi-min-bounds", no_argument, nullptr, 543 },
  { "no-cegqi-min-bounds", no_argument, nullptr, 544 },
  { "cegqi-model", no_argument, nullptr, 545 },
  { "no-cegqi-model", no_argument, nullptr, 546 },
  { "cegqi-multi-inst", no_argument, nullptr, 547 },
  { "no-cegqi-multi-inst", no_argument, nullptr, 548 },
  { "cegqi-nested-qe", no_argument, nullptr, 549 },
  { "no-cegqi-nested-qe", no_argument, nullptr, 550 },
  { "cegqi-nopt", no_argument, nullptr, 551 },
  { "no-cegqi-nopt", no_argument, nullptr, 552 },
  { "cegqi-prereg-inst", no_argument, nullptr, 553 },
  { "no-cegqi-prereg-inst", no_argument, nullptr, 554 },
  { "cegqi-repeat-lit", no_argument, nullptr, 555 },
  { "no-cegqi-repeat-lit", no_argument, nullptr, 556 },
  { "cegqi-round-up-lia", no_argument, nullptr, 557 },
  { "no-cegqi-round-up-lia", no_argument, nullptr, 558 },
  { "cegqi-sat", no_argument, nullptr, 559 },
  { "no-cegqi-sat", no_argument, nullptr, 560 },
  { "cegqi-use-inf-int", no_argument, nullptr, 561 },
  { "no-cegqi-use-inf-int", no_argument, nullptr, 562 },
  { "cegqi-use-inf-real", no_argument, nullptr, 563 },
  { "no-cegqi-use-inf-real", no_argument, nullptr, 564 },
  { "cond-var-split-agg-quant", no_argument, nullptr, 565 },
  { "no-cond-var-split-agg-quant", no_argument, nullptr, 566 },
  { "cond-var-split-quant", no_argument, nullptr, 567 },
  { "no-cond-var-split-quant", no_argument, nullptr, 568 },
  { "conjecture-filter-active-terms", no_argument, nullptr, 569 },
  { "no-conjecture-filter-active-terms", no_argument, nullptr, 570 },
  { "conjecture-filter-canonical", no_argument, nullptr, 571 },
  { "no-conjecture-filter-canonical", no_argument, nullptr, 572 },
  { "conjecture-filter-model", no_argument, nullptr, 573 },
  { "no-conjecture-filter-model", no_argument, nullptr, 574 },
  { "conjecture-gen", no_argument, nullptr, 575 },
  { "no-conjecture-gen", no_argument, nullptr, 576 },
  { "conjecture-gen-gt-enum", required_argument, nullptr, 577 },
  { "conjecture-gen-max-depth", required_argument, nullptr, 578 },
  { "conjecture-gen-per-round", required_argument, nullptr, 579 },
  { "conjecture-gen-uee-intro", no_argument, nullptr, 580 },
  { "no-conjecture-gen-uee-intro", no_argument, nullptr, 581 },
  { "conjecture-no-filter", no_argument, nullptr, 582 },
  { "no-conjecture-no-filter", no_argument, nullptr, 583 },
  { "dt-stc-ind", no_argument, nullptr, 584 },
  { "no-dt-stc-ind", no_argument, nullptr, 585 },
  { "dt-var-exp-quant", no_argument, nullptr, 586 },
  { "no-dt-var-exp-quant", no_argument, nullptr, 587 },
  { "e-matching", no_argument, nullptr, 588 },
  { "no-e-matching", no_argument, nullptr, 589 },
  { "elim-ext-arith-quant", no_argument, nullptr, 590 },
  { "no-elim-ext-arith-quant", no_argument, nullptr, 591 },
  { "elim-taut-quant", no_argument, nullptr, 592 },
  { "no-elim-taut-quant", no_argument, nullptr, 593 },
  { "ext-rewrite-quant", no_argument, nullptr, 594 },
  { "no-ext-rewrite-quant", no_argument, nullptr, 595 },
  { "finite-model-find", no_argument, nullptr, 596 },
  { "no-finite-model-find", no_argument, nullptr, 597 },
  { "fmf-bound", no_argument, nullptr, 598 },
  { "no-fmf-bound", no_argument, nullptr, 599 },
  { "fmf-bound-int", no_argument, nullptr, 600 },
  { "no-fmf-bound-int", no_argument, nullptr, 601 },
  { "fmf-bound-lazy", no_argument, nullptr, 602 },
  { "no-fmf-bound-lazy", no_argument, nullptr, 603 },
  { "fmf-fmc-simple", no_argument, nullptr, 604 },
  { "no-fmf-fmc-simple", no_argument, nullptr, 605 },
  { "fmf-fresh-dc", no_argument, nullptr, 606 },
  { "no-fmf-fresh-dc", no_argument, nullptr, 607 },
  { "fmf-fun", no_argument, nullptr, 608 },
  { "no-fmf-fun", no_argument, nullptr, 609 },
  { "fmf-fun-rlv", no_argument, nullptr, 610 },
  { "no-fmf-fun-rlv", no_argument, nullptr, 611 },
  { "fmf-inst-engine", no_argument, nullptr, 612 },
  { "no-fmf-inst-engine", no_argument, nullptr, 613 },
  { "fmf-type-completion-thresh", required_argument, nullptr, 614 },
  { "fs-interleave", no_argument, nullptr, 615 },
  { "no-fs-interleave", no_argument, nullptr, 616 },
  { "fs-stratify", no_argument, nullptr, 617 },
  { "no-fs-stratify", no_argument, nullptr, 618 },
  { "full-saturate-quant", no_argument, nullptr, 619 },
  { "no-full-saturate-quant", no_argument, nullptr, 620 },
  { "full-saturate-quant-limit", required_argument, nullptr, 621 },
  { "full-saturate-quant-rd", no_argument, nullptr, 622 },
  { "no-full-saturate-quant-rd", no_argument, nullptr, 623 },
  { "global-negate", no_argument, nullptr, 624 },
  { "no-global-negate", no_argument, nullptr, 625 },
  { "ho-elim", no_argument, nullptr, 626 },
  { "no-ho-elim", no_argument, nullptr, 627 },
  { "ho-elim-store-ax", no_argument, nullptr, 628 },
  { "no-ho-elim-store-ax", no_argument, nullptr, 629 },
  { "ho-matching", no_argument, nullptr, 630 },
  { "no-ho-matching", no_argument, nullptr, 631 },
  { "ho-matching-var-priority", no_argument, nullptr, 632 },
  { "no-ho-matching-var-priority", no_argument, nullptr, 633 },
  { "ho-merge-term-db", no_argument, nullptr, 634 },
  { "no-ho-merge-term-db", no_argument, nullptr, 635 },
  { "increment-triggers", no_argument, nullptr, 636 },
  { "no-increment-triggers", no_argument, nullptr, 637 },
  { "inst-level-input-only", no_argument, nullptr, 638 },
  { "no-inst-level-input-only", no_argument, nullptr, 639 },
  { "inst-max-level", required_argument, nullptr, 640 },
  { "inst-no-entail", no_argument, nullptr, 641 },
  { "no-inst-no-entail", no_argument, nullptr, 642 },
  { "inst-no-model-true", no_argument, nullptr, 643 },
  { "no-inst-no-model-true", no_argument, nullptr, 644 },
  { "inst-when-phase", required_argument, nullptr, 645 },
  { "inst-when-strict-interleave", no_argument, nullptr, 646 },
  { "no-inst-when-strict-interleave", no_argument, nullptr, 647 },
  { "inst-when-tc-first", no_argument, nullptr, 648 },
  { "no-inst-when-tc-first", no_argument, nullptr, 649 },
  { "inst-when", required_argument, nullptr, 650 },
  { "int-wf-ind", no_argument, nullptr, 651 },
  { "no-int-wf-ind", no_argument, nullptr, 652 },
  { "ite-dtt-split-quant", no_argument, nullptr, 653 },
  { "no-ite-dtt-split-quant", no_argument, nullptr, 654 },
  { "ite-lift-quant", required_argument, nullptr, 655 },
  { "literal-matching", required_argument, nullptr, 656 },
  { "lte-restrict-inst-closure", no_argument, nullptr, 657 },
  { "no-lte-restrict-inst-closure", no_argument, nullptr, 658 },
  { "macros-quant", no_argument, nullptr, 659 },
  { "no-macros-quant", no_argument, nullptr, 660 },
  { "macros-quant-mode", required_argument, nullptr, 661 },
  { "mbqi-interleave", no_argument, nullptr, 662 },
  { "no-mbqi-interleave", no_argument, nullptr, 663 },
  { "mbqi-one-inst-per-round", no_argument, nullptr, 664 },
  { "no-mbqi-one-inst-per-round", no_argument, nullptr, 665 },
  { "mbqi", required_argument, nullptr, 666 },
  { "miniscope-quant", no_argument, nullptr, 667 },
  { "no-miniscope-quant", no_argument, nullptr, 668 },
  { "miniscope-quant-fv", no_argument, nullptr, 669 },
  { "no-miniscope-quant-fv", no_argument, nullptr, 670 },
  { "multi-trigger-cache", no_argument, nullptr, 671 },
  { "no-multi-trigger-cache", no_argument, nullptr, 672 },
  { "multi-trigger-linear", no_argument, nullptr, 673 },
  { "no-multi-trigger-linear", no_argument, nullptr, 674 },
  { "multi-trigger-priority", no_argument, nullptr, 675 },
  { "no-multi-trigger-priority", no_argument, nullptr, 676 },
  { "multi-trigger-when-single", no_argument, nullptr, 677 },
  { "no-multi-trigger-when-single", no_argument, nullptr, 678 },
  { "partial-triggers", no_argument, nullptr, 679 },
  { "no-partial-triggers", no_argument, nullptr, 680 },
  { "pre-skolem-quant", no_argument, nullptr, 681 },
  { "no-pre-skolem-quant", no_argument, nullptr, 682 },
  { "pre-skolem-quant-agg", no_argument, nullptr, 683 },
  { "no-pre-skolem-quant-agg", no_argument, nullptr, 684 },
  { "pre-skolem-quant-nested", no_argument, nullptr, 685 },
  { "no-pre-skolem-quant-nested", no_argument, nullptr, 686 },
  { "prenex-quant-user", no_argument, nullptr, 687 },
  { "no-prenex-quant-user", no_argument, nullptr, 688 },
  { "prenex-quant", required_argument, nullptr, 689 },
  { "pure-th-triggers", no_argument, nullptr, 690 },
  { "no-pure-th-triggers", no_argument, nullptr, 691 },
  { "purify-dt-triggers", no_argument, nullptr, 692 },
  { "no-purify-dt-triggers", no_argument, nullptr, 693 },
  { "purify-triggers", no_argument, nullptr, 694 },
  { "no-purify-triggers", no_argument, nullptr, 695 },
  { "qcf-all-conflict", no_argument, nullptr, 696 },
  { "no-qcf-all-conflict", no_argument, nullptr, 697 },
  { "qcf-eager-check-rd", no_argument, nullptr, 698 },
  { "no-qcf-eager-check-rd", no_argument, nullptr, 699 },
  { "qcf-eager-test", no_argument, nullptr, 700 },
  { "no-qcf-eager-test", no_argument, nullptr, 701 },
  { "qcf-nested-conflict", no_argument, nullptr, 702 },
  { "no-qcf-nested-conflict", no_argument, nullptr, 703 },
  { "qcf-skip-rd", no_argument, nullptr, 704 },
  { "no-qcf-skip-rd", no_argument, nullptr, 705 },
  { "qcf-tconstraint", no_argument, nullptr, 706 },
  { "no-qcf-tconstraint", no_argument, nullptr, 707 },
  { "qcf-vo-exp", no_argument, nullptr, 708 },
  { "no-qcf-vo-exp", no_argument, nullptr, 709 },
  { "quant-alpha-equiv", no_argument, nullptr, 710 },
  { "no-quant-alpha-equiv", no_argument, nullptr, 711 },
  { "quant-anti-skolem", no_argument, nullptr, 712 },
  { "no-quant-anti-skolem", no_argument, nullptr, 713 },
  { "quant-cf", no_argument, nullptr, 714 },
  { "no-quant-cf", no_argument, nullptr, 715 },
  { "quant-cf-mode", required_argument, nullptr, 716 },
  { "quant-cf-when", required_argument, nullptr, 717 },
  { "quant-dsplit-mode", required_argument, nullptr, 718 },
  { "quant-epr", no_argument, nullptr, 719 },
  { "no-quant-epr", no_argument, nullptr, 720 },
  { "quant-epr-match", no_argument, nullptr, 721 },
  { "no-quant-epr-match", no_argument, nullptr, 722 },
  { "quant-fun-wd", no_argument, nullptr, 723 },
  { "no-quant-fun-wd", no_argument, nullptr, 724 },
  { "quant-ind", no_argument, nullptr, 725 },
  { "no-quant-ind", no_argument, nullptr, 726 },
  { "quant-model-ee", no_argument, nullptr, 727 },
  { "no-quant-model-ee", no_argument, nullptr, 728 },
  { "quant-rep-mode", required_argument, nullptr, 729 },
  { "quant-split", no_argument, nullptr, 730 },
  { "no-quant-split", no_argument, nullptr, 731 },
  { "register-quant-body-terms", no_argument, nullptr, 732 },
  { "no-register-quant-body-terms", no_argument, nullptr, 733 },
  { "relational-triggers", no_argument, nullptr, 734 },
  { "no-relational-triggers", no_argument, nullptr, 735 },
  { "relevant-triggers", no_argument, nullptr, 736 },
  { "no-relevant-triggers", no_argument, nullptr, 737 },
  { "strict-triggers", no_argument, nullptr, 738 },
  { "no-strict-triggers", no_argument, nullptr, 739 },
  { "sygus", no_argument, nullptr, 740 },
  { "no-sygus", no_argument, nullptr, 741 },
  { "sygus-active-gen-cfactor", required_argument, nullptr, 742 },
  { "sygus-active-gen", required_argument, nullptr, 743 },
  { "sygus-add-const-grammar", no_argument, nullptr, 744 },
  { "no-sygus-add-const-grammar", no_argument, nullptr, 745 },
  { "sygus-arg-relevant", no_argument, nullptr, 746 },
  { "no-sygus-arg-relevant", no_argument, nullptr, 747 },
  { "sygus-auto-unfold", no_argument, nullptr, 748 },
  { "no-sygus-auto-unfold", no_argument, nullptr, 749 },
  { "sygus-bool-ite-return-const", no_argument, nullptr, 750 },
  { "no-sygus-bool-ite-return-const", no_argument, nullptr, 751 },
  { "sygus-core-connective", no_argument, nullptr, 752 },
  { "no-sygus-core-connective", no_argument, nullptr, 753 },
  { "sygus-crepair-abort", no_argument, nullptr, 754 },
  { "no-sygus-crepair-abort", no_argument, nullptr, 755 },
  { "sygus-eval-opt", no_argument, nullptr, 756 },
  { "no-sygus-eval-opt", no_argument, nullptr, 757 },
  { "sygus-eval-unfold", no_argument, nullptr, 758 },
  { "no-sygus-eval-unfold", no_argument, nullptr, 759 },
  { "sygus-eval-unfold-bool", no_argument, nullptr, 760 },
  { "no-sygus-eval-unfold-bool", no_argument, nullptr, 761 },
  { "sygus-expr-miner-check-timeout", required_argument, nullptr, 762 },
  { "sygus-expr-miner-check-use-export", no_argument, nullptr, 763 },
  { "no-sygus-expr-miner-check-use-export", no_argument, nullptr, 764 },
  { "sygus-ext-rew", no_argument, nullptr, 765 },
  { "no-sygus-ext-rew", no_argument, nullptr, 766 },
  { "sygus-filter-sol-rev", no_argument, nullptr, 767 },
  { "no-sygus-filter-sol-rev", no_argument, nullptr, 768 },
  { "sygus-filter-sol", required_argument, nullptr, 769 },
  { "sygus-grammar-cons", required_argument, nullptr, 770 },
  { "sygus-grammar-norm", no_argument, nullptr, 771 },
  { "no-sygus-grammar-norm", no_argument, nullptr, 772 },
  { "sygus-inference", no_argument, nullptr, 773 },
  { "no-sygus-inference", no_argument, nullptr, 774 },
  { "sygus-inst", no_argument, nullptr, 775 },
  { "no-sygus-inst", no_argument, nullptr, 776 },
  { "sygus-inv-templ-when-sg", no_argument, nullptr, 777 },
  { "no-sygus-inv-templ-when-sg", no_argument, nullptr, 778 },
  { "sygus-inv-templ", required_argument, nullptr, 779 },
  { "sygus-min-grammar", no_argument, nullptr, 780 },
  { "no-sygus-min-grammar", no_argument, nullptr, 781 },
  { "sygus-pbe", no_argument, nullptr, 782 },
  { "no-sygus-pbe", no_argument, nullptr, 783 },
  { "sygus-pbe-multi-fair", no_argument, nullptr, 784 },
  { "no-sygus-pbe-multi-fair", no_argument, nullptr, 785 },
  { "sygus-pbe-multi-fair-diff", required_argument, nullptr, 786 },
  { "sygus-qe-preproc", no_argument, nullptr, 787 },
  { "no-sygus-qe-preproc", no_argument, nullptr, 788 },
  { "sygus-query-gen", no_argument, nullptr, 789 },
  { "no-sygus-query-gen", no_argument, nullptr, 790 },
  { "sygus-query-gen-check", no_argument, nullptr, 791 },
  { "no-sygus-query-gen-check", no_argument, nullptr, 792 },
  { "sygus-query-gen-dump-files", required_argument, nullptr, 793 },
  { "sygus-query-gen-thresh", required_argument, nullptr, 794 },
  { "sygus-rec-fun", no_argument, nullptr, 795 },
  { "no-sygus-rec-fun", no_argument, nullptr, 796 },
  { "sygus-rec-fun-eval-limit", required_argument, nullptr, 797 },
  { "sygus-repair-const", no_argument, nullptr, 798 },
  { "no-sygus-repair-const", no_argument, nullptr, 799 },
  { "sygus-repair-const-timeout", required_argument, nullptr, 800 },
  { "sygus-rr", no_argument, nullptr, 801 },
  { "no-sygus-rr", no_argument, nullptr, 802 },
  { "sygus-rr-synth", no_argument, nullptr, 803 },
  { "no-sygus-rr-synth", no_argument, nullptr, 804 },
  { "sygus-rr-synth-accel", no_argument, nullptr, 805 },
  { "no-sygus-rr-synth-accel", no_argument, nullptr, 806 },
  { "sygus-rr-synth-check", no_argument, nullptr, 807 },
  { "no-sygus-rr-synth-check", no_argument, nullptr, 808 },
  { "sygus-rr-synth-filter-cong", no_argument, nullptr, 809 },
  { "no-sygus-rr-synth-filter-cong", no_argument, nullptr, 810 },
  { "sygus-rr-synth-filter-match", no_argument, nullptr, 811 },
  { "no-sygus-rr-synth-filter-match", no_argument, nullptr, 812 },
  { "sygus-rr-synth-filter-nl", no_argument, nullptr, 813 },
  { "no-sygus-rr-synth-filter-nl", no_argument, nullptr, 814 },
  { "sygus-rr-synth-filter-order", no_argument, nullptr, 815 },
  { "no-sygus-rr-synth-filter-order", no_argument, nullptr, 816 },
  { "sygus-rr-synth-input", no_argument, nullptr, 817 },
  { "no-sygus-rr-synth-input", no_argument, nullptr, 818 },
  { "sygus-rr-synth-input-nvars", required_argument, nullptr, 819 },
  { "sygus-rr-synth-input-use-bool", no_argument, nullptr, 820 },
  { "no-sygus-rr-synth-input-use-bool", no_argument, nullptr, 821 },
  { "sygus-rr-synth-rec", no_argument, nullptr, 822 },
  { "no-sygus-rr-synth-rec", no_argument, nullptr, 823 },
  { "sygus-rr-verify", no_argument, nullptr, 824 },
  { "no-sygus-rr-verify", no_argument, nullptr, 825 },
  { "sygus-rr-verify-abort", no_argument, nullptr, 826 },
  { "no-sygus-rr-verify-abort", no_argument, nullptr, 827 },
  { "sygus-sample-fp-uniform", no_argument, nullptr, 828 },
  { "no-sygus-sample-fp-uniform", no_argument, nullptr, 829 },
  { "sygus-sample-grammar", no_argument, nullptr, 830 },
  { "no-sygus-sample-grammar", no_argument, nullptr, 831 },
  { "sygus-samples", required_argument, nullptr, 832 },
  { "sygus-si-abort", no_argument, nullptr, 833 },
  { "no-sygus-si-abort", no_argument, nullptr, 834 },
  { "sygus-si-partial", no_argument, nullptr, 835 },
  { "no-sygus-si-partial", no_argument, nullptr, 836 },
  { "sygus-si-rcons-limit", required_argument, nullptr, 837 },
  { "sygus-si-rcons", required_argument, nullptr, 838 },
  { "sygus-si-reconstruct-const", no_argument, nullptr, 839 },
  { "no-sygus-si-reconstruct-const", no_argument, nullptr, 840 },
  { "sygus-si", required_argument, nullptr, 841 },
  { "sygus-stream", no_argument, nullptr, 842 },
  { "no-sygus-stream", no_argument, nullptr, 843 },
  { "sygus-templ-embed-grammar", no_argument, nullptr, 844 },
  { "no-sygus-templ-embed-grammar", no_argument, nullptr, 845 },
  { "sygus-unif-cond-independent-no-repeat-sol", no_argument, nullptr, 846 },
  { "no-sygus-unif-cond-independent-no-repeat-sol", no_argument, nullptr, 847 },
  { "sygus-unif-pi", required_argument, nullptr, 848 },
  { "sygus-unif-shuffle-cond", no_argument, nullptr, 849 },
  { "no-sygus-unif-shuffle-cond", no_argument, nullptr, 850 },
  { "sygus-verify-subcall", no_argument, nullptr, 851 },
  { "no-sygus-verify-subcall", no_argument, nullptr, 852 },
  { "term-db-mode", required_argument, nullptr, 853 },
  { "track-inst-lemmas", no_argument, nullptr, 854 },
  { "no-track-inst-lemmas", no_argument, nullptr, 855 },
  { "trigger-active-sel", required_argument, nullptr, 856 },
  { "trigger-sel", required_argument, nullptr, 857 },
  { "user-pat", required_argument, nullptr, 858 },
  { "var-elim-quant", no_argument, nullptr, 859 },
  { "no-var-elim-quant", no_argument, nullptr, 860 },
  { "var-ineq-elim-quant", no_argument, nullptr, 861 },
  { "no-var-ineq-elim-quant", no_argument, nullptr, 862 },
  { "sep-check-neg", no_argument, nullptr, 863 },
  { "no-sep-check-neg", no_argument, nullptr, 864 },
  { "sep-child-refine", no_argument, nullptr, 865 },
  { "no-sep-child-refine", no_argument, nullptr, 866 },
  { "sep-deq-c", no_argument, nullptr, 867 },
  { "no-sep-deq-c", no_argument, nullptr, 868 },
  { "sep-exp", no_argument, nullptr, 869 },
  { "no-sep-exp", no_argument, nullptr, 870 },
  { "sep-min-refine", no_argument, nullptr, 871 },
  { "no-sep-min-refine", no_argument, nullptr, 872 },
  { "sep-pre-skolem-emp", no_argument, nullptr, 873 },
  { "no-sep-pre-skolem-emp", no_argument, nullptr, 874 },
  { "sets-ext", no_argument, nullptr, 875 },
  { "no-sets-ext", no_argument, nullptr, 876 },
  { "sets-infer-as-lemmas", no_argument, nullptr, 877 },
  { "no-sets-infer-as-lemmas", no_argument, nullptr, 878 },
  { "sets-proxy-lemmas", no_argument, nullptr, 879 },
  { "no-sets-proxy-lemmas", no_argument, nullptr, 880 },
  { "abstract-values", no_argument, nullptr, 881 },
  { "no-abstract-values", no_argument, nullptr, 882 },
  { "ackermann", no_argument, nullptr, 883 },
  { "no-ackermann", no_argument, nullptr, 884 },
  { "bitblast-step", required_argument, nullptr, 885 },
  { "block-models", required_argument, nullptr, 886 },
  { "bv-eager-assert-step", required_argument, nullptr, 887 },
  { "bv-propagation-step", required_argument, nullptr, 888 },
  { "bv-sat-conflict-step", required_argument, nullptr, 889 },
  { "check-abducts", no_argument, nullptr, 890 },
  { "no-check-abducts", no_argument, nullptr, 891 },
  { "check-models", no_argument, nullptr, 892 },
  { "no-check-models", no_argument, nullptr, 893 },
  { "check-proofs", no_argument, nullptr, 894 },
  { "no-check-proofs", no_argument, nullptr, 895 },
  { "check-synth-sol", no_argument, nullptr, 896 },
  { "no-check-synth-sol", no_argument, nullptr, 897 },
  { "check-unsat-cores", no_argument, nullptr, 898 },
  { "no-check-unsat-cores", no_argument, nullptr, 899 },
  { "cnf-step", required_argument, nullptr, 900 },
  { "cpu-time", no_argument, nullptr, 901 },
  { "no-cpu-time", no_argument, nullptr, 902 },
  { "debug-check-models", no_argument, nullptr, 903 },
  { "no-debug-check-models", no_argument, nullptr, 904 },
  { "decision-step", required_argument, nullptr, 905 },
  { "dump-instantiations", no_argument, nullptr, 906 },
  { "no-dump-instantiations", no_argument, nullptr, 907 },
  { "dump-models", no_argument, nullptr, 908 },
  { "no-dump-models", no_argument, nullptr, 909 },
  { "dump-proofs", no_argument, nullptr, 910 },
  { "no-dump-proofs", no_argument, nullptr, 911 },
  { "dump-synth", no_argument, nullptr, 912 },
  { "no-dump-synth", no_argument, nullptr, 913 },
  { "dump-to", required_argument, nullptr, 914 },
  { "dump-unsat-cores", no_argument, nullptr, 915 },
  { "no-dump-unsat-cores", no_argument, nullptr, 916 },
  { "dump-unsat-cores-full", no_argument, nullptr, 917 },
  { "no-dump-unsat-cores-full", no_argument, nullptr, 918 },
  { "dump", required_argument, nullptr, 919 },
  { "ext-rew-prep", no_argument, nullptr, 920 },
  { "no-ext-rew-prep", no_argument, nullptr, 921 },
  { "ext-rew-prep-agg", no_argument, nullptr, 922 },
  { "no-ext-rew-prep-agg", no_argument, nullptr, 923 },
  { "force-no-limit-cpu-while-dump", no_argument, nullptr, 924 },
  { "no-force-no-limit-cpu-while-dump", no_argument, nullptr, 925 },
  { "hard-limit", no_argument, nullptr, 926 },
  { "no-hard-limit", no_argument, nullptr, 927 },
  { "incremental", no_argument, nullptr, 928 },
  { "no-incremental", no_argument, nullptr, 929 },
  { "ite-simp", no_argument, nullptr, 930 },
  { "no-ite-simp", no_argument, nullptr, 931 },
  { "lemma-step", required_argument, nullptr, 932 },
  { "model-cores", required_argument, nullptr, 933 },
  { "model-u-dt-enum", no_argument, nullptr, 934 },
  { "no-model-u-dt-enum", no_argument, nullptr, 935 },
  { "model-witness-value", no_argument, nullptr, 936 },
  { "no-model-witness-value", no_argument, nullptr, 937 },
  { "on-repeat-ite-simp", no_argument, nullptr, 938 },
  { "no-on-repeat-ite-simp", no_argument, nullptr, 939 },
  { "parse-step", required_argument, nullptr, 940 },
  { "preprocess-step", required_argument, nullptr, 941 },
  { "produce-abducts", no_argument, nullptr, 942 },
  { "no-produce-abducts", no_argument, nullptr, 943 },
  { "produce-assertions", no_argument, nullptr, 944 },
  { "no-produce-assertions", no_argument, nullptr, 945 },
  { "produce-assignments", no_argument, nullptr, 946 },
  { "no-produce-assignments", no_argument, nullptr, 947 },
  { "produce-models", no_argument, nullptr, 948 },
  { "no-produce-models", no_argument, nullptr, 949 },
  { "produce-unsat-assumptions", no_argument, nullptr, 950 },
  { "no-produce-unsat-assumptions", no_argument, nullptr, 951 },
  { "produce-unsat-cores", no_argument, nullptr, 952 },
  { "no-produce-unsat-cores", no_argument, nullptr, 953 },
  { "proof", no_argument, nullptr, 954 },
  { "no-proof", no_argument, nullptr, 955 },
  { "quantifier-step", required_argument, nullptr, 956 },
  { "repeat-simp", no_argument, nullptr, 957 },
  { "no-repeat-simp", no_argument, nullptr, 958 },
  { "restart-step", required_argument, nullptr, 959 },
  { "rewrite-step", required_argument, nullptr, 960 },
  { "rlimit-per", required_argument, nullptr, 961 },
  { "rlimit", required_argument, nullptr, 962 },
  { "sat-conflict-step", required_argument, nullptr, 963 },
  { "simp-ite-compress", no_argument, nullptr, 964 },
  { "no-simp-ite-compress", no_argument, nullptr, 965 },
  { "simp-ite-hunt-zombies", required_argument, nullptr, 966 },
  { "simp-with-care", no_argument, nullptr, 967 },
  { "no-simp-with-care", no_argument, nullptr, 968 },
  { "simplification", required_argument, nullptr, 969 },
  { "solve-bv-as-int", required_argument, nullptr, 970 },
  { "solve-int-as-bv", required_argument, nullptr, 971 },
  { "solve-real-as-int", no_argument, nullptr, 972 },
  { "no-solve-real-as-int", no_argument, nullptr, 973 },
  { "sort-inference", no_argument, nullptr, 974 },
  { "no-sort-inference", no_argument, nullptr, 975 },
  { "static-learning", no_argument, nullptr, 976 },
  { "no-static-learning", no_argument, nullptr, 977 },
  { "sygus-out", required_argument, nullptr, 978 },
  { "sygus-print-callbacks", no_argument, nullptr, 979 },
  { "no-sygus-print-callbacks", no_argument, nullptr, 980 },
  { "theory-check-step", required_argument, nullptr, 981 },
  { "tlimit-per", required_argument, nullptr, 982 },
  { "tlimit", required_argument, nullptr, 983 },
  { "unconstrained-simp", no_argument, nullptr, 984 },
  { "no-unconstrained-simp", no_argument, nullptr, 985 },
  { "re-elim", no_argument, nullptr, 986 },
  { "no-re-elim", no_argument, nullptr, 987 },
  { "re-elim-agg", no_argument, nullptr, 988 },
  { "no-re-elim-agg", no_argument, nullptr, 989 },
  { "re-inter-mode", required_argument, nullptr, 990 },
  { "strings-check-entail-len", no_argument, nullptr, 991 },
  { "no-strings-check-entail-len", no_argument, nullptr, 992 },
  { "strings-eager", no_argument, nullptr, 993 },
  { "no-strings-eager", no_argument, nullptr, 994 },
  { "strings-eager-len", no_argument, nullptr, 995 },
  { "no-strings-eager-len", no_argument, nullptr, 996 },
  { "strings-exp", no_argument, nullptr, 997 },
  { "no-strings-exp", no_argument, nullptr, 998 },
  { "strings-ff", no_argument, nullptr, 999 },
  { "no-strings-ff", no_argument, nullptr, 1000 },
  { "strings-fmf", no_argument, nullptr, 1001 },
  { "no-strings-fmf", no_argument, nullptr, 1002 },
  { "strings-guess-model", no_argument, nullptr, 1003 },
  { "no-strings-guess-model", no_argument, nullptr, 1004 },
  { "strings-infer-as-lemmas", no_argument, nullptr, 1005 },
  { "no-strings-infer-as-lemmas", no_argument, nullptr, 1006 },
  { "strings-infer-sym", no_argument, nullptr, 1007 },
  { "no-strings-infer-sym", no_argument, nullptr, 1008 },
  { "strings-inm", no_argument, nullptr, 1009 },
  { "no-strings-inm", no_argument, nullptr, 1010 },
  { "strings-lazy-pp", no_argument, nullptr, 1011 },
  { "no-strings-lazy-pp", no_argument, nullptr, 1012 },
  { "strings-len-norm", no_argument, nullptr, 1013 },
  { "no-strings-len-norm", no_argument, nullptr, 1014 },
  { "strings-lprop-csp", no_argument, nullptr, 1015 },
  { "no-strings-lprop-csp", no_argument, nullptr, 1016 },
  { "strings-min-prefix-explain", no_argument, nullptr, 1017 },
  { "no-strings-min-prefix-explain", no_argument, nullptr, 1018 },
  { "strings-print-ascii", no_argument, nullptr, 1019 },
  { "no-strings-print-ascii", no_argument, nullptr, 1020 },
  { "strings-process-loop-mode", required_argument, nullptr, 1021 },
  { "strings-rexplain-lemmas", no_argument, nullptr, 1022 },
  { "no-strings-rexplain-lemmas", no_argument, nullptr, 1023 },
  { "assign-function-values", no_argument, nullptr, 1024 },
  { "no-assign-function-values", no_argument, nullptr, 1025 },
  { "condense-function-values", no_argument, nullptr, 1026 },
  { "no-condense-function-values", no_argument, nullptr, 1027 },
  { "theoryof-mode", required_argument, nullptr, 1028 },
  { "symmetry-breaker", no_argument, nullptr, 1029 },
  { "no-symmetry-breaker", no_argument, nullptr, 1030 },
  { "uf-ho", no_argument, nullptr, 1031 },
  { "no-uf-ho", no_argument, nullptr, 1032 },
  { "uf-ho-ext", no_argument, nullptr, 1033 },
  { "no-uf-ho-ext", no_argument, nullptr, 1034 },
  { "uf-ss-abort-card", required_argument, nullptr, 1035 },
  { "uf-ss-fair", no_argument, nullptr, 1036 },
  { "no-uf-ss-fair", no_argument, nullptr, 1037 },
  { "uf-ss-fair-monotone", no_argument, nullptr, 1038 },
  { "no-uf-ss-fair-monotone", no_argument, nullptr, 1039 },
  { "uf-ss-totality", no_argument, nullptr, 1040 },
  { "no-uf-ss-totality", no_argument, nullptr, 1041 },
  { "uf-ss-totality-limited", required_argument, nullptr, 1042 },
  { "uf-ss-totality-sym-break", no_argument, nullptr, 1043 },
  { "no-uf-ss-totality-sym-break", no_argument, nullptr, 1044 },
  { "uf-ss", required_argument, nullptr, 1045 },
  { NULL, no_argument, NULL, '\0' }
};/* cmdlineOptions */

namespace options {

/** Set a given Options* as "current" just for a particular scope. */
class OptionsGuard {
  Options** d_field;
  Options* d_old;
public:
  OptionsGuard(Options** field, Options* opts) :
    d_field(field),
    d_old(*field) {
    *field = opts;
  }
  ~OptionsGuard() {
    *d_field = d_old;
  }
};/* class OptionsGuard */

}/* CVC4::options namespace */

/**
 * Parse argc/argv and put the result into a CVC4::Options.
 * The return value is what's left of the command line (that is, the
 * non-option arguments).
 *
 * Throws OptionException on failures.
 */
std::vector<std::string> Options::parseOptions(Options* options,
                                               int argc,
                                               char* argv[])
{
  Assert(options != NULL);
  Assert(argv != NULL);

  options::OptionsGuard guard(&s_current, options);

  const char *progName = argv[0];

  // To debug options parsing, you may prefer to simply uncomment this
  // and recompile. Debug flags have not been parsed yet so these have
  // not been set.
  //DebugChannel.on("options");

  Debug("options") << "Options::parseOptions == " << options << std::endl;
  Debug("options") << "argv == " << argv << std::endl;

  // Find the base name of the program.
  const char *x = strrchr(progName, '/');
  if(x != NULL) {
    progName = x + 1;
  }
  options->d_holder->binary_name = std::string(progName);


  std::vector<std::string> nonoptions;
  parseOptionsRecursive(options, argc, argv, &nonoptions);
  if(Debug.isOn("options")){
    for(std::vector<std::string>::const_iterator i = nonoptions.begin(),
          iend = nonoptions.end(); i != iend; ++i){
      Debug("options") << "nonoptions " << *i << std::endl;
    }
  }

  return nonoptions;
}

void Options::parseOptionsRecursive(Options* options,
                                    int argc,
                                    char* argv[],
                                    std::vector<std::string>* nonoptions)
{

  if(Debug.isOn("options")) {
    Debug("options") << "starting a new parseOptionsRecursive with "
                     << argc << " arguments" << std::endl;
    for( int i = 0; i < argc ; i++ ){
      Assert(argv[i] != NULL);
      Debug("options") << "  argv[" << i << "] = " << argv[i] << std::endl;
    }
  }

  // Having this synonym simplifies the generation code in mkoptions.
  options::OptionsHandler* handler = options->d_handler;

  // Reset getopt(), in the case of multiple calls to parseOptions().
  // This can be = 1 in newer GNU getopt, but older (< 2007) require = 0.
  optind = 0;
#if HAVE_DECL_OPTRESET
  optreset = 1; // on BSD getopt() (e.g. Mac OS), might need this
#endif /* HAVE_DECL_OPTRESET */

  // We must parse the binary name, which is manually ignored below. Setting
  // this to 1 leads to incorrect behavior on some platforms.
  int main_optind = 0;
  int old_optind;


  while(true) { // Repeat Forever

    optopt = 0;
    std::string option, optionarg;

    optind = main_optind;
    old_optind = main_optind;

    // If we encounter an element that is not at zero and does not start
    // with a "-", this is a non-option. We consume this element as a
    // non-option.
    if (main_optind > 0 && main_optind < argc &&
        argv[main_optind][0] != '-') {
      Debug("options") << "enqueueing " << argv[main_optind]
                       << " as a non-option." << std::endl;
      nonoptions->push_back(argv[main_optind]);
      ++main_optind;
      continue;
    }


    Debug("options") << "[ before, main_optind == " << main_optind << " ]"
                     << std::endl;
    Debug("options") << "[ before, optind == " << optind << " ]" << std::endl;
    Debug("options") << "[ argc == " << argc << ", argv == " << argv << " ]"
                     << std::endl;
    int c = getopt_long(argc, argv,
                        "+:d:L:qt:vhs:Vim",
                        cmdlineOptions, NULL);

    main_optind = optind;

    Debug("options") << "[ got " << int(c) << " (" << char(c) << ") ]"
                     << "[ next option will be at pos: " << optind << " ]"
                     << std::endl;

    // The initial getopt_long call should always determine that argv[0]
    // is not an option and returns -1. We always manually advance beyond
    // this element.
    if ( old_optind == 0  && c == -1 ) {
      Assert(main_optind > 0);
      continue;
    }

    if ( c == -1 ) {
      if(Debug.isOn("options")) {
        Debug("options") << "done with option parsing" << std::endl;
        for(int index = optind; index < argc; ++index) {
          Debug("options") << "remaining " << argv[index] << std::endl;
        }
      }
      break;
    }

    option = argv[old_optind == 0 ? 1 : old_optind];
    optionarg = (optarg == NULL) ? "" : optarg;

    Debug("preemptGetopt") << "processing option " << c
                           << " (`" << char(c) << "'), " << option << std::endl;

    switch(c) {
case 256:// --approx-branch-depth=N
      options->assign(options::maxApproxDepth, option, optionarg);
      break;

    case 257:// --arith-brab
      options->assignBool(options::brabTest, option, true);
      break;

    case 258:// --no-arith-brab
      options->assignBool(options::brabTest, option, false);
      break;

    case 259:// --arith-no-partial-fun
      options->assignBool(options::arithNoPartialFun, option, true);
      break;

    case 260:// --no-arith-no-partial-fun
      options->assignBool(options::arithNoPartialFun, option, false);
      break;

    case 261:// --arith-prop-clauses=N
      options->assign(options::arithPropAsLemmaLength, option, optionarg);
      break;

    case 262:// --arith-prop=MODE
      options->assign(options::arithPropagationMode, option, optionarg);
      break;

    case 263:// --arith-rewrite-equalities
      options->assignBool(options::arithRewriteEq, option, true);
      break;

    case 264:// --no-arith-rewrite-equalities
      options->assignBool(options::arithRewriteEq, option, false);
      break;

    case 265:// --collect-pivot-stats
      options->assignBool(options::collectPivots, option, true);
      break;

    case 266:// --no-collect-pivot-stats
      options->assignBool(options::collectPivots, option, false);
      break;

    case 267:// --cut-all-bounded
      options->assignBool(options::doCutAllBounded, option, true);
      break;

    case 268:// --no-cut-all-bounded
      options->assignBool(options::doCutAllBounded, option, false);
      break;

    case 269:// --dio-decomps
      options->assignBool(options::exportDioDecompositions, option, true);
      break;

    case 270:// --no-dio-decomps
      options->assignBool(options::exportDioDecompositions, option, false);
      break;

    case 271:// --dio-repeat
      options->assignBool(options::dioRepeat, option, true);
      break;

    case 272:// --no-dio-repeat
      options->assignBool(options::dioRepeat, option, false);
      break;

    case 273:// --dio-solver
      options->assignBool(options::arithDioSolver, option, true);
      break;

    case 274:// --no-dio-solver
      options->assignBool(options::arithDioSolver, option, false);
      break;

    case 275:// --dio-turns=N
      options->assign(options::dioSolverTurns, option, optionarg);
      break;

    case 276:// --error-selection-rule=RULE
      options->assign(options::arithErrorSelectionRule, option, optionarg);
      break;

    case 277:// --fc-penalties
      options->assignBool(options::havePenalties, option, true);
      break;

    case 278:// --no-fc-penalties
      options->assignBool(options::havePenalties, option, false);
      break;

    case 279:// --heuristic-pivots=N
      options->assign(options::arithHeuristicPivots, option, optionarg);
      break;

    case 280:// --lemmas-on-replay-failure
      options->assignBool(options::replayFailureLemma, option, true);
      break;

    case 281:// --no-lemmas-on-replay-failure
      options->assignBool(options::replayFailureLemma, option, false);
      break;

    case 282:// --maxCutsInContext=N
      options->assign(options::maxCutsInContext, option, optionarg);
      break;

    case 283:// --miplib-trick
      options->assignBool(options::arithMLTrick, option, true);
      break;

    case 284:// --no-miplib-trick
      options->assignBool(options::arithMLTrick, option, false);
      break;

    case 285:// --miplib-trick-subs=N
      options->assign(options::arithMLTrickSubstitutions, option, optionarg);
      break;

    case 286:// --new-prop
      options->assignBool(options::newProp, option, true);
      break;

    case 287:// --no-new-prop
      options->assignBool(options::newProp, option, false);
      break;

    case 288:// --nl-ext
      options->assignBool(options::nlExt, option, true);
      break;

    case 289:// --no-nl-ext
      options->assignBool(options::nlExt, option, false);
      break;

    case 290:// --nl-ext-ent-conf
      options->assignBool(options::nlExtEntailConflicts, option, true);
      break;

    case 291:// --no-nl-ext-ent-conf
      options->assignBool(options::nlExtEntailConflicts, option, false);
      break;

    case 292:// --nl-ext-factor
      options->assignBool(options::nlExtFactor, option, true);
      break;

    case 293:// --no-nl-ext-factor
      options->assignBool(options::nlExtFactor, option, false);
      break;

    case 294:// --nl-ext-inc-prec
      options->assignBool(options::nlExtIncPrecision, option, true);
      break;

    case 295:// --no-nl-ext-inc-prec
      options->assignBool(options::nlExtIncPrecision, option, false);
      break;

    case 296:// --nl-ext-purify
      options->assignBool(options::nlExtPurify, option, true);
      break;

    case 297:// --no-nl-ext-purify
      options->assignBool(options::nlExtPurify, option, false);
      break;

    case 298:// --nl-ext-rbound
      options->assignBool(options::nlExtResBound, option, true);
      break;

    case 299:// --no-nl-ext-rbound
      options->assignBool(options::nlExtResBound, option, false);
      break;

    case 300:// --nl-ext-rewrite
      options->assignBool(options::nlExtRewrites, option, true);
      break;

    case 301:// --no-nl-ext-rewrite
      options->assignBool(options::nlExtRewrites, option, false);
      break;

    case 302:// --nl-ext-split-zero
      options->assignBool(options::nlExtSplitZero, option, true);
      break;

    case 303:// --no-nl-ext-split-zero
      options->assignBool(options::nlExtSplitZero, option, false);
      break;

    case 304:// --nl-ext-tf-taylor-deg=N
      options->assign(options::nlExtTfTaylorDegree, option, optionarg);
      break;

    case 305:// --nl-ext-tf-tplanes
      options->assignBool(options::nlExtTfTangentPlanes, option, true);
      break;

    case 306:// --no-nl-ext-tf-tplanes
      options->assignBool(options::nlExtTfTangentPlanes, option, false);
      break;

    case 307:// --nl-ext-tplanes
      options->assignBool(options::nlExtTangentPlanes, option, true);
      break;

    case 308:// --no-nl-ext-tplanes
      options->assignBool(options::nlExtTangentPlanes, option, false);
      break;

    case 309:// --nl-ext-tplanes-interleave
      options->assignBool(options::nlExtTangentPlanesInterleave, option, true);
      break;

    case 310:// --no-nl-ext-tplanes-interleave
      options->assignBool(options::nlExtTangentPlanesInterleave, option, false);
      break;

    case 311:// --pb-rewrites
      options->assignBool(options::pbRewrites, option, true);
      break;

    case 312:// --no-pb-rewrites
      options->assignBool(options::pbRewrites, option, false);
      break;

    case 313:// --pivot-threshold=N
      options->assign(options::arithPivotThreshold, option, optionarg);
      break;

    case 314:// --pp-assert-max-sub-size=N
      options->assign(options::ppAssertMaxSubSize, option, optionarg);
      break;

    case 315:// --prop-row-length=N
      options->assign(options::arithPropagateMaxLength, option, optionarg);
      break;

    case 316:// --replay-early-close-depth=N
      options->assign(options::replayEarlyCloseDepths, option, optionarg);
      break;

    case 317:// --replay-failure-penalty=N
      options->assign(options::replayFailurePenalty, option, optionarg);
      break;

    case 318:// --replay-lemma-reject-cut=N
      options->assign(options::lemmaRejectCutSize, option, optionarg);
      break;

    case 319:// --replay-num-err-penalty=N
      options->assign(options::replayNumericFailurePenalty, option, optionarg);
      break;

    case 320:// --replay-reject-cut=N
      options->assign(options::replayRejectCutSize, option, optionarg);
      break;

    case 321:// --replay-soi-major-threshold-pen=N
      options->assign(options::soiApproxMajorFailurePen, option, optionarg);
      break;

    case 322:// --replay-soi-major-threshold=T
      options->assign(options::soiApproxMajorFailure, option, optionarg);
      break;

    case 323:// --replay-soi-minor-threshold-pen=N
      options->assign(options::soiApproxMinorFailurePen, option, optionarg);
      break;

    case 324:// --replay-soi-minor-threshold=T
      options->assign(options::soiApproxMinorFailure, option, optionarg);
      break;

    case 325:// --restrict-pivots
      options->assignBool(options::restrictedPivots, option, true);
      break;

    case 326:// --no-restrict-pivots
      options->assignBool(options::restrictedPivots, option, false);
      break;

    case 327:// --revert-arith-models-on-unsat
      options->assignBool(options::revertArithModels, option, true);
      break;

    case 328:// --no-revert-arith-models-on-unsat
      options->assignBool(options::revertArithModels, option, false);
      break;

    case 329:// --rr-turns=N
      options->assign(options::rrTurns, option, optionarg);
      break;

    case 330:// --se-solve-int
      options->assignBool(options::trySolveIntStandardEffort, option, true);
      break;

    case 331:// --no-se-solve-int
      options->assignBool(options::trySolveIntStandardEffort, option, false);
      break;

    case 332:// --simplex-check-period=N
      options->assign(options::arithSimplexCheckPeriod, option, optionarg);
      break;

    case 333:// --soi-qe
      options->assignBool(options::soiQuickExplain, option, true);
      break;

    case 334:// --no-soi-qe
      options->assignBool(options::soiQuickExplain, option, false);
      break;

    case 335:// --standard-effort-variable-order-pivots=N
      options->assign(options::arithStandardCheckVarOrderPivots, option, optionarg);
      break;

    case 336:// --unate-lemmas=MODE
      options->assign(options::arithUnateLemmaMode, option, optionarg);
      break;

    case 337:// --use-approx
      options->assignBool(options::useApprox, option, true);
      break;

    case 338:// --no-use-approx
      options->assignBool(options::useApprox, option, false);
      break;

    case 339:// --use-fcsimplex
      options->assignBool(options::useFC, option, true);
      break;

    case 340:// --no-use-fcsimplex
      options->assignBool(options::useFC, option, false);
      break;

    case 341:// --use-soi
      options->assignBool(options::useSOI, option, true);
      break;

    case 342:// --no-use-soi
      options->assignBool(options::useSOI, option, false);
      break;

    case 343:// --arrays-config=N
      options->assign(options::arraysConfig, option, optionarg);
      break;

    case 344:// --arrays-eager-index
      options->assignBool(options::arraysEagerIndexSplitting, option, true);
      break;

    case 345:// --no-arrays-eager-index
      options->assignBool(options::arraysEagerIndexSplitting, option, false);
      break;

    case 346:// --arrays-eager-lemmas
      options->assignBool(options::arraysEagerLemmas, option, true);
      break;

    case 347:// --no-arrays-eager-lemmas
      options->assignBool(options::arraysEagerLemmas, option, false);
      break;

    case 348:// --arrays-lazy-rintro1
      options->assignBool(options::arraysLazyRIntro1, option, true);
      break;

    case 349:// --no-arrays-lazy-rintro1
      options->assignBool(options::arraysLazyRIntro1, option, false);
      break;

    case 350:// --arrays-model-based
      options->assignBool(options::arraysModelBased, option, true);
      break;

    case 351:// --no-arrays-model-based
      options->assignBool(options::arraysModelBased, option, false);
      break;

    case 352:// --arrays-optimize-linear
      options->assignBool(options::arraysOptimizeLinear, option, true);
      break;

    case 353:// --no-arrays-optimize-linear
      options->assignBool(options::arraysOptimizeLinear, option, false);
      break;

    case 354:// --arrays-prop=N
      options->assign(options::arraysPropagate, option, optionarg);
      break;

    case 355:// --arrays-reduce-sharing
      options->assignBool(options::arraysReduceSharing, option, true);
      break;

    case 356:// --no-arrays-reduce-sharing
      options->assignBool(options::arraysReduceSharing, option, false);
      break;

    case 357:// --arrays-weak-equiv
      options->assignBool(options::arraysWeakEquivalence, option, true);
      break;

    case 358:// --no-arrays-weak-equiv
      options->assignBool(options::arraysWeakEquivalence, option, false);
      break;

    case 'd':
    case 359:// --debug=TAG
    handler->enableDebugTag(option, optionarg);
      break;

    case 'L':
    case 360:// --lang=LANG
      options->assign(options::inputLanguage, option, optionarg);
      break;

    case 361:// --output-lang=LANG
      options->assign(options::outputLanguage, option, optionarg);
      break;

    case 362:// --parse-only
      options->assignBool(options::parseOnly, option, true);
      break;

    case 363:// --no-parse-only
      options->assignBool(options::parseOnly, option, false);
      break;

    case 364:// --preprocess-only
      options->assignBool(options::preprocessOnly, option, true);
      break;

    case 365:// --no-preprocess-only
      options->assignBool(options::preprocessOnly, option, false);
      break;

    case 366:// --print-success
      options->assignBool(options::printSuccess, option, true);
      break;

    case 367:// --no-print-success
      options->assignBool(options::printSuccess, option, false);
      break;

    case 'q':
    case 368:// --quiet
    handler->decreaseVerbosity(option);
      break;

    case 369:// --stats
      options->assignBool(options::statistics, option, true);
      break;

    case 370:// --no-stats
      options->assignBool(options::statistics, option, false);
      break;

    case 371:// --stats-every-query
      options->assignBool(options::statsEveryQuery, option, true);
      break;

    case 372:// --no-stats-every-query
      options->assignBool(options::statsEveryQuery, option, false);
      break;

    case 373:// --stats-hide-zeros
      options->assignBool(options::statsHideZeros, option, true);
      break;

    case 374:// --no-stats-hide-zeros
      options->assignBool(options::statsHideZeros, option, false);
      break;

    case 't':
    case 375:// --trace=TAG
    handler->enableTraceTag(option, optionarg);
      break;

    case 'v':
    case 376:// --verbose
    handler->increaseVerbosity(option);
      break;

    case 377:// --bitblast-aig
      options->assignBool(options::bitvectorAig, option, true);
      break;

    case 378:// --no-bitblast-aig
      options->assignBool(options::bitvectorAig, option, false);
      break;

    case 379:// --bitblast=MODE
      options->assign(options::bitblastMode, option, optionarg);
      break;

    case 380:// --bitwise-eq
      options->assignBool(options::bitwiseEq, option, true);
      break;

    case 381:// --no-bitwise-eq
      options->assignBool(options::bitwiseEq, option, false);
      break;

    case 382:// --bool-to-bv=MODE
      options->assign(options::boolToBitvector, option, optionarg);
      break;

    case 383:// --bv-abstraction
      options->assignBool(options::bvAbstraction, option, true);
      break;

    case 384:// --no-bv-abstraction
      options->assignBool(options::bvAbstraction, option, false);
      break;

    case 385:// --bv-aig-simp=COMMAND
      options->assign(options::bitvectorAigSimplifications, option, optionarg);
      break;

    case 386:// --bv-alg-extf
      options->assignBool(options::bvAlgExtf, option, true);
      break;

    case 387:// --no-bv-alg-extf
      options->assignBool(options::bvAlgExtf, option, false);
      break;

    case 388:// --bv-algebraic-budget=N
      options->assign(options::bitvectorAlgebraicBudget, option, optionarg);
      break;

    case 389:// --bv-algebraic-solver
      options->assignBool(options::bitvectorAlgebraicSolver, option, true);
      break;

    case 390:// --no-bv-algebraic-solver
      options->assignBool(options::bitvectorAlgebraicSolver, option, false);
      break;

    case 391:// --bv-div-zero-const
      options->assignBool(options::bitvectorDivByZeroConst, option, true);
      break;

    case 392:// --no-bv-div-zero-const
      options->assignBool(options::bitvectorDivByZeroConst, option, false);
      break;

    case 393:// --bv-eager-explanations
      options->assignBool(options::bvEagerExplanations, option, true);
      break;

    case 394:// --no-bv-eager-explanations
      options->assignBool(options::bvEagerExplanations, option, false);
      break;

    case 395:// --bv-eq-slicer=MODE
      options->assign(options::bitvectorEqualitySlicer, option, optionarg);
      break;

    case 396:// --bv-eq-solver
      options->assignBool(options::bitvectorEqualitySolver, option, true);
      break;

    case 397:// --no-bv-eq-solver
      options->assignBool(options::bitvectorEqualitySolver, option, false);
      break;

    case 398:// --bv-extract-arith
      options->assignBool(options::bvExtractArithRewrite, option, true);
      break;

    case 399:// --no-bv-extract-arith
      options->assignBool(options::bvExtractArithRewrite, option, false);
      break;

    case 400:// --bv-gauss-elim
      options->assignBool(options::bvGaussElim, option, true);
      break;

    case 401:// --no-bv-gauss-elim
      options->assignBool(options::bvGaussElim, option, false);
      break;

    case 402:// --bv-inequality-solver
      options->assignBool(options::bitvectorInequalitySolver, option, true);
      break;

    case 403:// --no-bv-inequality-solver
      options->assignBool(options::bitvectorInequalitySolver, option, false);
      break;

    case 404:// --bv-intro-pow2
      options->assignBool(options::bvIntroducePow2, option, true);
      break;

    case 405:// --no-bv-intro-pow2
      options->assignBool(options::bvIntroducePow2, option, false);
      break;

    case 406:// --bv-lazy-reduce-extf
      options->assignBool(options::bvLazyReduceExtf, option, true);
      break;

    case 407:// --no-bv-lazy-reduce-extf
      options->assignBool(options::bvLazyReduceExtf, option, false);
      break;

    case 408:// --bv-lazy-rewrite-extf
      options->assignBool(options::bvLazyRewriteExtf, option, true);
      break;

    case 409:// --no-bv-lazy-rewrite-extf
      options->assignBool(options::bvLazyRewriteExtf, option, false);
      break;

    case 410:// --bv-num-func=N
      options->assign(options::bvNumFunc, option, optionarg);
      break;

    case 411:// --bv-optimize-sat-proof=MODE
      options->assign(options::bvOptimizeSatProof, option, optionarg);
      break;

    case 412:// --bv-print-consts-as-indexed-symbols
      options->assignBool(options::bvPrintConstsAsIndexedSymbols, option, true);
      break;

    case 413:// --no-bv-print-consts-as-indexed-symbols
      options->assignBool(options::bvPrintConstsAsIndexedSymbols, option, false);
      break;

    case 414:// --bv-proof-format=MODE
      options->assign(options::bvProofFormat, option, optionarg);
      break;

    case 415:// --bv-propagate
      options->assignBool(options::bitvectorPropagate, option, true);
      break;

    case 416:// --no-bv-propagate
      options->assignBool(options::bitvectorPropagate, option, false);
      break;

    case 417:// --bv-quick-xplain
      options->assignBool(options::bitvectorQuickXplain, option, true);
      break;

    case 418:// --no-bv-quick-xplain
      options->assignBool(options::bitvectorQuickXplain, option, false);
      break;

    case 419:// --bv-sat-solver=MODE
      options->assign(options::bvSatSolver, option, optionarg);
      break;

    case 420:// --bv-skolemize
      options->assignBool(options::skolemizeArguments, option, true);
      break;

    case 421:// --no-bv-skolemize
      options->assignBool(options::skolemizeArguments, option, false);
      break;

    case 422:// --bv-to-bool
      options->assignBool(options::bitvectorToBool, option, true);
      break;

    case 423:// --no-bv-to-bool
      options->assignBool(options::bitvectorToBool, option, false);
      break;

    case 424:// --cdt-bisimilar
      options->assignBool(options::cdtBisimilar, option, true);
      break;

    case 425:// --no-cdt-bisimilar
      options->assignBool(options::cdtBisimilar, option, false);
      break;

    case 426:// --dt-binary-split
      options->assignBool(options::dtBinarySplit, option, true);
      break;

    case 427:// --no-dt-binary-split
      options->assignBool(options::dtBinarySplit, option, false);
      break;

    case 428:// --dt-blast-splits
      options->assignBool(options::dtBlastSplits, option, true);
      break;

    case 429:// --no-dt-blast-splits
      options->assignBool(options::dtBlastSplits, option, false);
      break;

    case 430:// --dt-cyclic
      options->assignBool(options::dtCyclic, option, true);
      break;

    case 431:// --no-dt-cyclic
      options->assignBool(options::dtCyclic, option, false);
      break;

    case 432:// --dt-force-assignment
      options->assignBool(options::dtForceAssignment, option, true);
      break;

    case 433:// --no-dt-force-assignment
      options->assignBool(options::dtForceAssignment, option, false);
      break;

    case 434:// --dt-infer-as-lemmas
      options->assignBool(options::dtInferAsLemmas, option, true);
      break;

    case 435:// --no-dt-infer-as-lemmas
      options->assignBool(options::dtInferAsLemmas, option, false);
      break;

    case 436:// --dt-nested-rec
      options->assignBool(options::dtNestedRec, option, true);
      break;

    case 437:// --no-dt-nested-rec
      options->assignBool(options::dtNestedRec, option, false);
      break;

    case 438:// --dt-rewrite-error-sel
      options->assignBool(options::dtRewriteErrorSel, option, true);
      break;

    case 439:// --no-dt-rewrite-error-sel
      options->assignBool(options::dtRewriteErrorSel, option, false);
      break;

    case 440:// --dt-share-sel
      options->assignBool(options::dtSharedSelectors, option, true);
      break;

    case 441:// --no-dt-share-sel
      options->assignBool(options::dtSharedSelectors, option, false);
      break;

    case 442:// --sygus-abort-size=N
      options->assign(options::sygusAbortSize, option, optionarg);
      break;

    case 443:// --sygus-fair-max
      options->assignBool(options::sygusFairMax, option, true);
      break;

    case 444:// --no-sygus-fair-max
      options->assignBool(options::sygusFairMax, option, false);
      break;

    case 445:// --sygus-fair=MODE
      options->assign(options::sygusFair, option, optionarg);
      break;

    case 446:// --sygus-sym-break
      options->assignBool(options::sygusSymBreak, option, true);
      break;

    case 447:// --no-sygus-sym-break
      options->assignBool(options::sygusSymBreak, option, false);
      break;

    case 448:// --sygus-sym-break-agg
      options->assignBool(options::sygusSymBreakAgg, option, true);
      break;

    case 449:// --no-sygus-sym-break-agg
      options->assignBool(options::sygusSymBreakAgg, option, false);
      break;

    case 450:// --sygus-sym-break-dynamic
      options->assignBool(options::sygusSymBreakDynamic, option, true);
      break;

    case 451:// --no-sygus-sym-break-dynamic
      options->assignBool(options::sygusSymBreakDynamic, option, false);
      break;

    case 452:// --sygus-sym-break-lazy
      options->assignBool(options::sygusSymBreakLazy, option, true);
      break;

    case 453:// --no-sygus-sym-break-lazy
      options->assignBool(options::sygusSymBreakLazy, option, false);
      break;

    case 454:// --sygus-sym-break-pbe
      options->assignBool(options::sygusSymBreakPbe, option, true);
      break;

    case 455:// --no-sygus-sym-break-pbe
      options->assignBool(options::sygusSymBreakPbe, option, false);
      break;

    case 456:// --sygus-sym-break-rlv
      options->assignBool(options::sygusSymBreakRlv, option, true);
      break;

    case 457:// --no-sygus-sym-break-rlv
      options->assignBool(options::sygusSymBreakRlv, option, false);
      break;

    case 458:// --decision-random-weight=N
      options->assign(options::decisionRandomWeight, option, optionarg);
      break;

    case 459:// --decision-threshold=N
      options->assign(options::decisionThreshold, option, optionarg);
      break;

    case 460:// --decision-use-weight
      options->assignBool(options::decisionUseWeight, option, true);
      break;

    case 461:// --no-decision-use-weight
      options->assignBool(options::decisionUseWeight, option, false);
      break;

    case 462:// --decision-weight-internal=HOW
      options->assign(options::decisionWeightInternal, option, optionarg);
      break;

    case 463:// --decision=MODE
      options->assign(options::decisionMode, option, optionarg);
      break;

    case 464:// --dag-thresh=N
      options->assign(options::defaultDagThresh, option, optionarg);
      break;

    case 465:// --expr-depth=N
      options->assign(options::defaultExprDepth, option, optionarg);
      break;

    case 466:// --print-expr-types
      options->assignBool(options::printExprTypes, option, true);
      break;

    case 467:// --no-print-expr-types
      options->assignBool(options::printExprTypes, option, false);
      break;

    case 468:// --type-checking
      options->assignBool(options::typeChecking, option, true);
      break;

    case 469:// --no-type-checking
      options->assignBool(options::typeChecking, option, false);
      break;

    case 470:// --fp-exp
      options->assignBool(options::fpExp, option, true);
      break;

    case 471:// --no-fp-exp
      options->assignBool(options::fpExp, option, false);
      break;

    case 472:// --copyright
    handler->copyright(option);
      break;

    case 473:// --early-exit
      options->assignBool(options::earlyExit, option, true);
      break;

    case 474:// --no-early-exit
      options->assignBool(options::earlyExit, option, false);
      break;

    case 'h':
    case 475:// --help
      options->assignBool(options::help, option, true);
      break;

    case 476:// --interactive
      options->assignBool(options::interactive, option, true);
      break;

    case 477:// --no-interactive
      options->assignBool(options::interactive, option, false);
      break;

    case 478:// --interactive-prompt
      options->assignBool(options::interactivePrompt, option, true);
      break;

    case 479:// --no-interactive-prompt
      options->assignBool(options::interactivePrompt, option, false);
      break;

    case 's':
    case 480:// --seed=N
      options->assign(options::seed, option, optionarg);
      break;

    case 481:// --segv-spin
      options->assignBool(options::segvSpin, option, true);
      break;

    case 482:// --no-segv-spin
      options->assignBool(options::segvSpin, option, false);
      break;

    case 483:// --show-config
    handler->showConfiguration(option);
      break;

    case 484:// --show-debug-tags
    handler->showDebugTags(option);
      break;

    case 485:// --show-trace-tags
    handler->showTraceTags(option);
      break;

    case 486:// --tear-down-incremental=N
      options->assign(options::tearDownIncremental, option, optionarg);
      break;

    case 'V':
    case 487:// --version
      options->assignBool(options::version, option, true);
      break;

    case 488:// --filesystem-access
      options->assignBool(options::filesystemAccess, option, true);
      break;

    case 489:// --no-filesystem-access
      options->assignBool(options::filesystemAccess, option, false);
      break;

    case 490:// --force-logic=LOGIC
      options->assign(options::forceLogicString, option, optionarg);
      break;

    case 491:// --mmap
      options->assignBool(options::memoryMap, option, true);
      break;

    case 492:// --no-mmap
      options->assignBool(options::memoryMap, option, false);
      break;

    case 493:// --strict-parsing
      options->assignBool(options::strictParsing, option, true);
      break;

    case 494:// --no-strict-parsing
      options->assignBool(options::strictParsing, option, false);
      break;

    case 495:// --flatten-ho-chains
      options->assignBool(options::flattenHOChains, option, true);
      break;

    case 496:// --no-flatten-ho-chains
      options->assignBool(options::flattenHOChains, option, false);
      break;

    case 497:// --inst-format=MODE
      options->assign(options::instFormatMode, option, optionarg);
      break;

    case 498:// --model-format=MODE
      options->assign(options::modelFormatMode, option, optionarg);
      break;

    case 499:// --aggressive-core-min
      options->assignBool(options::aggressiveCoreMin, option, true);
      break;

    case 500:// --no-aggressive-core-min
      options->assignBool(options::aggressiveCoreMin, option, false);
      break;

    case 501:// --allow-empty-dependencies
      options->assignBool(options::allowEmptyDependencies, option, true);
      break;

    case 502:// --no-allow-empty-dependencies
      options->assignBool(options::allowEmptyDependencies, option, false);
      break;

    case 503:// --fewer-preprocessing-holes
      options->assignBool(options::fewerPreprocessingHoles, option, true);
      break;

    case 504:// --no-fewer-preprocessing-holes
      options->assignBool(options::fewerPreprocessingHoles, option, false);
      break;

    case 505:// --lfsc-letification
      options->assignBool(options::lfscLetification, option, true);
      break;

    case 506:// --no-lfsc-letification
      options->assignBool(options::lfscLetification, option, false);
      break;

    case 507:// --minisat-dump-dimacs
      options->assignBool(options::minisatDumpDimacs, option, true);
      break;

    case 508:// --no-minisat-dump-dimacs
      options->assignBool(options::minisatDumpDimacs, option, false);
      break;

    case 509:// --minisat-elimination
      options->assignBool(options::minisatUseElim, option, true);
      break;

    case 510:// --no-minisat-elimination
      options->assignBool(options::minisatUseElim, option, false);
      break;

    case 511:// --random-freq=P
      options->assign(options::satRandomFreq, option, optionarg);
      break;

    case 512:// --random-seed=S
      options->assign(options::satRandomSeed, option, optionarg);
      break;

    case 513:// --refine-conflicts
      options->assignBool(options::sat_refine_conflicts, option, true);
      break;

    case 514:// --no-refine-conflicts
      options->assignBool(options::sat_refine_conflicts, option, false);
      break;

    case 515:// --restart-int-base=N
      options->assign(options::satRestartFirst, option, optionarg);
      break;

    case 516:// --restart-int-inc=F
      options->assign(options::satRestartInc, option, optionarg);
      break;

    case 517:// --ag-miniscope-quant
      options->assignBool(options::aggressiveMiniscopeQuant, option, true);
      break;

    case 518:// --no-ag-miniscope-quant
      options->assignBool(options::aggressiveMiniscopeQuant, option, false);
      break;

    case 519:// --cegis-sample=MODE
      options->assign(options::cegisSample, option, optionarg);
      break;

    case 520:// --cegqi
      options->assignBool(options::cegqi, option, true);
      break;

    case 521:// --no-cegqi
      options->assignBool(options::cegqi, option, false);
      break;

    case 522:// --cegqi-all
      options->assignBool(options::cegqiAll, option, true);
      break;

    case 523:// --no-cegqi-all
      options->assignBool(options::cegqiAll, option, false);
      break;

    case 524:// --cegqi-bv
      options->assignBool(options::cegqiBv, option, true);
      break;

    case 525:// --no-cegqi-bv
      options->assignBool(options::cegqiBv, option, false);
      break;

    case 526:// --cegqi-bv-concat-inv
      options->assignBool(options::cegqiBvConcInv, option, true);
      break;

    case 527:// --no-cegqi-bv-concat-inv
      options->assignBool(options::cegqiBvConcInv, option, false);
      break;

    case 528:// --cegqi-bv-ineq=MODE
      options->assign(options::cegqiBvIneqMode, option, optionarg);
      break;

    case 529:// --cegqi-bv-interleave-value
      options->assignBool(options::cegqiBvInterleaveValue, option, true);
      break;

    case 530:// --no-cegqi-bv-interleave-value
      options->assignBool(options::cegqiBvInterleaveValue, option, false);
      break;

    case 531:// --cegqi-bv-linear
      options->assignBool(options::cegqiBvLinearize, option, true);
      break;

    case 532:// --no-cegqi-bv-linear
      options->assignBool(options::cegqiBvLinearize, option, false);
      break;

    case 533:// --cegqi-bv-rm-extract
      options->assignBool(options::cegqiBvRmExtract, option, true);
      break;

    case 534:// --no-cegqi-bv-rm-extract
      options->assignBool(options::cegqiBvRmExtract, option, false);
      break;

    case 535:// --cegqi-bv-solve-nl
      options->assignBool(options::cegqiBvSolveNl, option, true);
      break;

    case 536:// --no-cegqi-bv-solve-nl
      options->assignBool(options::cegqiBvSolveNl, option, false);
      break;

    case 537:// --cegqi-full
      options->assignBool(options::cegqiFullEffort, option, true);
      break;

    case 538:// --no-cegqi-full
      options->assignBool(options::cegqiFullEffort, option, false);
      break;

    case 539:// --cegqi-innermost
      options->assignBool(options::cegqiInnermost, option, true);
      break;

    case 540:// --no-cegqi-innermost
      options->assignBool(options::cegqiInnermost, option, false);
      break;

    case 541:// --cegqi-midpoint
      options->assignBool(options::cegqiMidpoint, option, true);
      break;

    case 542:// --no-cegqi-midpoint
      options->assignBool(options::cegqiMidpoint, option, false);
      break;

    case 543:// --cegqi-min-bounds
      options->assignBool(options::cegqiMinBounds, option, true);
      break;

    case 544:// --no-cegqi-min-bounds
      options->assignBool(options::cegqiMinBounds, option, false);
      break;

    case 545:// --cegqi-model
      options->assignBool(options::cegqiModel, option, true);
      break;

    case 546:// --no-cegqi-model
      options->assignBool(options::cegqiModel, option, false);
      break;

    case 547:// --cegqi-multi-inst
      options->assignBool(options::cegqiMultiInst, option, true);
      break;

    case 548:// --no-cegqi-multi-inst
      options->assignBool(options::cegqiMultiInst, option, false);
      break;

    case 549:// --cegqi-nested-qe
      options->assignBool(options::cegqiNestedQE, option, true);
      break;

    case 550:// --no-cegqi-nested-qe
      options->assignBool(options::cegqiNestedQE, option, false);
      break;

    case 551:// --cegqi-nopt
      options->assignBool(options::cegqiNopt, option, true);
      break;

    case 552:// --no-cegqi-nopt
      options->assignBool(options::cegqiNopt, option, false);
      break;

    case 553:// --cegqi-prereg-inst
      options->assignBool(options::cegqiPreRegInst, option, true);
      break;

    case 554:// --no-cegqi-prereg-inst
      options->assignBool(options::cegqiPreRegInst, option, false);
      break;

    case 555:// --cegqi-repeat-lit
      options->assignBool(options::cegqiRepeatLit, option, true);
      break;

    case 556:// --no-cegqi-repeat-lit
      options->assignBool(options::cegqiRepeatLit, option, false);
      break;

    case 557:// --cegqi-round-up-lia
      options->assignBool(options::cegqiRoundUpLowerLia, option, true);
      break;

    case 558:// --no-cegqi-round-up-lia
      options->assignBool(options::cegqiRoundUpLowerLia, option, false);
      break;

    case 559:// --cegqi-sat
      options->assignBool(options::cegqiSat, option, true);
      break;

    case 560:// --no-cegqi-sat
      options->assignBool(options::cegqiSat, option, false);
      break;

    case 561:// --cegqi-use-inf-int
      options->assignBool(options::cegqiUseInfInt, option, true);
      break;

    case 562:// --no-cegqi-use-inf-int
      options->assignBool(options::cegqiUseInfInt, option, false);
      break;

    case 563:// --cegqi-use-inf-real
      options->assignBool(options::cegqiUseInfReal, option, true);
      break;

    case 564:// --no-cegqi-use-inf-real
      options->assignBool(options::cegqiUseInfReal, option, false);
      break;

    case 565:// --cond-var-split-agg-quant
      options->assignBool(options::condVarSplitQuantAgg, option, true);
      break;

    case 566:// --no-cond-var-split-agg-quant
      options->assignBool(options::condVarSplitQuantAgg, option, false);
      break;

    case 567:// --cond-var-split-quant
      options->assignBool(options::condVarSplitQuant, option, true);
      break;

    case 568:// --no-cond-var-split-quant
      options->assignBool(options::condVarSplitQuant, option, false);
      break;

    case 569:// --conjecture-filter-active-terms
      options->assignBool(options::conjectureFilterActiveTerms, option, true);
      break;

    case 570:// --no-conjecture-filter-active-terms
      options->assignBool(options::conjectureFilterActiveTerms, option, false);
      break;

    case 571:// --conjecture-filter-canonical
      options->assignBool(options::conjectureFilterCanonical, option, true);
      break;

    case 572:// --no-conjecture-filter-canonical
      options->assignBool(options::conjectureFilterCanonical, option, false);
      break;

    case 573:// --conjecture-filter-model
      options->assignBool(options::conjectureFilterModel, option, true);
      break;

    case 574:// --no-conjecture-filter-model
      options->assignBool(options::conjectureFilterModel, option, false);
      break;

    case 575:// --conjecture-gen
      options->assignBool(options::conjectureGen, option, true);
      break;

    case 576:// --no-conjecture-gen
      options->assignBool(options::conjectureGen, option, false);
      break;

    case 577:// --conjecture-gen-gt-enum=N
      options->assign(options::conjectureGenGtEnum, option, optionarg);
      break;

    case 578:// --conjecture-gen-max-depth=N
      options->assign(options::conjectureGenMaxDepth, option, optionarg);
      break;

    case 579:// --conjecture-gen-per-round=N
      options->assign(options::conjectureGenPerRound, option, optionarg);
      break;

    case 580:// --conjecture-gen-uee-intro
      options->assignBool(options::conjectureUeeIntro, option, true);
      break;

    case 581:// --no-conjecture-gen-uee-intro
      options->assignBool(options::conjectureUeeIntro, option, false);
      break;

    case 582:// --conjecture-no-filter
      options->assignBool(options::conjectureNoFilter, option, true);
      break;

    case 583:// --no-conjecture-no-filter
      options->assignBool(options::conjectureNoFilter, option, false);
      break;

    case 584:// --dt-stc-ind
      options->assignBool(options::dtStcInduction, option, true);
      break;

    case 585:// --no-dt-stc-ind
      options->assignBool(options::dtStcInduction, option, false);
      break;

    case 586:// --dt-var-exp-quant
      options->assignBool(options::dtVarExpandQuant, option, true);
      break;

    case 587:// --no-dt-var-exp-quant
      options->assignBool(options::dtVarExpandQuant, option, false);
      break;

    case 588:// --e-matching
      options->assignBool(options::eMatching, option, true);
      break;

    case 589:// --no-e-matching
      options->assignBool(options::eMatching, option, false);
      break;

    case 590:// --elim-ext-arith-quant
      options->assignBool(options::elimExtArithQuant, option, true);
      break;

    case 591:// --no-elim-ext-arith-quant
      options->assignBool(options::elimExtArithQuant, option, false);
      break;

    case 592:// --elim-taut-quant
      options->assignBool(options::elimTautQuant, option, true);
      break;

    case 593:// --no-elim-taut-quant
      options->assignBool(options::elimTautQuant, option, false);
      break;

    case 594:// --ext-rewrite-quant
      options->assignBool(options::extRewriteQuant, option, true);
      break;

    case 595:// --no-ext-rewrite-quant
      options->assignBool(options::extRewriteQuant, option, false);
      break;

    case 596:// --finite-model-find
      options->assignBool(options::finiteModelFind, option, true);
      break;

    case 597:// --no-finite-model-find
      options->assignBool(options::finiteModelFind, option, false);
      break;

    case 598:// --fmf-bound
      options->assignBool(options::fmfBound, option, true);
      break;

    case 599:// --no-fmf-bound
      options->assignBool(options::fmfBound, option, false);
      break;

    case 600:// --fmf-bound-int
      options->assignBool(options::fmfBoundInt, option, true);
      break;

    case 601:// --no-fmf-bound-int
      options->assignBool(options::fmfBoundInt, option, false);
      break;

    case 602:// --fmf-bound-lazy
      options->assignBool(options::fmfBoundLazy, option, true);
      break;

    case 603:// --no-fmf-bound-lazy
      options->assignBool(options::fmfBoundLazy, option, false);
      break;

    case 604:// --fmf-fmc-simple
      options->assignBool(options::fmfFmcSimple, option, true);
      break;

    case 605:// --no-fmf-fmc-simple
      options->assignBool(options::fmfFmcSimple, option, false);
      break;

    case 606:// --fmf-fresh-dc
      options->assignBool(options::fmfFreshDistConst, option, true);
      break;

    case 607:// --no-fmf-fresh-dc
      options->assignBool(options::fmfFreshDistConst, option, false);
      break;

    case 608:// --fmf-fun
      options->assignBool(options::fmfFunWellDefined, option, true);
      break;

    case 609:// --no-fmf-fun
      options->assignBool(options::fmfFunWellDefined, option, false);
      break;

    case 610:// --fmf-fun-rlv
      options->assignBool(options::fmfFunWellDefinedRelevant, option, true);
      break;

    case 611:// --no-fmf-fun-rlv
      options->assignBool(options::fmfFunWellDefinedRelevant, option, false);
      break;

    case 612:// --fmf-inst-engine
      options->assignBool(options::fmfInstEngine, option, true);
      break;

    case 613:// --no-fmf-inst-engine
      options->assignBool(options::fmfInstEngine, option, false);
      break;

    case 614:// --fmf-type-completion-thresh=N
      options->assign(options::fmfTypeCompletionThresh, option, optionarg);
      break;

    case 615:// --fs-interleave
      options->assignBool(options::fullSaturateInterleave, option, true);
      break;

    case 616:// --no-fs-interleave
      options->assignBool(options::fullSaturateInterleave, option, false);
      break;

    case 617:// --fs-stratify
      options->assignBool(options::fullSaturateStratify, option, true);
      break;

    case 618:// --no-fs-stratify
      options->assignBool(options::fullSaturateStratify, option, false);
      break;

    case 619:// --full-saturate-quant
      options->assignBool(options::fullSaturateQuant, option, true);
      break;

    case 620:// --no-full-saturate-quant
      options->assignBool(options::fullSaturateQuant, option, false);
      break;

    case 621:// --full-saturate-quant-limit=N
      options->assign(options::fullSaturateLimit, option, optionarg);
      break;

    case 622:// --full-saturate-quant-rd
      options->assignBool(options::fullSaturateQuantRd, option, true);
      break;

    case 623:// --no-full-saturate-quant-rd
      options->assignBool(options::fullSaturateQuantRd, option, false);
      break;

    case 624:// --global-negate
      options->assignBool(options::globalNegate, option, true);
      break;

    case 625:// --no-global-negate
      options->assignBool(options::globalNegate, option, false);
      break;

    case 626:// --ho-elim
      options->assignBool(options::hoElim, option, true);
      break;

    case 627:// --no-ho-elim
      options->assignBool(options::hoElim, option, false);
      break;

    case 628:// --ho-elim-store-ax
      options->assignBool(options::hoElimStoreAx, option, true);
      break;

    case 629:// --no-ho-elim-store-ax
      options->assignBool(options::hoElimStoreAx, option, false);
      break;

    case 630:// --ho-matching
      options->assignBool(options::hoMatching, option, true);
      break;

    case 631:// --no-ho-matching
      options->assignBool(options::hoMatching, option, false);
      break;

    case 632:// --ho-matching-var-priority
      options->assignBool(options::hoMatchingVarArgPriority, option, true);
      break;

    case 633:// --no-ho-matching-var-priority
      options->assignBool(options::hoMatchingVarArgPriority, option, false);
      break;

    case 634:// --ho-merge-term-db
      options->assignBool(options::hoMergeTermDb, option, true);
      break;

    case 635:// --no-ho-merge-term-db
      options->assignBool(options::hoMergeTermDb, option, false);
      break;

    case 636:// --increment-triggers
      options->assignBool(options::incrementTriggers, option, true);
      break;

    case 637:// --no-increment-triggers
      options->assignBool(options::incrementTriggers, option, false);
      break;

    case 638:// --inst-level-input-only
      options->assignBool(options::instLevelInputOnly, option, true);
      break;

    case 639:// --no-inst-level-input-only
      options->assignBool(options::instLevelInputOnly, option, false);
      break;

    case 640:// --inst-max-level=N
      options->assign(options::instMaxLevel, option, optionarg);
      break;

    case 641:// --inst-no-entail
      options->assignBool(options::instNoEntail, option, true);
      break;

    case 642:// --no-inst-no-entail
      options->assignBool(options::instNoEntail, option, false);
      break;

    case 643:// --inst-no-model-true
      options->assignBool(options::instNoModelTrue, option, true);
      break;

    case 644:// --no-inst-no-model-true
      options->assignBool(options::instNoModelTrue, option, false);
      break;

    case 645:// --inst-when-phase=N
      options->assign(options::instWhenPhase, option, optionarg);
      break;

    case 646:// --inst-when-strict-interleave
      options->assignBool(options::instWhenStrictInterleave, option, true);
      break;

    case 647:// --no-inst-when-strict-interleave
      options->assignBool(options::instWhenStrictInterleave, option, false);
      break;

    case 648:// --inst-when-tc-first
      options->assignBool(options::instWhenTcFirst, option, true);
      break;

    case 649:// --no-inst-when-tc-first
      options->assignBool(options::instWhenTcFirst, option, false);
      break;

    case 650:// --inst-when=MODE
      options->assign(options::instWhenMode, option, optionarg);
      break;

    case 651:// --int-wf-ind
      options->assignBool(options::intWfInduction, option, true);
      break;

    case 652:// --no-int-wf-ind
      options->assignBool(options::intWfInduction, option, false);
      break;

    case 653:// --ite-dtt-split-quant
      options->assignBool(options::iteDtTesterSplitQuant, option, true);
      break;

    case 654:// --no-ite-dtt-split-quant
      options->assignBool(options::iteDtTesterSplitQuant, option, false);
      break;

    case 655:// --ite-lift-quant=MODE
      options->assign(options::iteLiftQuant, option, optionarg);
      break;

    case 656:// --literal-matching=MODE
      options->assign(options::literalMatchMode, option, optionarg);
      break;

    case 657:// --lte-restrict-inst-closure
      options->assignBool(options::lteRestrictInstClosure, option, true);
      break;

    case 658:// --no-lte-restrict-inst-closure
      options->assignBool(options::lteRestrictInstClosure, option, false);
      break;

    case 659:// --macros-quant
      options->assignBool(options::macrosQuant, option, true);
      break;

    case 660:// --no-macros-quant
      options->assignBool(options::macrosQuant, option, false);
      break;

    case 661:// --macros-quant-mode=MODE
      options->assign(options::macrosQuantMode, option, optionarg);
      break;

    case 662:// --mbqi-interleave
      options->assignBool(options::mbqiInterleave, option, true);
      break;

    case 663:// --no-mbqi-interleave
      options->assignBool(options::mbqiInterleave, option, false);
      break;

    case 664:// --mbqi-one-inst-per-round
      options->assignBool(options::fmfOneInstPerRound, option, true);
      break;

    case 665:// --no-mbqi-one-inst-per-round
      options->assignBool(options::fmfOneInstPerRound, option, false);
      break;

    case 666:// --mbqi=MODE
      options->assign(options::mbqiMode, option, optionarg);
      break;

    case 667:// --miniscope-quant
      options->assignBool(options::miniscopeQuant, option, true);
      break;

    case 668:// --no-miniscope-quant
      options->assignBool(options::miniscopeQuant, option, false);
      break;

    case 669:// --miniscope-quant-fv
      options->assignBool(options::miniscopeQuantFreeVar, option, true);
      break;

    case 670:// --no-miniscope-quant-fv
      options->assignBool(options::miniscopeQuantFreeVar, option, false);
      break;

    case 671:// --multi-trigger-cache
      options->assignBool(options::multiTriggerCache, option, true);
      break;

    case 672:// --no-multi-trigger-cache
      options->assignBool(options::multiTriggerCache, option, false);
      break;

    case 673:// --multi-trigger-linear
      options->assignBool(options::multiTriggerLinear, option, true);
      break;

    case 674:// --no-multi-trigger-linear
      options->assignBool(options::multiTriggerLinear, option, false);
      break;

    case 675:// --multi-trigger-priority
      options->assignBool(options::multiTriggerPriority, option, true);
      break;

    case 676:// --no-multi-trigger-priority
      options->assignBool(options::multiTriggerPriority, option, false);
      break;

    case 677:// --multi-trigger-when-single
      options->assignBool(options::multiTriggerWhenSingle, option, true);
      break;

    case 678:// --no-multi-trigger-when-single
      options->assignBool(options::multiTriggerWhenSingle, option, false);
      break;

    case 679:// --partial-triggers
      options->assignBool(options::partialTriggers, option, true);
      break;

    case 680:// --no-partial-triggers
      options->assignBool(options::partialTriggers, option, false);
      break;

    case 681:// --pre-skolem-quant
      options->assignBool(options::preSkolemQuant, option, true);
      break;

    case 682:// --no-pre-skolem-quant
      options->assignBool(options::preSkolemQuant, option, false);
      break;

    case 683:// --pre-skolem-quant-agg
      options->assignBool(options::preSkolemQuantAgg, option, true);
      break;

    case 684:// --no-pre-skolem-quant-agg
      options->assignBool(options::preSkolemQuantAgg, option, false);
      break;

    case 685:// --pre-skolem-quant-nested
      options->assignBool(options::preSkolemQuantNested, option, true);
      break;

    case 686:// --no-pre-skolem-quant-nested
      options->assignBool(options::preSkolemQuantNested, option, false);
      break;

    case 687:// --prenex-quant-user
      options->assignBool(options::prenexQuantUser, option, true);
      break;

    case 688:// --no-prenex-quant-user
      options->assignBool(options::prenexQuantUser, option, false);
      break;

    case 689:// --prenex-quant=MODE
      options->assign(options::prenexQuant, option, optionarg);
      break;

    case 690:// --pure-th-triggers
      options->assignBool(options::pureThTriggers, option, true);
      break;

    case 691:// --no-pure-th-triggers
      options->assignBool(options::pureThTriggers, option, false);
      break;

    case 692:// --purify-dt-triggers
      options->assignBool(options::purifyDtTriggers, option, true);
      break;

    case 693:// --no-purify-dt-triggers
      options->assignBool(options::purifyDtTriggers, option, false);
      break;

    case 694:// --purify-triggers
      options->assignBool(options::purifyTriggers, option, true);
      break;

    case 695:// --no-purify-triggers
      options->assignBool(options::purifyTriggers, option, false);
      break;

    case 696:// --qcf-all-conflict
      options->assignBool(options::qcfAllConflict, option, true);
      break;

    case 697:// --no-qcf-all-conflict
      options->assignBool(options::qcfAllConflict, option, false);
      break;

    case 698:// --qcf-eager-check-rd
      options->assignBool(options::qcfEagerCheckRd, option, true);
      break;

    case 699:// --no-qcf-eager-check-rd
      options->assignBool(options::qcfEagerCheckRd, option, false);
      break;

    case 700:// --qcf-eager-test
      options->assignBool(options::qcfEagerTest, option, true);
      break;

    case 701:// --no-qcf-eager-test
      options->assignBool(options::qcfEagerTest, option, false);
      break;

    case 702:// --qcf-nested-conflict
      options->assignBool(options::qcfNestedConflict, option, true);
      break;

    case 703:// --no-qcf-nested-conflict
      options->assignBool(options::qcfNestedConflict, option, false);
      break;

    case 704:// --qcf-skip-rd
      options->assignBool(options::qcfSkipRd, option, true);
      break;

    case 705:// --no-qcf-skip-rd
      options->assignBool(options::qcfSkipRd, option, false);
      break;

    case 706:// --qcf-tconstraint
      options->assignBool(options::qcfTConstraint, option, true);
      break;

    case 707:// --no-qcf-tconstraint
      options->assignBool(options::qcfTConstraint, option, false);
      break;

    case 708:// --qcf-vo-exp
      options->assignBool(options::qcfVoExp, option, true);
      break;

    case 709:// --no-qcf-vo-exp
      options->assignBool(options::qcfVoExp, option, false);
      break;

    case 710:// --quant-alpha-equiv
      options->assignBool(options::quantAlphaEquiv, option, true);
      break;

    case 711:// --no-quant-alpha-equiv
      options->assignBool(options::quantAlphaEquiv, option, false);
      break;

    case 712:// --quant-anti-skolem
      options->assignBool(options::quantAntiSkolem, option, true);
      break;

    case 713:// --no-quant-anti-skolem
      options->assignBool(options::quantAntiSkolem, option, false);
      break;

    case 714:// --quant-cf
      options->assignBool(options::quantConflictFind, option, true);
      break;

    case 715:// --no-quant-cf
      options->assignBool(options::quantConflictFind, option, false);
      break;

    case 716:// --quant-cf-mode=MODE
      options->assign(options::qcfMode, option, optionarg);
      break;

    case 717:// --quant-cf-when=MODE
      options->assign(options::qcfWhenMode, option, optionarg);
      break;

    case 718:// --quant-dsplit-mode=MODE
      options->assign(options::quantDynamicSplit, option, optionarg);
      break;

    case 719:// --quant-epr
      options->assignBool(options::quantEpr, option, true);
      break;

    case 720:// --no-quant-epr
      options->assignBool(options::quantEpr, option, false);
      break;

    case 721:// --quant-epr-match
      options->assignBool(options::quantEprMatching, option, true);
      break;

    case 722:// --no-quant-epr-match
      options->assignBool(options::quantEprMatching, option, false);
      break;

    case 723:// --quant-fun-wd
      options->assignBool(options::quantFunWellDefined, option, true);
      break;

    case 724:// --no-quant-fun-wd
      options->assignBool(options::quantFunWellDefined, option, false);
      break;

    case 725:// --quant-ind
      options->assignBool(options::quantInduction, option, true);
      break;

    case 726:// --no-quant-ind
      options->assignBool(options::quantInduction, option, false);
      break;

    case 727:// --quant-model-ee
      options->assignBool(options::quantModelEe, option, true);
      break;

    case 728:// --no-quant-model-ee
      options->assignBool(options::quantModelEe, option, false);
      break;

    case 729:// --quant-rep-mode=MODE
      options->assign(options::quantRepMode, option, optionarg);
      break;

    case 730:// --quant-split
      options->assignBool(options::quantSplit, option, true);
      break;

    case 731:// --no-quant-split
      options->assignBool(options::quantSplit, option, false);
      break;

    case 732:// --register-quant-body-terms
      options->assignBool(options::registerQuantBodyTerms, option, true);
      break;

    case 733:// --no-register-quant-body-terms
      options->assignBool(options::registerQuantBodyTerms, option, false);
      break;

    case 734:// --relational-triggers
      options->assignBool(options::relationalTriggers, option, true);
      break;

    case 735:// --no-relational-triggers
      options->assignBool(options::relationalTriggers, option, false);
      break;

    case 736:// --relevant-triggers
      options->assignBool(options::relevantTriggers, option, true);
      break;

    case 737:// --no-relevant-triggers
      options->assignBool(options::relevantTriggers, option, false);
      break;

    case 738:// --strict-triggers
      options->assignBool(options::strictTriggers, option, true);
      break;

    case 739:// --no-strict-triggers
      options->assignBool(options::strictTriggers, option, false);
      break;

    case 740:// --sygus
      options->assignBool(options::sygus, option, true);
      break;

    case 741:// --no-sygus
      options->assignBool(options::sygus, option, false);
      break;

    case 742:// --sygus-active-gen-cfactor=N
      options->assign(options::sygusActiveGenEnumConsts, option, optionarg);
      break;

    case 743:// --sygus-active-gen=MODE
      options->assign(options::sygusActiveGenMode, option, optionarg);
      break;

    case 744:// --sygus-add-const-grammar
      options->assignBool(options::sygusAddConstGrammar, option, true);
      break;

    case 745:// --no-sygus-add-const-grammar
      options->assignBool(options::sygusAddConstGrammar, option, false);
      break;

    case 746:// --sygus-arg-relevant
      options->assignBool(options::sygusArgRelevant, option, true);
      break;

    case 747:// --no-sygus-arg-relevant
      options->assignBool(options::sygusArgRelevant, option, false);
      break;

    case 748:// --sygus-auto-unfold
      options->assignBool(options::sygusInvAutoUnfold, option, true);
      break;

    case 749:// --no-sygus-auto-unfold
      options->assignBool(options::sygusInvAutoUnfold, option, false);
      break;

    case 750:// --sygus-bool-ite-return-const
      options->assignBool(options::sygusBoolIteReturnConst, option, true);
      break;

    case 751:// --no-sygus-bool-ite-return-const
      options->assignBool(options::sygusBoolIteReturnConst, option, false);
      break;

    case 752:// --sygus-core-connective
      options->assignBool(options::sygusCoreConnective, option, true);
      break;

    case 753:// --no-sygus-core-connective
      options->assignBool(options::sygusCoreConnective, option, false);
      break;

    case 754:// --sygus-crepair-abort
      options->assignBool(options::sygusConstRepairAbort, option, true);
      break;

    case 755:// --no-sygus-crepair-abort
      options->assignBool(options::sygusConstRepairAbort, option, false);
      break;

    case 756:// --sygus-eval-opt
      options->assignBool(options::sygusEvalOpt, option, true);
      break;

    case 757:// --no-sygus-eval-opt
      options->assignBool(options::sygusEvalOpt, option, false);
      break;

    case 758:// --sygus-eval-unfold
      options->assignBool(options::sygusEvalUnfold, option, true);
      break;

    case 759:// --no-sygus-eval-unfold
      options->assignBool(options::sygusEvalUnfold, option, false);
      break;

    case 760:// --sygus-eval-unfold-bool
      options->assignBool(options::sygusEvalUnfoldBool, option, true);
      break;

    case 761:// --no-sygus-eval-unfold-bool
      options->assignBool(options::sygusEvalUnfoldBool, option, false);
      break;

    case 762:// --sygus-expr-miner-check-timeout=N
      options->assign(options::sygusExprMinerCheckTimeout, option, optionarg);
      break;

    case 763:// --sygus-expr-miner-check-use-export
      options->assignBool(options::sygusExprMinerCheckUseExport, option, true);
      break;

    case 764:// --no-sygus-expr-miner-check-use-export
      options->assignBool(options::sygusExprMinerCheckUseExport, option, false);
      break;

    case 765:// --sygus-ext-rew
      options->assignBool(options::sygusExtRew, option, true);
      break;

    case 766:// --no-sygus-ext-rew
      options->assignBool(options::sygusExtRew, option, false);
      break;

    case 767:// --sygus-filter-sol-rev
      options->assignBool(options::sygusFilterSolRevSubsume, option, true);
      break;

    case 768:// --no-sygus-filter-sol-rev
      options->assignBool(options::sygusFilterSolRevSubsume, option, false);
      break;

    case 769:// --sygus-filter-sol=MODE
      options->assign(options::sygusFilterSolMode, option, optionarg);
      break;

    case 770:// --sygus-grammar-cons=MODE
      options->assign(options::sygusGrammarConsMode, option, optionarg);
      break;

    case 771:// --sygus-grammar-norm
      options->assignBool(options::sygusGrammarNorm, option, true);
      break;

    case 772:// --no-sygus-grammar-norm
      options->assignBool(options::sygusGrammarNorm, option, false);
      break;

    case 773:// --sygus-inference
      options->assignBool(options::sygusInference, option, true);
      break;

    case 774:// --no-sygus-inference
      options->assignBool(options::sygusInference, option, false);
      break;

    case 775:// --sygus-inst
      options->assignBool(options::sygusInst, option, true);
      break;

    case 776:// --no-sygus-inst
      options->assignBool(options::sygusInst, option, false);
      break;

    case 777:// --sygus-inv-templ-when-sg
      options->assignBool(options::sygusInvTemplWhenSyntax, option, true);
      break;

    case 778:// --no-sygus-inv-templ-when-sg
      options->assignBool(options::sygusInvTemplWhenSyntax, option, false);
      break;

    case 779:// --sygus-inv-templ=MODE
      options->assign(options::sygusInvTemplMode, option, optionarg);
      break;

    case 780:// --sygus-min-grammar
      options->assignBool(options::sygusMinGrammar, option, true);
      break;

    case 781:// --no-sygus-min-grammar
      options->assignBool(options::sygusMinGrammar, option, false);
      break;

    case 782:// --sygus-pbe
      options->assignBool(options::sygusUnifPbe, option, true);
      break;

    case 783:// --no-sygus-pbe
      options->assignBool(options::sygusUnifPbe, option, false);
      break;

    case 784:// --sygus-pbe-multi-fair
      options->assignBool(options::sygusPbeMultiFair, option, true);
      break;

    case 785:// --no-sygus-pbe-multi-fair
      options->assignBool(options::sygusPbeMultiFair, option, false);
      break;

    case 786:// --sygus-pbe-multi-fair-diff=N
      options->assign(options::sygusPbeMultiFairDiff, option, optionarg);
      break;

    case 787:// --sygus-qe-preproc
      options->assignBool(options::sygusQePreproc, option, true);
      break;

    case 788:// --no-sygus-qe-preproc
      options->assignBool(options::sygusQePreproc, option, false);
      break;

    case 789:// --sygus-query-gen
      options->assignBool(options::sygusQueryGen, option, true);
      break;

    case 790:// --no-sygus-query-gen
      options->assignBool(options::sygusQueryGen, option, false);
      break;

    case 791:// --sygus-query-gen-check
      options->assignBool(options::sygusQueryGenCheck, option, true);
      break;

    case 792:// --no-sygus-query-gen-check
      options->assignBool(options::sygusQueryGenCheck, option, false);
      break;

    case 793:// --sygus-query-gen-dump-files=MODE
      options->assign(options::sygusQueryGenDumpFiles, option, optionarg);
      break;

    case 794:// --sygus-query-gen-thresh=N
      options->assign(options::sygusQueryGenThresh, option, optionarg);
      break;

    case 795:// --sygus-rec-fun
      options->assignBool(options::sygusRecFun, option, true);
      break;

    case 796:// --no-sygus-rec-fun
      options->assignBool(options::sygusRecFun, option, false);
      break;

    case 797:// --sygus-rec-fun-eval-limit=N
      options->assign(options::sygusRecFunEvalLimit, option, optionarg);
      break;

    case 798:// --sygus-repair-const
      options->assignBool(options::sygusRepairConst, option, true);
      break;

    case 799:// --no-sygus-repair-const
      options->assignBool(options::sygusRepairConst, option, false);
      break;

    case 800:// --sygus-repair-const-timeout=N
      options->assign(options::sygusRepairConstTimeout, option, optionarg);
      break;

    case 801:// --sygus-rr
      options->assignBool(options::sygusRew, option, true);
      break;

    case 802:// --no-sygus-rr
      options->assignBool(options::sygusRew, option, false);
      break;

    case 803:// --sygus-rr-synth
      options->assignBool(options::sygusRewSynth, option, true);
      break;

    case 804:// --no-sygus-rr-synth
      options->assignBool(options::sygusRewSynth, option, false);
      break;

    case 805:// --sygus-rr-synth-accel
      options->assignBool(options::sygusRewSynthAccel, option, true);
      break;

    case 806:// --no-sygus-rr-synth-accel
      options->assignBool(options::sygusRewSynthAccel, option, false);
      break;

    case 807:// --sygus-rr-synth-check
      options->assignBool(options::sygusRewSynthCheck, option, true);
      break;

    case 808:// --no-sygus-rr-synth-check
      options->assignBool(options::sygusRewSynthCheck, option, false);
      break;

    case 809:// --sygus-rr-synth-filter-cong
      options->assignBool(options::sygusRewSynthFilterCong, option, true);
      break;

    case 810:// --no-sygus-rr-synth-filter-cong
      options->assignBool(options::sygusRewSynthFilterCong, option, false);
      break;

    case 811:// --sygus-rr-synth-filter-match
      options->assignBool(options::sygusRewSynthFilterMatch, option, true);
      break;

    case 812:// --no-sygus-rr-synth-filter-match
      options->assignBool(options::sygusRewSynthFilterMatch, option, false);
      break;

    case 813:// --sygus-rr-synth-filter-nl
      options->assignBool(options::sygusRewSynthFilterNonLinear, option, true);
      break;

    case 814:// --no-sygus-rr-synth-filter-nl
      options->assignBool(options::sygusRewSynthFilterNonLinear, option, false);
      break;

    case 815:// --sygus-rr-synth-filter-order
      options->assignBool(options::sygusRewSynthFilterOrder, option, true);
      break;

    case 816:// --no-sygus-rr-synth-filter-order
      options->assignBool(options::sygusRewSynthFilterOrder, option, false);
      break;

    case 817:// --sygus-rr-synth-input
      options->assignBool(options::sygusRewSynthInput, option, true);
      break;

    case 818:// --no-sygus-rr-synth-input
      options->assignBool(options::sygusRewSynthInput, option, false);
      break;

    case 819:// --sygus-rr-synth-input-nvars=N
      options->assign(options::sygusRewSynthInputNVars, option, optionarg);
      break;

    case 820:// --sygus-rr-synth-input-use-bool
      options->assignBool(options::sygusRewSynthInputUseBool, option, true);
      break;

    case 821:// --no-sygus-rr-synth-input-use-bool
      options->assignBool(options::sygusRewSynthInputUseBool, option, false);
      break;

    case 822:// --sygus-rr-synth-rec
      options->assignBool(options::sygusRewSynthRec, option, true);
      break;

    case 823:// --no-sygus-rr-synth-rec
      options->assignBool(options::sygusRewSynthRec, option, false);
      break;

    case 824:// --sygus-rr-verify
      options->assignBool(options::sygusRewVerify, option, true);
      break;

    case 825:// --no-sygus-rr-verify
      options->assignBool(options::sygusRewVerify, option, false);
      break;

    case 826:// --sygus-rr-verify-abort
      options->assignBool(options::sygusRewVerifyAbort, option, true);
      break;

    case 827:// --no-sygus-rr-verify-abort
      options->assignBool(options::sygusRewVerifyAbort, option, false);
      break;

    case 828:// --sygus-sample-fp-uniform
      options->assignBool(options::sygusSampleFpUniform, option, true);
      break;

    case 829:// --no-sygus-sample-fp-uniform
      options->assignBool(options::sygusSampleFpUniform, option, false);
      break;

    case 830:// --sygus-sample-grammar
      options->assignBool(options::sygusSampleGrammar, option, true);
      break;

    case 831:// --no-sygus-sample-grammar
      options->assignBool(options::sygusSampleGrammar, option, false);
      break;

    case 832:// --sygus-samples=N
      options->assign(options::sygusSamples, option, optionarg);
      break;

    case 833:// --sygus-si-abort
      options->assignBool(options::cegqiSingleInvAbort, option, true);
      break;

    case 834:// --no-sygus-si-abort
      options->assignBool(options::cegqiSingleInvAbort, option, false);
      break;

    case 835:// --sygus-si-partial
      options->assignBool(options::cegqiSingleInvPartial, option, true);
      break;

    case 836:// --no-sygus-si-partial
      options->assignBool(options::cegqiSingleInvPartial, option, false);
      break;

    case 837:// --sygus-si-rcons-limit=N
      options->assign(options::cegqiSingleInvReconstructLimit, option, optionarg);
      break;

    case 838:// --sygus-si-rcons=MODE
      options->assign(options::cegqiSingleInvReconstruct, option, optionarg);
      break;

    case 839:// --sygus-si-reconstruct-const
      options->assignBool(options::cegqiSingleInvReconstructConst, option, true);
      break;

    case 840:// --no-sygus-si-reconstruct-const
      options->assignBool(options::cegqiSingleInvReconstructConst, option, false);
      break;

    case 841:// --sygus-si=MODE
      options->assign(options::cegqiSingleInvMode, option, optionarg);
      break;

    case 842:// --sygus-stream
      options->assignBool(options::sygusStream, option, true);
      break;

    case 843:// --no-sygus-stream
      options->assignBool(options::sygusStream, option, false);
      break;

    case 844:// --sygus-templ-embed-grammar
      options->assignBool(options::sygusTemplEmbedGrammar, option, true);
      break;

    case 845:// --no-sygus-templ-embed-grammar
      options->assignBool(options::sygusTemplEmbedGrammar, option, false);
      break;

    case 846:// --sygus-unif-cond-independent-no-repeat-sol
      options->assignBool(options::sygusUnifCondIndNoRepeatSol, option, true);
      break;

    case 847:// --no-sygus-unif-cond-independent-no-repeat-sol
      options->assignBool(options::sygusUnifCondIndNoRepeatSol, option, false);
      break;

    case 848:// --sygus-unif-pi=MODE
      options->assign(options::sygusUnifPi, option, optionarg);
      break;

    case 849:// --sygus-unif-shuffle-cond
      options->assignBool(options::sygusUnifShuffleCond, option, true);
      break;

    case 850:// --no-sygus-unif-shuffle-cond
      options->assignBool(options::sygusUnifShuffleCond, option, false);
      break;

    case 851:// --sygus-verify-subcall
      options->assignBool(options::sygusVerifySubcall, option, true);
      break;

    case 852:// --no-sygus-verify-subcall
      options->assignBool(options::sygusVerifySubcall, option, false);
      break;

    case 853:// --term-db-mode=MODE
      options->assign(options::termDbMode, option, optionarg);
      break;

    case 854:// --track-inst-lemmas
      options->assignBool(options::trackInstLemmas, option, true);
      break;

    case 855:// --no-track-inst-lemmas
      options->assignBool(options::trackInstLemmas, option, false);
      break;

    case 856:// --trigger-active-sel=MODE
      options->assign(options::triggerActiveSelMode, option, optionarg);
      break;

    case 857:// --trigger-sel=MODE
      options->assign(options::triggerSelMode, option, optionarg);
      break;

    case 858:// --user-pat=MODE
      options->assign(options::userPatternsQuant, option, optionarg);
      break;

    case 859:// --var-elim-quant
      options->assignBool(options::varElimQuant, option, true);
      break;

    case 860:// --no-var-elim-quant
      options->assignBool(options::varElimQuant, option, false);
      break;

    case 861:// --var-ineq-elim-quant
      options->assignBool(options::varIneqElimQuant, option, true);
      break;

    case 862:// --no-var-ineq-elim-quant
      options->assignBool(options::varIneqElimQuant, option, false);
      break;

    case 863:// --sep-check-neg
      options->assignBool(options::sepCheckNeg, option, true);
      break;

    case 864:// --no-sep-check-neg
      options->assignBool(options::sepCheckNeg, option, false);
      break;

    case 865:// --sep-child-refine
      options->assignBool(options::sepChildRefine, option, true);
      break;

    case 866:// --no-sep-child-refine
      options->assignBool(options::sepChildRefine, option, false);
      break;

    case 867:// --sep-deq-c
      options->assignBool(options::sepDisequalC, option, true);
      break;

    case 868:// --no-sep-deq-c
      options->assignBool(options::sepDisequalC, option, false);
      break;

    case 869:// --sep-exp
      options->assignBool(options::sepExp, option, true);
      break;

    case 870:// --no-sep-exp
      options->assignBool(options::sepExp, option, false);
      break;

    case 871:// --sep-min-refine
      options->assignBool(options::sepMinimalRefine, option, true);
      break;

    case 872:// --no-sep-min-refine
      options->assignBool(options::sepMinimalRefine, option, false);
      break;

    case 873:// --sep-pre-skolem-emp
      options->assignBool(options::sepPreSkolemEmp, option, true);
      break;

    case 874:// --no-sep-pre-skolem-emp
      options->assignBool(options::sepPreSkolemEmp, option, false);
      break;

    case 875:// --sets-ext
      options->assignBool(options::setsExt, option, true);
      break;

    case 876:// --no-sets-ext
      options->assignBool(options::setsExt, option, false);
      break;

    case 877:// --sets-infer-as-lemmas
      options->assignBool(options::setsInferAsLemmas, option, true);
      break;

    case 878:// --no-sets-infer-as-lemmas
      options->assignBool(options::setsInferAsLemmas, option, false);
      break;

    case 879:// --sets-proxy-lemmas
      options->assignBool(options::setsProxyLemmas, option, true);
      break;

    case 880:// --no-sets-proxy-lemmas
      options->assignBool(options::setsProxyLemmas, option, false);
      break;

    case 881:// --abstract-values
      options->assignBool(options::abstractValues, option, true);
      break;

    case 882:// --no-abstract-values
      options->assignBool(options::abstractValues, option, false);
      break;

    case 883:// --ackermann
      options->assignBool(options::ackermann, option, true);
      break;

    case 884:// --no-ackermann
      options->assignBool(options::ackermann, option, false);
      break;

    case 885:// --bitblast-step=N
      options->assign(options::bitblastStep, option, optionarg);
      break;

    case 886:// --block-models=MODE
      options->assign(options::blockModelsMode, option, optionarg);
      break;

    case 887:// --bv-eager-assert-step=N
      options->assign(options::bvEagerAssertStep, option, optionarg);
      break;

    case 888:// --bv-propagation-step=N
      options->assign(options::bvPropagationStep, option, optionarg);
      break;

    case 889:// --bv-sat-conflict-step=N
      options->assign(options::bvSatConflictStep, option, optionarg);
      break;

    case 890:// --check-abducts
      options->assignBool(options::checkAbducts, option, true);
      break;

    case 891:// --no-check-abducts
      options->assignBool(options::checkAbducts, option, false);
      break;

    case 892:// --check-models
      options->assignBool(options::checkModels, option, true);
      break;

    case 893:// --no-check-models
      options->assignBool(options::checkModels, option, false);
      break;

    case 894:// --check-proofs
      options->assignBool(options::checkProofs, option, true);
      break;

    case 895:// --no-check-proofs
      options->assignBool(options::checkProofs, option, false);
      break;

    case 896:// --check-synth-sol
      options->assignBool(options::checkSynthSol, option, true);
      break;

    case 897:// --no-check-synth-sol
      options->assignBool(options::checkSynthSol, option, false);
      break;

    case 898:// --check-unsat-cores
      options->assignBool(options::checkUnsatCores, option, true);
      break;

    case 899:// --no-check-unsat-cores
      options->assignBool(options::checkUnsatCores, option, false);
      break;

    case 900:// --cnf-step=N
      options->assign(options::cnfStep, option, optionarg);
      break;

    case 901:// --cpu-time
      options->assignBool(options::cpuTime, option, true);
      break;

    case 902:// --no-cpu-time
      options->assignBool(options::cpuTime, option, false);
      break;

    case 903:// --debug-check-models
      options->assignBool(options::debugCheckModels, option, true);
      break;

    case 904:// --no-debug-check-models
      options->assignBool(options::debugCheckModels, option, false);
      break;

    case 905:// --decision-step=N
      options->assign(options::decisionStep, option, optionarg);
      break;

    case 906:// --dump-instantiations
      options->assignBool(options::dumpInstantiations, option, true);
      break;

    case 907:// --no-dump-instantiations
      options->assignBool(options::dumpInstantiations, option, false);
      break;

    case 908:// --dump-models
      options->assignBool(options::dumpModels, option, true);
      break;

    case 909:// --no-dump-models
      options->assignBool(options::dumpModels, option, false);
      break;

    case 910:// --dump-proofs
      options->assignBool(options::dumpProofs, option, true);
      break;

    case 911:// --no-dump-proofs
      options->assignBool(options::dumpProofs, option, false);
      break;

    case 912:// --dump-synth
      options->assignBool(options::dumpSynth, option, true);
      break;

    case 913:// --no-dump-synth
      options->assignBool(options::dumpSynth, option, false);
      break;

    case 914:// --dump-to=FILE
      options->assign(options::dumpToFileName, option, optionarg);
      break;

    case 915:// --dump-unsat-cores
      options->assignBool(options::dumpUnsatCores, option, true);
      break;

    case 916:// --no-dump-unsat-cores
      options->assignBool(options::dumpUnsatCores, option, false);
      break;

    case 917:// --dump-unsat-cores-full
      options->assignBool(options::dumpUnsatCoresFull, option, true);
      break;

    case 918:// --no-dump-unsat-cores-full
      options->assignBool(options::dumpUnsatCoresFull, option, false);
      break;

    case 919:// --dump=MODE
      options->assign(options::dumpModeString, option, optionarg);
      break;

    case 920:// --ext-rew-prep
      options->assignBool(options::extRewPrep, option, true);
      break;

    case 921:// --no-ext-rew-prep
      options->assignBool(options::extRewPrep, option, false);
      break;

    case 922:// --ext-rew-prep-agg
      options->assignBool(options::extRewPrepAgg, option, true);
      break;

    case 923:// --no-ext-rew-prep-agg
      options->assignBool(options::extRewPrepAgg, option, false);
      break;

    case 924:// --force-no-limit-cpu-while-dump
      options->assignBool(options::forceNoLimitCpuWhileDump, option, true);
      break;

    case 925:// --no-force-no-limit-cpu-while-dump
      options->assignBool(options::forceNoLimitCpuWhileDump, option, false);
      break;

    case 926:// --hard-limit
      options->assignBool(options::hardLimit, option, true);
      break;

    case 927:// --no-hard-limit
      options->assignBool(options::hardLimit, option, false);
      break;

    case 'i':
    case 928:// --incremental
      options->assignBool(options::incrementalSolving, option, true);
      break;

    case 929:// --no-incremental
      options->assignBool(options::incrementalSolving, option, false);
      break;

    case 930:// --ite-simp
      options->assignBool(options::doITESimp, option, true);
      break;

    case 931:// --no-ite-simp
      options->assignBool(options::doITESimp, option, false);
      break;

    case 932:// --lemma-step=N
      options->assign(options::lemmaStep, option, optionarg);
      break;

    case 933:// --model-cores=MODE
      options->assign(options::modelCoresMode, option, optionarg);
      break;

    case 934:// --model-u-dt-enum
      options->assignBool(options::modelUninterpDtEnum, option, true);
      break;

    case 935:// --no-model-u-dt-enum
      options->assignBool(options::modelUninterpDtEnum, option, false);
      break;

    case 936:// --model-witness-value
      options->assignBool(options::modelWitnessValue, option, true);
      break;

    case 937:// --no-model-witness-value
      options->assignBool(options::modelWitnessValue, option, false);
      break;

    case 938:// --on-repeat-ite-simp
      options->assignBool(options::doITESimpOnRepeat, option, true);
      break;

    case 939:// --no-on-repeat-ite-simp
      options->assignBool(options::doITESimpOnRepeat, option, false);
      break;

    case 940:// --parse-step=N
      options->assign(options::parseStep, option, optionarg);
      break;

    case 941:// --preprocess-step=N
      options->assign(options::preprocessStep, option, optionarg);
      break;

    case 942:// --produce-abducts
      options->assignBool(options::produceAbducts, option, true);
      break;

    case 943:// --no-produce-abducts
      options->assignBool(options::produceAbducts, option, false);
      break;

    case 944:// --produce-assertions
      options->assignBool(options::produceAssertions, option, true);
      break;

    case 945:// --no-produce-assertions
      options->assignBool(options::produceAssertions, option, false);
      break;

    case 946:// --produce-assignments
      options->assignBool(options::produceAssignments, option, true);
      break;

    case 947:// --no-produce-assignments
      options->assignBool(options::produceAssignments, option, false);
      break;

    case 'm':
    case 948:// --produce-models
      options->assignBool(options::produceModels, option, true);
      break;

    case 949:// --no-produce-models
      options->assignBool(options::produceModels, option, false);
      break;

    case 950:// --produce-unsat-assumptions
      options->assignBool(options::unsatAssumptions, option, true);
      break;

    case 951:// --no-produce-unsat-assumptions
      options->assignBool(options::unsatAssumptions, option, false);
      break;

    case 952:// --produce-unsat-cores
      options->assignBool(options::unsatCores, option, true);
      break;

    case 953:// --no-produce-unsat-cores
      options->assignBool(options::unsatCores, option, false);
      break;

    case 954:// --proof
      options->assignBool(options::proof, option, true);
      break;

    case 955:// --no-proof
      options->assignBool(options::proof, option, false);
      break;

    case 956:// --quantifier-step=N
      options->assign(options::quantifierStep, option, optionarg);
      break;

    case 957:// --repeat-simp
      options->assignBool(options::repeatSimp, option, true);
      break;

    case 958:// --no-repeat-simp
      options->assignBool(options::repeatSimp, option, false);
      break;

    case 959:// --restart-step=N
      options->assign(options::restartStep, option, optionarg);
      break;

    case 960:// --rewrite-step=N
      options->assign(options::rewriteStep, option, optionarg);
      break;

    case 961:// --rlimit-per=N
      options->assign(options::perCallResourceLimit, option, optionarg);
      break;

    case 962:// --rlimit=N
      options->assign(options::cumulativeResourceLimit, option, optionarg);
      break;

    case 963:// --sat-conflict-step=N
      options->assign(options::satConflictStep, option, optionarg);
      break;

    case 964:// --simp-ite-compress
      options->assignBool(options::compressItes, option, true);
      break;

    case 965:// --no-simp-ite-compress
      options->assignBool(options::compressItes, option, false);
      break;

    case 966:// --simp-ite-hunt-zombies=N
      options->assign(options::zombieHuntThreshold, option, optionarg);
      break;

    case 967:// --simp-with-care
      options->assignBool(options::simplifyWithCareEnabled, option, true);
      break;

    case 968:// --no-simp-with-care
      options->assignBool(options::simplifyWithCareEnabled, option, false);
      break;

    case 969:// --simplification=MODE
      options->assign(options::simplificationMode, option, optionarg);
      break;

    case 970:// --solve-bv-as-int=N
      options->assign(options::solveBVAsInt, option, optionarg);
      break;

    case 971:// --solve-int-as-bv=N
      options->assign(options::solveIntAsBV, option, optionarg);
      break;

    case 972:// --solve-real-as-int
      options->assignBool(options::solveRealAsInt, option, true);
      break;

    case 973:// --no-solve-real-as-int
      options->assignBool(options::solveRealAsInt, option, false);
      break;

    case 974:// --sort-inference
      options->assignBool(options::sortInference, option, true);
      break;

    case 975:// --no-sort-inference
      options->assignBool(options::sortInference, option, false);
      break;

    case 976:// --static-learning
      options->assignBool(options::doStaticLearning, option, true);
      break;

    case 977:// --no-static-learning
      options->assignBool(options::doStaticLearning, option, false);
      break;

    case 978:// --sygus-out=MODE
      options->assign(options::sygusOut, option, optionarg);
      break;

    case 979:// --sygus-print-callbacks
      options->assignBool(options::sygusPrintCallbacks, option, true);
      break;

    case 980:// --no-sygus-print-callbacks
      options->assignBool(options::sygusPrintCallbacks, option, false);
      break;

    case 981:// --theory-check-step=N
      options->assign(options::theoryCheckStep, option, optionarg);
      break;

    case 982:// --tlimit-per=MS
      options->assign(options::perCallMillisecondLimit, option, optionarg);
      break;

    case 983:// --tlimit=MS
      options->assign(options::cumulativeMillisecondLimit, option, optionarg);
      break;

    case 984:// --unconstrained-simp
      options->assignBool(options::unconstrainedSimp, option, true);
      break;

    case 985:// --no-unconstrained-simp
      options->assignBool(options::unconstrainedSimp, option, false);
      break;

    case 986:// --re-elim
      options->assignBool(options::regExpElim, option, true);
      break;

    case 987:// --no-re-elim
      options->assignBool(options::regExpElim, option, false);
      break;

    case 988:// --re-elim-agg
      options->assignBool(options::regExpElimAgg, option, true);
      break;

    case 989:// --no-re-elim-agg
      options->assignBool(options::regExpElimAgg, option, false);
      break;

    case 990:// --re-inter-mode=MODE
      options->assign(options::stringRegExpInterMode, option, optionarg);
      break;

    case 991:// --strings-check-entail-len
      options->assignBool(options::stringCheckEntailLen, option, true);
      break;

    case 992:// --no-strings-check-entail-len
      options->assignBool(options::stringCheckEntailLen, option, false);
      break;

    case 993:// --strings-eager
      options->assignBool(options::stringEager, option, true);
      break;

    case 994:// --no-strings-eager
      options->assignBool(options::stringEager, option, false);
      break;

    case 995:// --strings-eager-len
      options->assignBool(options::stringEagerLen, option, true);
      break;

    case 996:// --no-strings-eager-len
      options->assignBool(options::stringEagerLen, option, false);
      break;

    case 997:// --strings-exp
      options->assignBool(options::stringExp, option, true);
      break;

    case 998:// --no-strings-exp
      options->assignBool(options::stringExp, option, false);
      break;

    case 999:// --strings-ff
      options->assignBool(options::stringFlatForms, option, true);
      break;

    case 1000:// --no-strings-ff
      options->assignBool(options::stringFlatForms, option, false);
      break;

    case 1001:// --strings-fmf
      options->assignBool(options::stringFMF, option, true);
      break;

    case 1002:// --no-strings-fmf
      options->assignBool(options::stringFMF, option, false);
      break;

    case 1003:// --strings-guess-model
      options->assignBool(options::stringGuessModel, option, true);
      break;

    case 1004:// --no-strings-guess-model
      options->assignBool(options::stringGuessModel, option, false);
      break;

    case 1005:// --strings-infer-as-lemmas
      options->assignBool(options::stringInferAsLemmas, option, true);
      break;

    case 1006:// --no-strings-infer-as-lemmas
      options->assignBool(options::stringInferAsLemmas, option, false);
      break;

    case 1007:// --strings-infer-sym
      options->assignBool(options::stringInferSym, option, true);
      break;

    case 1008:// --no-strings-infer-sym
      options->assignBool(options::stringInferSym, option, false);
      break;

    case 1009:// --strings-inm
      options->assignBool(options::stringIgnNegMembership, option, true);
      break;

    case 1010:// --no-strings-inm
      options->assignBool(options::stringIgnNegMembership, option, false);
      break;

    case 1011:// --strings-lazy-pp
      options->assignBool(options::stringLazyPreproc, option, true);
      break;

    case 1012:// --no-strings-lazy-pp
      options->assignBool(options::stringLazyPreproc, option, false);
      break;

    case 1013:// --strings-len-norm
      options->assignBool(options::stringLenNorm, option, true);
      break;

    case 1014:// --no-strings-len-norm
      options->assignBool(options::stringLenNorm, option, false);
      break;

    case 1015:// --strings-lprop-csp
      options->assignBool(options::stringLenPropCsp, option, true);
      break;

    case 1016:// --no-strings-lprop-csp
      options->assignBool(options::stringLenPropCsp, option, false);
      break;

    case 1017:// --strings-min-prefix-explain
      options->assignBool(options::stringMinPrefixExplain, option, true);
      break;

    case 1018:// --no-strings-min-prefix-explain
      options->assignBool(options::stringMinPrefixExplain, option, false);
      break;

    case 1019:// --strings-print-ascii
      options->assignBool(options::stdPrintASCII, option, true);
      break;

    case 1020:// --no-strings-print-ascii
      options->assignBool(options::stdPrintASCII, option, false);
      break;

    case 1021:// --strings-process-loop-mode=MODE
      options->assign(options::stringProcessLoopMode, option, optionarg);
      break;

    case 1022:// --strings-rexplain-lemmas
      options->assignBool(options::stringRExplainLemmas, option, true);
      break;

    case 1023:// --no-strings-rexplain-lemmas
      options->assignBool(options::stringRExplainLemmas, option, false);
      break;

    case 1024:// --assign-function-values
      options->assignBool(options::assignFunctionValues, option, true);
      break;

    case 1025:// --no-assign-function-values
      options->assignBool(options::assignFunctionValues, option, false);
      break;

    case 1026:// --condense-function-values
      options->assignBool(options::condenseFunctionValues, option, true);
      break;

    case 1027:// --no-condense-function-values
      options->assignBool(options::condenseFunctionValues, option, false);
      break;

    case 1028:// --theoryof-mode=MODE
      options->assign(options::theoryOfMode, option, optionarg);
      break;

    case 1029:// --symmetry-breaker
      options->assignBool(options::ufSymmetryBreaker, option, true);
      break;

    case 1030:// --no-symmetry-breaker
      options->assignBool(options::ufSymmetryBreaker, option, false);
      break;

    case 1031:// --uf-ho
      options->assignBool(options::ufHo, option, true);
      break;

    case 1032:// --no-uf-ho
      options->assignBool(options::ufHo, option, false);
      break;

    case 1033:// --uf-ho-ext
      options->assignBool(options::ufHoExt, option, true);
      break;

    case 1034:// --no-uf-ho-ext
      options->assignBool(options::ufHoExt, option, false);
      break;

    case 1035:// --uf-ss-abort-card=N
      options->assign(options::ufssAbortCardinality, option, optionarg);
      break;

    case 1036:// --uf-ss-fair
      options->assignBool(options::ufssFairness, option, true);
      break;

    case 1037:// --no-uf-ss-fair
      options->assignBool(options::ufssFairness, option, false);
      break;

    case 1038:// --uf-ss-fair-monotone
      options->assignBool(options::ufssFairnessMonotone, option, true);
      break;

    case 1039:// --no-uf-ss-fair-monotone
      options->assignBool(options::ufssFairnessMonotone, option, false);
      break;

    case 1040:// --uf-ss-totality
      options->assignBool(options::ufssTotality, option, true);
      break;

    case 1041:// --no-uf-ss-totality
      options->assignBool(options::ufssTotality, option, false);
      break;

    case 1042:// --uf-ss-totality-limited=N
      options->assign(options::ufssTotalityLimited, option, optionarg);
      break;

    case 1043:// --uf-ss-totality-sym-break
      options->assignBool(options::ufssTotalitySymBreak, option, true);
      break;

    case 1044:// --no-uf-ss-totality-sym-break
      options->assignBool(options::ufssTotalitySymBreak, option, false);
      break;

    case 1045:// --uf-ss=MODE
      options->assign(options::ufssMode, option, optionarg);
      break;



    case ':':
      // This can be a long or short option, and the way to get at the
      // name of it is different.
      throw OptionException(std::string("option `") + option +
                            "' missing its required argument");

    case '?':
    default:
      throw OptionException(std::string("can't understand option `") + option +
                            "'" + suggestCommandLineOptions(option));
    }
  }

  Debug("options") << "got " << nonoptions->size()
                   << " non-option arguments." << std::endl;
}

std::string Options::suggestCommandLineOptions(const std::string& optionName)
{
  DidYouMean didYouMean;

  const char* opt;
  for(size_t i = 0; (opt = cmdlineOptions[i].name) != NULL; ++i) {
    didYouMean.addWord(std::string("--") + cmdlineOptions[i].name);
  }

  return didYouMean.getMatchAsString(optionName.substr(0, optionName.find('=')));
}

static const char* smtOptions[] = {
  "approx-branch-depth=N",
  "approx-branch-depth=N",
  "arith-brab",
  "arith-brab",
  "arith-no-partial-fun",
  "arith-no-partial-fun",
  "arith-prop-clauses=N",
  "arith-prop-clauses=N",
  "arith-prop=MODE",
  "arith-prop=MODE",
  "arith-rewrite-equalities",
  "arith-rewrite-equalities",
  "collect-pivot-stats",
  "collect-pivot-stats",
  "cut-all-bounded",
  "cut-all-bounded",
  "dio-decomps",
  "dio-decomps",
  "dio-repeat",
  "dio-repeat",
  "dio-solver",
  "dio-solver",
  "dio-turns=N",
  "dio-turns=N",
  "error-selection-rule=RULE",
  "error-selection-rule=RULE",
  "fc-penalties",
  "fc-penalties",
  "heuristic-pivots=N",
  "heuristic-pivots=N",
  "lemmas-on-replay-failure",
  "lemmas-on-replay-failure",
  "maxCutsInContext=N",
  "maxCutsInContext=N",
  "miplib-trick",
  "miplib-trick",
  "miplib-trick-subs",
  "miplib-trick-subs",
  "new-prop",
  "new-prop",
  "nl-ext",
  "nl-ext",
  "nl-ext-ent-conf",
  "nl-ext-ent-conf",
  "nl-ext-factor",
  "nl-ext-factor",
  "nl-ext-inc-prec",
  "nl-ext-inc-prec",
  "nl-ext-purify",
  "nl-ext-purify",
  "nl-ext-rbound",
  "nl-ext-rbound",
  "nl-ext-rewrite",
  "nl-ext-rewrite",
  "nl-ext-split-zero",
  "nl-ext-split-zero",
  "nl-ext-tf-taylor-deg=N",
  "nl-ext-tf-taylor-deg=N",
  "nl-ext-tf-tplanes",
  "nl-ext-tf-tplanes",
  "nl-ext-tplanes",
  "nl-ext-tplanes",
  "nl-ext-tplanes-interleave",
  "nl-ext-tplanes-interleave",
  "pb-rewrites",
  "pb-rewrites",
  "pivot-threshold=N",
  "pivot-threshold=N",
  "pp-assert-max-sub-size=N",
  "pp-assert-max-sub-size=N",
  "prop-row-length=N",
  "prop-row-length=N",
  "replay-early-close-depth=N",
  "replay-early-close-depth=N",
  "replay-failure-penalty=N",
  "replay-failure-penalty=N",
  "replay-lemma-reject-cut=N",
  "replay-lemma-reject-cut=N",
  "replay-num-err-penalty=N",
  "replay-num-err-penalty=N",
  "replay-reject-cut=N",
  "replay-reject-cut=N",
  "replay-soi-major-threshold-pen=N",
  "replay-soi-major-threshold-pen=N",
  "replay-soi-major-threshold=T",
  "replay-soi-major-threshold=T",
  "replay-soi-minor-threshold-pen=N",
  "replay-soi-minor-threshold-pen=N",
  "replay-soi-minor-threshold=T",
  "replay-soi-minor-threshold=T",
  "restrict-pivots",
  "restrict-pivots",
  "revert-arith-models-on-unsat",
  "revert-arith-models-on-unsat",
  "rr-turns=N",
  "rr-turns=N",
  "se-solve-int",
  "se-solve-int",
  "simplex-check-period=N",
  "simplex-check-period=N",
  "soi-qe",
  "soi-qe",
  "standard-effort-variable-order-pivots=N",
  "standard-effort-variable-order-pivots=N",
  "unate-lemmas=MODE",
  "unate-lemmas=MODE",
  "use-approx",
  "use-approx",
  "use-fcsimplex",
  "use-fcsimplex",
  "use-soi",
  "use-soi",
  "arrays-config=N",
  "arrays-config=N",
  "arrays-eager-index",
  "arrays-eager-index",
  "arrays-eager-lemmas",
  "arrays-eager-lemmas",
  "arrays-lazy-rintro1",
  "arrays-lazy-rintro1",
  "arrays-model-based",
  "arrays-model-based",
  "arrays-optimize-linear",
  "arrays-optimize-linear",
  "arrays-prop=N",
  "arrays-prop=N",
  "arrays-reduce-sharing",
  "arrays-reduce-sharing",
  "arrays-weak-equiv",
  "arrays-weak-equiv",
  "debug",
  "input-language",
  "input-language",
  "output-language",
  "output-language",
  "parse-only",
  "parse-only",
  "preprocess-only",
  "preprocess-only",
  "print-success",
  "print-success",
  "quiet",
  "statistics",
  "statistics",
  "stats-every-query",
  "stats-every-query",
  "stats-hide-zeros",
  "stats-hide-zeros",
  "trace",
  "verbose",
  "verbosity",
  "verbosity",
  "bitblast-aig",
  "bitblast-aig",
  "bitblast",
  "bitblast",
  "bitwise-eq",
  "bitwise-eq",
  "bool-to-bv",
  "bool-to-bv",
  "bv-abstraction",
  "bv-abstraction",
  "bv-aig-simp=COMMAND",
  "bv-aig-simp=COMMAND",
  "bv-alg-extf",
  "bv-alg-extf",
  "bv-algebraic-budget=N",
  "bv-algebraic-budget=N",
  "bv-algebraic-solver",
  "bv-algebraic-solver",
  "bv-div-zero-const",
  "bv-div-zero-const",
  "bv-eager-explanations",
  "bv-eager-explanations",
  "bv-eq-slicer=MODE",
  "bv-eq-slicer=MODE",
  "bv-eq-solver",
  "bv-eq-solver",
  "bv-extract-arith",
  "bv-extract-arith",
  "bv-gauss-elim",
  "bv-gauss-elim",
  "bv-inequality-solver",
  "bv-inequality-solver",
  "bv-intro-pow2",
  "bv-intro-pow2",
  "bv-lazy-reduce-extf",
  "bv-lazy-reduce-extf",
  "bv-lazy-rewrite-extf",
  "bv-lazy-rewrite-extf",
  "bv-num-func=N",
  "bv-num-func=N",
  "bv-optimize-sat-proof=MODE",
  "bv-optimize-sat-proof=MODE",
  "bv-print-consts-as-indexed-symbols",
  "bv-print-consts-as-indexed-symbols",
  "bv-proof-format=MODE",
  "bv-proof-format=MODE",
  "bv-propagate",
  "bv-propagate",
  "bv-quick-xplain",
  "bv-quick-xplain",
  "bv-sat-solver",
  "bv-sat-solver",
  "bv-skolemize",
  "bv-skolemize",
  "bv-to-bool",
  "bv-to-bool",
  "cdt-bisimilar",
  "cdt-bisimilar",
  "dt-binary-split",
  "dt-binary-split",
  "dt-blast-splits",
  "dt-blast-splits",
  "dt-cyclic",
  "dt-cyclic",
  "dt-force-assignment",
  "dt-force-assignment",
  "dt-infer-as-lemmas",
  "dt-infer-as-lemmas",
  "dt-nested-rec",
  "dt-nested-rec",
  "dt-rewrite-error-sel",
  "dt-rewrite-error-sel",
  "dt-share-sel",
  "dt-share-sel",
  "sygus-abort-size=N",
  "sygus-abort-size=N",
  "sygus-fair-max",
  "sygus-fair-max",
  "sygus-fair=MODE",
  "sygus-fair=MODE",
  "sygus-sym-break",
  "sygus-sym-break",
  "sygus-sym-break-agg",
  "sygus-sym-break-agg",
  "sygus-sym-break-dynamic",
  "sygus-sym-break-dynamic",
  "sygus-sym-break-lazy",
  "sygus-sym-break-lazy",
  "sygus-sym-break-pbe",
  "sygus-sym-break-pbe",
  "sygus-sym-break-rlv",
  "sygus-sym-break-rlv",
  "decision-random-weight=N",
  "decision-random-weight=N",
  "decision-threshold=N",
  "decision-threshold=N",
  "decision-use-weight",
  "decision-use-weight",
  "decision-weight-internal=HOW",
  "decision-weight-internal=HOW",
  "decision-mode",
  "decision-mode",
  "dag-thresh",
  "dag-thresh",
  "expr-depth=N",
  "expr-depth=N",
  "print-expr-types",
  "print-expr-types",
  "type-checking",
  "type-checking",
  "fp-exp",
  "fp-exp",
  "copyright",
  "early-exit",
  "early-exit",
  "help",
  "help",
  "interactive",
  "interactive",
  "interactive-prompt",
  "interactive-prompt",
  "seed=N",
  "seed=N",
  "segv-spin",
  "segv-spin",
  "show-config",
  "show-debug-tags",
  "show-trace-tags",
  "tear-down-incremental=N",
  "tear-down-incremental=N",
  "version",
  "version",
  "filesystem-access",
  "filesystem-access",
  "force-logic",
  "force-logic",
  "global-declarations",
  "global-declarations",
  "mmap",
  "mmap",
  "semantic-checks",
  "semantic-checks",
  "strict-parsing",
  "strict-parsing",
  "flatten-ho-chains",
  "flatten-ho-chains",
  "inst-format=MODE",
  "inst-format=MODE",
  "model-format=MODE",
  "model-format=MODE",
  "aggressive-core-min",
  "aggressive-core-min",
  "allow-empty-dependencies",
  "allow-empty-dependencies",
  "fewer-preprocessing-holes",
  "fewer-preprocessing-holes",
  "lfsc-letification",
  "lfsc-letification",
  "minisat-dump-dimacs",
  "minisat-dump-dimacs",
  "minisat-elimination",
  "minisat-elimination",
  "random-frequency",
  "random-frequency",
  "random-seed",
  "random-seed",
  "refine-conflicts",
  "refine-conflicts",
  "restart-int-base=N",
  "restart-int-base=N",
  "restart-int-inc=F",
  "restart-int-inc=F",
  "ag-miniscope-quant",
  "ag-miniscope-quant",
  "cegis-sample=MODE",
  "cegis-sample=MODE",
  "cegqi",
  "cegqi",
  "cegqi-all",
  "cegqi-all",
  "cegqi-bv",
  "cegqi-bv",
  "cegqi-bv-concat-inv",
  "cegqi-bv-concat-inv",
  "cegqi-bv-ineq=MODE",
  "cegqi-bv-ineq=MODE",
  "cegqi-bv-interleave-value",
  "cegqi-bv-interleave-value",
  "cegqi-bv-linear",
  "cegqi-bv-linear",
  "cegqi-bv-rm-extract",
  "cegqi-bv-rm-extract",
  "cegqi-bv-solve-nl",
  "cegqi-bv-solve-nl",
  "cegqi-full",
  "cegqi-full",
  "cegqi-innermost",
  "cegqi-innermost",
  "cegqi-midpoint",
  "cegqi-midpoint",
  "cegqi-min-bounds",
  "cegqi-min-bounds",
  "cegqi-model",
  "cegqi-model",
  "cegqi-multi-inst",
  "cegqi-multi-inst",
  "cegqi-nested-qe",
  "cegqi-nested-qe",
  "cegqi-nopt",
  "cegqi-nopt",
  "cegqi-prereg-inst",
  "cegqi-prereg-inst",
  "cegqi-repeat-lit",
  "cegqi-repeat-lit",
  "cegqi-round-up-lia",
  "cegqi-round-up-lia",
  "cegqi-sat",
  "cegqi-sat",
  "cegqi-use-inf-int",
  "cegqi-use-inf-int",
  "cegqi-use-inf-real",
  "cegqi-use-inf-real",
  "cond-var-split-agg-quant",
  "cond-var-split-agg-quant",
  "cond-var-split-quant",
  "cond-var-split-quant",
  "conjecture-filter-active-terms",
  "conjecture-filter-active-terms",
  "conjecture-filter-canonical",
  "conjecture-filter-canonical",
  "conjecture-filter-model",
  "conjecture-filter-model",
  "conjecture-gen",
  "conjecture-gen",
  "conjecture-gen-gt-enum=N",
  "conjecture-gen-gt-enum=N",
  "conjecture-gen-max-depth=N",
  "conjecture-gen-max-depth=N",
  "conjecture-gen-per-round=N",
  "conjecture-gen-per-round=N",
  "conjecture-gen-uee-intro",
  "conjecture-gen-uee-intro",
  "conjecture-no-filter",
  "conjecture-no-filter",
  "dt-stc-ind",
  "dt-stc-ind",
  "dt-var-exp-quant",
  "dt-var-exp-quant",
  "e-matching",
  "e-matching",
  "elim-ext-arith-quant",
  "elim-ext-arith-quant",
  "elim-taut-quant",
  "elim-taut-quant",
  "ext-rewrite-quant",
  "ext-rewrite-quant",
  "finite-model-find",
  "finite-model-find",
  "fmf-bound",
  "fmf-bound",
  "fmf-bound-int",
  "fmf-bound-int",
  "fmf-bound-lazy",
  "fmf-bound-lazy",
  "fmf-fmc-simple",
  "fmf-fmc-simple",
  "fmf-fresh-dc",
  "fmf-fresh-dc",
  "fmf-fun",
  "fmf-fun",
  "fmf-fun-rlv",
  "fmf-fun-rlv",
  "fmf-inst-engine",
  "fmf-inst-engine",
  "fmf-type-completion-thresh=N",
  "fmf-type-completion-thresh=N",
  "fs-interleave",
  "fs-interleave",
  "fs-stratify",
  "fs-stratify",
  "full-saturate-quant",
  "full-saturate-quant",
  "full-saturate-quant-limit=N",
  "full-saturate-quant-limit=N",
  "full-saturate-quant-rd",
  "full-saturate-quant-rd",
  "global-negate",
  "global-negate",
  "ho-elim",
  "ho-elim",
  "ho-elim-store-ax",
  "ho-elim-store-ax",
  "ho-matching",
  "ho-matching",
  "ho-matching-var-priority",
  "ho-matching-var-priority",
  "ho-merge-term-db",
  "ho-merge-term-db",
  "increment-triggers",
  "increment-triggers",
  "inst-level-input-only",
  "inst-level-input-only",
  "inst-max-level=N",
  "inst-max-level=N",
  "inst-no-entail",
  "inst-no-entail",
  "inst-no-model-true",
  "inst-no-model-true",
  "inst-when-phase=N",
  "inst-when-phase=N",
  "inst-when-strict-interleave",
  "inst-when-strict-interleave",
  "inst-when-tc-first",
  "inst-when-tc-first",
  "inst-when=MODE",
  "inst-when=MODE",
  "int-wf-ind",
  "int-wf-ind",
  "ite-dtt-split-quant",
  "ite-dtt-split-quant",
  "ite-lift-quant=MODE",
  "ite-lift-quant=MODE",
  "literal-matching=MODE",
  "literal-matching=MODE",
  "lte-restrict-inst-closure",
  "lte-restrict-inst-closure",
  "macros-quant",
  "macros-quant",
  "macros-quant-mode=MODE",
  "macros-quant-mode=MODE",
  "mbqi-interleave",
  "mbqi-interleave",
  "mbqi-one-inst-per-round",
  "mbqi-one-inst-per-round",
  "mbqi=MODE",
  "mbqi=MODE",
  "miniscope-quant",
  "miniscope-quant",
  "miniscope-quant-fv",
  "miniscope-quant-fv",
  "multi-trigger-cache",
  "multi-trigger-cache",
  "multi-trigger-linear",
  "multi-trigger-linear",
  "multi-trigger-priority",
  "multi-trigger-priority",
  "multi-trigger-when-single",
  "multi-trigger-when-single",
  "partial-triggers",
  "partial-triggers",
  "pre-skolem-quant",
  "pre-skolem-quant",
  "pre-skolem-quant-agg",
  "pre-skolem-quant-agg",
  "pre-skolem-quant-nested",
  "pre-skolem-quant-nested",
  "prenex-quant-user",
  "prenex-quant-user",
  "prenex-quant=MODE",
  "prenex-quant=MODE",
  "pure-th-triggers",
  "pure-th-triggers",
  "purify-dt-triggers",
  "purify-dt-triggers",
  "purify-triggers",
  "purify-triggers",
  "qcf-all-conflict",
  "qcf-all-conflict",
  "qcf-eager-check-rd",
  "qcf-eager-check-rd",
  "qcf-eager-test",
  "qcf-eager-test",
  "qcf-nested-conflict",
  "qcf-nested-conflict",
  "qcf-skip-rd",
  "qcf-skip-rd",
  "qcf-tconstraint",
  "qcf-tconstraint",
  "qcf-vo-exp",
  "qcf-vo-exp",
  "quant-alpha-equiv",
  "quant-alpha-equiv",
  "quant-anti-skolem",
  "quant-anti-skolem",
  "quant-cf",
  "quant-cf",
  "quant-cf-mode=MODE",
  "quant-cf-mode=MODE",
  "quant-cf-when=MODE",
  "quant-cf-when=MODE",
  "quant-dsplit-mode=MODE",
  "quant-dsplit-mode=MODE",
  "quant-epr",
  "quant-epr",
  "quant-epr-match",
  "quant-epr-match",
  "quant-fun-wd",
  "quant-fun-wd",
  "quant-ind",
  "quant-ind",
  "quant-model-ee",
  "quant-model-ee",
  "quant-rep-mode=MODE",
  "quant-rep-mode=MODE",
  "quant-split",
  "quant-split",
  "register-quant-body-terms",
  "register-quant-body-terms",
  "relational-triggers",
  "relational-triggers",
  "relevant-triggers",
  "relevant-triggers",
  "strict-triggers",
  "strict-triggers",
  "sygus",
  "sygus",
  "sygus-active-gen-cfactor=N",
  "sygus-active-gen-cfactor=N",
  "sygus-active-gen=MODE",
  "sygus-active-gen=MODE",
  "sygus-add-const-grammar",
  "sygus-add-const-grammar",
  "sygus-arg-relevant",
  "sygus-arg-relevant",
  "sygus-auto-unfold",
  "sygus-auto-unfold",
  "sygus-bool-ite-return-const",
  "sygus-bool-ite-return-const",
  "sygus-core-connective",
  "sygus-core-connective",
  "sygus-crepair-abort",
  "sygus-crepair-abort",
  "sygus-eval-opt",
  "sygus-eval-opt",
  "sygus-eval-unfold",
  "sygus-eval-unfold",
  "sygus-eval-unfold-bool",
  "sygus-eval-unfold-bool",
  "sygus-expr-miner-check-timeout=N",
  "sygus-expr-miner-check-timeout=N",
  "sygus-expr-miner-check-use-export",
  "sygus-expr-miner-check-use-export",
  "sygus-ext-rew",
  "sygus-ext-rew",
  "sygus-filter-sol-rev",
  "sygus-filter-sol-rev",
  "sygus-filter-sol=MODE",
  "sygus-filter-sol=MODE",
  "sygus-grammar-cons=MODE",
  "sygus-grammar-cons=MODE",
  "sygus-grammar-norm",
  "sygus-grammar-norm",
  "sygus-inference",
  "sygus-inference",
  "sygus-inst",
  "sygus-inst",
  "sygus-inv-templ-when-sg",
  "sygus-inv-templ-when-sg",
  "sygus-inv-templ=MODE",
  "sygus-inv-templ=MODE",
  "sygus-min-grammar",
  "sygus-min-grammar",
  "sygus-pbe",
  "sygus-pbe",
  "sygus-pbe-multi-fair",
  "sygus-pbe-multi-fair",
  "sygus-pbe-multi-fair-diff=N",
  "sygus-pbe-multi-fair-diff=N",
  "sygus-qe-preproc",
  "sygus-qe-preproc",
  "sygus-query-gen",
  "sygus-query-gen",
  "sygus-query-gen-check",
  "sygus-query-gen-check",
  "sygus-query-gen-dump-files=MODE",
  "sygus-query-gen-dump-files=MODE",
  "sygus-query-gen-thresh=N",
  "sygus-query-gen-thresh=N",
  "sygus-rec-fun",
  "sygus-rec-fun",
  "sygus-rec-fun-eval-limit=N",
  "sygus-rec-fun-eval-limit=N",
  "sygus-repair-const",
  "sygus-repair-const",
  "sygus-repair-const-timeout=N",
  "sygus-repair-const-timeout=N",
  "sygus-rr",
  "sygus-rr",
  "sygus-rr-synth",
  "sygus-rr-synth",
  "sygus-rr-synth-accel",
  "sygus-rr-synth-accel",
  "sygus-rr-synth-check",
  "sygus-rr-synth-check",
  "sygus-rr-synth-filter-cong",
  "sygus-rr-synth-filter-cong",
  "sygus-rr-synth-filter-match",
  "sygus-rr-synth-filter-match",
  "sygus-rr-synth-filter-nl",
  "sygus-rr-synth-filter-nl",
  "sygus-rr-synth-filter-order",
  "sygus-rr-synth-filter-order",
  "sygus-rr-synth-input",
  "sygus-rr-synth-input",
  "sygus-rr-synth-input-nvars=N",
  "sygus-rr-synth-input-nvars=N",
  "sygus-rr-synth-input-use-bool",
  "sygus-rr-synth-input-use-bool",
  "sygus-rr-synth-rec",
  "sygus-rr-synth-rec",
  "sygus-rr-verify",
  "sygus-rr-verify",
  "sygus-rr-verify-abort",
  "sygus-rr-verify-abort",
  "sygus-sample-fp-uniform",
  "sygus-sample-fp-uniform",
  "sygus-sample-grammar",
  "sygus-sample-grammar",
  "sygus-samples=N",
  "sygus-samples=N",
  "sygus-si-abort",
  "sygus-si-abort",
  "sygus-si-partial",
  "sygus-si-partial",
  "sygus-si-rcons-limit=N",
  "sygus-si-rcons-limit=N",
  "sygus-si-rcons=MODE",
  "sygus-si-rcons=MODE",
  "sygus-si-reconstruct-const",
  "sygus-si-reconstruct-const",
  "sygus-si=MODE",
  "sygus-si=MODE",
  "sygus-stream",
  "sygus-stream",
  "sygus-templ-embed-grammar",
  "sygus-templ-embed-grammar",
  "sygus-unif-cond-independent-no-repeat-sol",
  "sygus-unif-cond-independent-no-repeat-sol",
  "sygus-unif-pi=MODE",
  "sygus-unif-pi=MODE",
  "sygus-unif-shuffle-cond",
  "sygus-unif-shuffle-cond",
  "sygus-verify-subcall",
  "sygus-verify-subcall",
  "term-db-mode=MODE",
  "term-db-mode=MODE",
  "track-inst-lemmas",
  "track-inst-lemmas",
  "trigger-active-sel=MODE",
  "trigger-active-sel=MODE",
  "trigger-sel=MODE",
  "trigger-sel=MODE",
  "user-pat=MODE",
  "user-pat=MODE",
  "var-elim-quant",
  "var-elim-quant",
  "var-ineq-elim-quant",
  "var-ineq-elim-quant",
  "sep-check-neg",
  "sep-check-neg",
  "sep-child-refine",
  "sep-child-refine",
  "sep-deq-c",
  "sep-deq-c",
  "sep-exp",
  "sep-exp",
  "sep-min-refine",
  "sep-min-refine",
  "sep-pre-skolem-emp",
  "sep-pre-skolem-emp",
  "sets-ext",
  "sets-ext",
  "sets-infer-as-lemmas",
  "sets-infer-as-lemmas",
  "sets-proxy-lemmas",
  "sets-proxy-lemmas",
  "abstract-values",
  "abstract-values",
  "ackermann",
  "ackermann",
  "bitblast-step=N",
  "bitblast-step=N",
  "block-models=MODE",
  "block-models=MODE",
  "bv-eager-assert-step=N",
  "bv-eager-assert-step=N",
  "bv-propagation-step=N",
  "bv-propagation-step=N",
  "bv-sat-conflict-step=N",
  "bv-sat-conflict-step=N",
  "check-abducts",
  "check-abducts",
  "check-models",
  "check-models",
  "check-proofs",
  "check-proofs",
  "check-synth-sol",
  "check-synth-sol",
  "check-unsat-cores",
  "check-unsat-cores",
  "cnf-step=N",
  "cnf-step=N",
  "cpu-time",
  "cpu-time",
  "debug-check-models",
  "debug-check-models",
  "decision-step=N",
  "decision-step=N",
  "diagnostic-output-channel",
  "diagnostic-output-channel",
  "dump-instantiations",
  "dump-instantiations",
  "dump-models",
  "dump-models",
  "dump-proofs",
  "dump-proofs",
  "dump-synth",
  "dump-synth",
  "dump-to",
  "dump-to",
  "dump-unsat-cores",
  "dump-unsat-cores",
  "dump-unsat-cores-full",
  "dump-unsat-cores-full",
  "dump",
  "dump",
  "expand-definitions",
  "expand-definitions",
  "ext-rew-prep",
  "ext-rew-prep",
  "ext-rew-prep-agg",
  "ext-rew-prep-agg",
  "force-no-limit-cpu-while-dump",
  "force-no-limit-cpu-while-dump",
  "hard-limit",
  "hard-limit",
  "incremental",
  "incremental",
  "interactive-mode",
  "interactive-mode",
  "ite-simp",
  "ite-simp",
  "lemma-step=N",
  "lemma-step=N",
  "model-cores=MODE",
  "model-cores=MODE",
  "model-u-dt-enum",
  "model-u-dt-enum",
  "model-witness-value",
  "model-witness-value",
  "on-repeat-ite-simp",
  "on-repeat-ite-simp",
  "parse-step=N",
  "parse-step=N",
  "preprocess-step=N",
  "preprocess-step=N",
  "produce-abducts",
  "produce-abducts",
  "produce-assertions",
  "produce-assertions",
  "produce-assignments",
  "produce-assignments",
  "produce-models",
  "produce-models",
  "produce-unsat-assumptions",
  "produce-unsat-assumptions",
  "produce-unsat-cores",
  "produce-unsat-cores",
  "produce-proofs",
  "produce-proofs",
  "quantifier-step=N",
  "quantifier-step=N",
  "regular-output-channel",
  "regular-output-channel",
  "repeat-simp",
  "repeat-simp",
  "restart-step=N",
  "restart-step=N",
  "rewrite-step=N",
  "rewrite-step=N",
  "reproducible-resource-limit",
  "reproducible-resource-limit",
  "rlimit",
  "rlimit",
  "sat-conflict-step=N",
  "sat-conflict-step=N",
  "simp-ite-compress",
  "simp-ite-compress",
  "simp-ite-hunt-zombies=N",
  "simp-ite-hunt-zombies=N",
  "simp-with-care",
  "simp-with-care",
  "simplification-mode",
  "simplification-mode",
  "solve-bv-as-int=N",
  "solve-bv-as-int=N",
  "solve-int-as-bv=N",
  "solve-int-as-bv=N",
  "solve-real-as-int",
  "solve-real-as-int",
  "sort-inference",
  "sort-inference",
  "static-learning",
  "static-learning",
  "sygus-out=MODE",
  "sygus-out=MODE",
  "sygus-print-callbacks",
  "sygus-print-callbacks",
  "theory-check-step=N",
  "theory-check-step=N",
  "tlimit-per",
  "tlimit-per",
  "tlimit",
  "tlimit",
  "unconstrained-simp",
  "unconstrained-simp",
  "re-elim",
  "re-elim",
  "re-elim-agg",
  "re-elim-agg",
  "re-inter-mode=MODE",
  "re-inter-mode=MODE",
  "strings-check-entail-len",
  "strings-check-entail-len",
  "strings-eager",
  "strings-eager",
  "strings-eager-len",
  "strings-eager-len",
  "strings-exp",
  "strings-exp",
  "strings-ff",
  "strings-ff",
  "strings-fmf",
  "strings-fmf",
  "strings-guess-model",
  "strings-guess-model",
  "strings-infer-as-lemmas",
  "strings-infer-as-lemmas",
  "strings-infer-sym",
  "strings-infer-sym",
  "strings-inm",
  "strings-inm",
  "strings-lazy-pp",
  "strings-lazy-pp",
  "strings-len-norm",
  "strings-len-norm",
  "strings-lprop-csp",
  "strings-lprop-csp",
  "strings-min-prefix-explain",
  "strings-min-prefix-explain",
  "strings-print-ascii",
  "strings-print-ascii",
  "strings-process-loop-mode=MODE",
  "strings-process-loop-mode=MODE",
  "strings-rexplain-lemmas",
  "strings-rexplain-lemmas",
  "assign-function-values",
  "assign-function-values",
  "condense-function-values",
  "condense-function-values",
  "theoryof-mode",
  "theoryof-mode",
  "uf-symmetry-breaker",
  "uf-symmetry-breaker",
  "uf-ho",
  "uf-ho",
  "uf-ho-ext",
  "uf-ho-ext",
  "uf-ss-abort-card=N",
  "uf-ss-abort-card=N",
  "uf-ss-fair",
  "uf-ss-fair",
  "uf-ss-fair-monotone",
  "uf-ss-fair-monotone",
  "uf-ss-totality",
  "uf-ss-totality",
  "uf-ss-totality-limited=N",
  "uf-ss-totality-limited=N",
  "uf-ss-totality-sym-break",
  "uf-ss-totality-sym-break",
  "uf-ss=MODE",
  "uf-ss=MODE",
  NULL
};/* smtOptions[] */

std::vector<std::string> Options::suggestSmtOptions(
    const std::string& optionName)
{
  std::vector<std::string> suggestions;

  const char* opt;
  for(size_t i = 0; (opt = smtOptions[i]) != NULL; ++i) {
    if(std::strstr(opt, optionName.c_str()) != NULL) {
      suggestions.push_back(opt);
    }
  }

  return suggestions;
}

std::vector<std::vector<std::string> > Options::getOptions() const
{
  std::vector< std::vector<std::string> > opts;

  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->maxApproxDepth; std::vector<std::string> v; v.push_back("approx-branch-depth=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("arith-brab"); v.push_back(std::string(d_holder->brabTest ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("arith-no-partial-fun"); v.push_back(std::string(d_holder->arithNoPartialFun ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->arithPropAsLemmaLength; std::vector<std::string> v; v.push_back("arith-prop-clauses=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->arithPropagationMode; std::vector<std::string> v; v.push_back("arith-prop=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("arith-rewrite-equalities"); v.push_back(std::string(d_holder->arithRewriteEq ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("collect-pivot-stats"); v.push_back(std::string(d_holder->collectPivots ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cut-all-bounded"); v.push_back(std::string(d_holder->doCutAllBounded ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dio-decomps"); v.push_back(std::string(d_holder->exportDioDecompositions ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dio-repeat"); v.push_back(std::string(d_holder->dioRepeat ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dio-solver"); v.push_back(std::string(d_holder->arithDioSolver ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->dioSolverTurns; std::vector<std::string> v; v.push_back("dio-turns=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->arithErrorSelectionRule; std::vector<std::string> v; v.push_back("error-selection-rule=RULE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fc-penalties"); v.push_back(std::string(d_holder->havePenalties ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->arithHeuristicPivots; std::vector<std::string> v; v.push_back("heuristic-pivots=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("lemmas-on-replay-failure"); v.push_back(std::string(d_holder->replayFailureLemma ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->maxCutsInContext; std::vector<std::string> v; v.push_back("maxCutsInContext=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("miplib-trick"); v.push_back(std::string(d_holder->arithMLTrick ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->arithMLTrickSubstitutions; std::vector<std::string> v; v.push_back("miplib-trick-subs"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("new-prop"); v.push_back(std::string(d_holder->newProp ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("nl-ext"); v.push_back(std::string(d_holder->nlExt ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("nl-ext-ent-conf"); v.push_back(std::string(d_holder->nlExtEntailConflicts ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("nl-ext-factor"); v.push_back(std::string(d_holder->nlExtFactor ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("nl-ext-inc-prec"); v.push_back(std::string(d_holder->nlExtIncPrecision ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("nl-ext-purify"); v.push_back(std::string(d_holder->nlExtPurify ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("nl-ext-rbound"); v.push_back(std::string(d_holder->nlExtResBound ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("nl-ext-rewrite"); v.push_back(std::string(d_holder->nlExtRewrites ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("nl-ext-split-zero"); v.push_back(std::string(d_holder->nlExtSplitZero ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->nlExtTfTaylorDegree; std::vector<std::string> v; v.push_back("nl-ext-tf-taylor-deg=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("nl-ext-tf-tplanes"); v.push_back(std::string(d_holder->nlExtTfTangentPlanes ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("nl-ext-tplanes"); v.push_back(std::string(d_holder->nlExtTangentPlanes ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("nl-ext-tplanes-interleave"); v.push_back(std::string(d_holder->nlExtTangentPlanesInterleave ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("pb-rewrites"); v.push_back(std::string(d_holder->pbRewrites ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->arithPivotThreshold; std::vector<std::string> v; v.push_back("pivot-threshold=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->ppAssertMaxSubSize; std::vector<std::string> v; v.push_back("pp-assert-max-sub-size=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->arithPropagateMaxLength; std::vector<std::string> v; v.push_back("prop-row-length=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->replayEarlyCloseDepths; std::vector<std::string> v; v.push_back("replay-early-close-depth=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->replayFailurePenalty; std::vector<std::string> v; v.push_back("replay-failure-penalty=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->lemmaRejectCutSize; std::vector<std::string> v; v.push_back("replay-lemma-reject-cut=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->replayNumericFailurePenalty; std::vector<std::string> v; v.push_back("replay-num-err-penalty=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->replayRejectCutSize; std::vector<std::string> v; v.push_back("replay-reject-cut=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->soiApproxMajorFailurePen; std::vector<std::string> v; v.push_back("replay-soi-major-threshold-pen=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->soiApproxMajorFailure; std::vector<std::string> v; v.push_back("replay-soi-major-threshold=T"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->soiApproxMinorFailurePen; std::vector<std::string> v; v.push_back("replay-soi-minor-threshold-pen=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->soiApproxMinorFailure; std::vector<std::string> v; v.push_back("replay-soi-minor-threshold=T"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("restrict-pivots"); v.push_back(std::string(d_holder->restrictedPivots ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("revert-arith-models-on-unsat"); v.push_back(std::string(d_holder->revertArithModels ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->rrTurns; std::vector<std::string> v; v.push_back("rr-turns=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("se-solve-int"); v.push_back(std::string(d_holder->trySolveIntStandardEffort ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->arithSimplexCheckPeriod; std::vector<std::string> v; v.push_back("simplex-check-period=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("soi-qe"); v.push_back(std::string(d_holder->soiQuickExplain ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->arithStandardCheckVarOrderPivots; std::vector<std::string> v; v.push_back("standard-effort-variable-order-pivots=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->arithUnateLemmaMode; std::vector<std::string> v; v.push_back("unate-lemmas=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("use-approx"); v.push_back(std::string(d_holder->useApprox ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("use-fcsimplex"); v.push_back(std::string(d_holder->useFC ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("use-soi"); v.push_back(std::string(d_holder->useSOI ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->arraysConfig; std::vector<std::string> v; v.push_back("arrays-config=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("arrays-eager-index"); v.push_back(std::string(d_holder->arraysEagerIndexSplitting ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("arrays-eager-lemmas"); v.push_back(std::string(d_holder->arraysEagerLemmas ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("arrays-lazy-rintro1"); v.push_back(std::string(d_holder->arraysLazyRIntro1 ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("arrays-model-based"); v.push_back(std::string(d_holder->arraysModelBased ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("arrays-optimize-linear"); v.push_back(std::string(d_holder->arraysOptimizeLinear ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->arraysPropagate; std::vector<std::string> v; v.push_back("arrays-prop=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("arrays-reduce-sharing"); v.push_back(std::string(d_holder->arraysReduceSharing ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("arrays-weak-equiv"); v.push_back(std::string(d_holder->arraysWeakEquivalence ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->inputLanguage; std::vector<std::string> v; v.push_back("input-language"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->outputLanguage; std::vector<std::string> v; v.push_back("output-language"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("parse-only"); v.push_back(std::string(d_holder->parseOnly ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("preprocess-only"); v.push_back(std::string(d_holder->preprocessOnly ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("print-success"); v.push_back(std::string(d_holder->printSuccess ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("statistics"); v.push_back(std::string(d_holder->statistics ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("stats-every-query"); v.push_back(std::string(d_holder->statsEveryQuery ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("stats-hide-zeros"); v.push_back(std::string(d_holder->statsHideZeros ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->verbosity; std::vector<std::string> v; v.push_back("verbosity"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bitblast-aig"); v.push_back(std::string(d_holder->bitvectorAig ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->bitblastMode; std::vector<std::string> v; v.push_back("bitblast"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bitwise-eq"); v.push_back(std::string(d_holder->bitwiseEq ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->boolToBitvector; std::vector<std::string> v; v.push_back("bool-to-bv"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-abstraction"); v.push_back(std::string(d_holder->bvAbstraction ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->bitvectorAigSimplifications; std::vector<std::string> v; v.push_back("bv-aig-simp=COMMAND"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-alg-extf"); v.push_back(std::string(d_holder->bvAlgExtf ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->bitvectorAlgebraicBudget; std::vector<std::string> v; v.push_back("bv-algebraic-budget=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-algebraic-solver"); v.push_back(std::string(d_holder->bitvectorAlgebraicSolver ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-div-zero-const"); v.push_back(std::string(d_holder->bitvectorDivByZeroConst ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-eager-explanations"); v.push_back(std::string(d_holder->bvEagerExplanations ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->bitvectorEqualitySlicer; std::vector<std::string> v; v.push_back("bv-eq-slicer=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-eq-solver"); v.push_back(std::string(d_holder->bitvectorEqualitySolver ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-extract-arith"); v.push_back(std::string(d_holder->bvExtractArithRewrite ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-gauss-elim"); v.push_back(std::string(d_holder->bvGaussElim ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-inequality-solver"); v.push_back(std::string(d_holder->bitvectorInequalitySolver ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-intro-pow2"); v.push_back(std::string(d_holder->bvIntroducePow2 ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-lazy-reduce-extf"); v.push_back(std::string(d_holder->bvLazyReduceExtf ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-lazy-rewrite-extf"); v.push_back(std::string(d_holder->bvLazyRewriteExtf ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->bvNumFunc; std::vector<std::string> v; v.push_back("bv-num-func=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->bvOptimizeSatProof; std::vector<std::string> v; v.push_back("bv-optimize-sat-proof=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-print-consts-as-indexed-symbols"); v.push_back(std::string(d_holder->bvPrintConstsAsIndexedSymbols ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->bvProofFormat; std::vector<std::string> v; v.push_back("bv-proof-format=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-propagate"); v.push_back(std::string(d_holder->bitvectorPropagate ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-quick-xplain"); v.push_back(std::string(d_holder->bitvectorQuickXplain ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->bvSatSolver; std::vector<std::string> v; v.push_back("bv-sat-solver"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-skolemize"); v.push_back(std::string(d_holder->skolemizeArguments ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("bv-to-bool"); v.push_back(std::string(d_holder->bitvectorToBool ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cdt-bisimilar"); v.push_back(std::string(d_holder->cdtBisimilar ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dt-binary-split"); v.push_back(std::string(d_holder->dtBinarySplit ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dt-blast-splits"); v.push_back(std::string(d_holder->dtBlastSplits ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dt-cyclic"); v.push_back(std::string(d_holder->dtCyclic ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dt-force-assignment"); v.push_back(std::string(d_holder->dtForceAssignment ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dt-infer-as-lemmas"); v.push_back(std::string(d_holder->dtInferAsLemmas ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dt-nested-rec"); v.push_back(std::string(d_holder->dtNestedRec ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dt-rewrite-error-sel"); v.push_back(std::string(d_holder->dtRewriteErrorSel ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dt-share-sel"); v.push_back(std::string(d_holder->dtSharedSelectors ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->sygusAbortSize; std::vector<std::string> v; v.push_back("sygus-abort-size=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-fair-max"); v.push_back(std::string(d_holder->sygusFairMax ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->sygusFair; std::vector<std::string> v; v.push_back("sygus-fair=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-sym-break"); v.push_back(std::string(d_holder->sygusSymBreak ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-sym-break-agg"); v.push_back(std::string(d_holder->sygusSymBreakAgg ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-sym-break-dynamic"); v.push_back(std::string(d_holder->sygusSymBreakDynamic ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-sym-break-lazy"); v.push_back(std::string(d_holder->sygusSymBreakLazy ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-sym-break-pbe"); v.push_back(std::string(d_holder->sygusSymBreakPbe ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-sym-break-rlv"); v.push_back(std::string(d_holder->sygusSymBreakRlv ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->decisionRandomWeight; std::vector<std::string> v; v.push_back("decision-random-weight=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->decisionThreshold; std::vector<std::string> v; v.push_back("decision-threshold=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("decision-use-weight"); v.push_back(std::string(d_holder->decisionUseWeight ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->decisionWeightInternal; std::vector<std::string> v; v.push_back("decision-weight-internal=HOW"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->decisionMode; std::vector<std::string> v; v.push_back("decision-mode"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->defaultDagThresh; std::vector<std::string> v; v.push_back("dag-thresh"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->defaultExprDepth; std::vector<std::string> v; v.push_back("expr-depth=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("print-expr-types"); v.push_back(std::string(d_holder->printExprTypes ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("type-checking"); v.push_back(std::string(d_holder->typeChecking ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fp-exp"); v.push_back(std::string(d_holder->fpExp ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("early-exit"); v.push_back(std::string(d_holder->earlyExit ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("help"); v.push_back(std::string(d_holder->help ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("interactive"); v.push_back(std::string(d_holder->interactive ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("interactive-prompt"); v.push_back(std::string(d_holder->interactivePrompt ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->seed; std::vector<std::string> v; v.push_back("seed=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("segv-spin"); v.push_back(std::string(d_holder->segvSpin ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->tearDownIncremental; std::vector<std::string> v; v.push_back("tear-down-incremental=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("version"); v.push_back(std::string(d_holder->version ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("filesystem-access"); v.push_back(std::string(d_holder->filesystemAccess ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->forceLogicString; std::vector<std::string> v; v.push_back("force-logic"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("global-declarations"); v.push_back(std::string(d_holder->globalDeclarations ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("mmap"); v.push_back(std::string(d_holder->memoryMap ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("semantic-checks"); v.push_back(std::string(d_holder->semanticChecks ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strict-parsing"); v.push_back(std::string(d_holder->strictParsing ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("flatten-ho-chains"); v.push_back(std::string(d_holder->flattenHOChains ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->instFormatMode; std::vector<std::string> v; v.push_back("inst-format=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->modelFormatMode; std::vector<std::string> v; v.push_back("model-format=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("aggressive-core-min"); v.push_back(std::string(d_holder->aggressiveCoreMin ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("allow-empty-dependencies"); v.push_back(std::string(d_holder->allowEmptyDependencies ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fewer-preprocessing-holes"); v.push_back(std::string(d_holder->fewerPreprocessingHoles ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("lfsc-letification"); v.push_back(std::string(d_holder->lfscLetification ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("minisat-dump-dimacs"); v.push_back(std::string(d_holder->minisatDumpDimacs ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("minisat-elimination"); v.push_back(std::string(d_holder->minisatUseElim ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->satRandomFreq; std::vector<std::string> v; v.push_back("random-frequency"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->satRandomSeed; std::vector<std::string> v; v.push_back("random-seed"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("refine-conflicts"); v.push_back(std::string(d_holder->sat_refine_conflicts ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->satRestartFirst; std::vector<std::string> v; v.push_back("restart-int-base=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->satRestartInc; std::vector<std::string> v; v.push_back("restart-int-inc=F"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("ag-miniscope-quant"); v.push_back(std::string(d_holder->aggressiveMiniscopeQuant ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->cegisSample; std::vector<std::string> v; v.push_back("cegis-sample=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi"); v.push_back(std::string(d_holder->cegqi ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-all"); v.push_back(std::string(d_holder->cegqiAll ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-bv"); v.push_back(std::string(d_holder->cegqiBv ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-bv-concat-inv"); v.push_back(std::string(d_holder->cegqiBvConcInv ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->cegqiBvIneqMode; std::vector<std::string> v; v.push_back("cegqi-bv-ineq=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-bv-interleave-value"); v.push_back(std::string(d_holder->cegqiBvInterleaveValue ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-bv-linear"); v.push_back(std::string(d_holder->cegqiBvLinearize ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-bv-rm-extract"); v.push_back(std::string(d_holder->cegqiBvRmExtract ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-bv-solve-nl"); v.push_back(std::string(d_holder->cegqiBvSolveNl ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-full"); v.push_back(std::string(d_holder->cegqiFullEffort ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-innermost"); v.push_back(std::string(d_holder->cegqiInnermost ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-midpoint"); v.push_back(std::string(d_holder->cegqiMidpoint ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-min-bounds"); v.push_back(std::string(d_holder->cegqiMinBounds ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-model"); v.push_back(std::string(d_holder->cegqiModel ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-multi-inst"); v.push_back(std::string(d_holder->cegqiMultiInst ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-nested-qe"); v.push_back(std::string(d_holder->cegqiNestedQE ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-nopt"); v.push_back(std::string(d_holder->cegqiNopt ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-prereg-inst"); v.push_back(std::string(d_holder->cegqiPreRegInst ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-repeat-lit"); v.push_back(std::string(d_holder->cegqiRepeatLit ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-round-up-lia"); v.push_back(std::string(d_holder->cegqiRoundUpLowerLia ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-sat"); v.push_back(std::string(d_holder->cegqiSat ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-use-inf-int"); v.push_back(std::string(d_holder->cegqiUseInfInt ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cegqi-use-inf-real"); v.push_back(std::string(d_holder->cegqiUseInfReal ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cond-var-split-agg-quant"); v.push_back(std::string(d_holder->condVarSplitQuantAgg ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cond-var-split-quant"); v.push_back(std::string(d_holder->condVarSplitQuant ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("conjecture-filter-active-terms"); v.push_back(std::string(d_holder->conjectureFilterActiveTerms ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("conjecture-filter-canonical"); v.push_back(std::string(d_holder->conjectureFilterCanonical ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("conjecture-filter-model"); v.push_back(std::string(d_holder->conjectureFilterModel ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("conjecture-gen"); v.push_back(std::string(d_holder->conjectureGen ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->conjectureGenGtEnum; std::vector<std::string> v; v.push_back("conjecture-gen-gt-enum=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->conjectureGenMaxDepth; std::vector<std::string> v; v.push_back("conjecture-gen-max-depth=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->conjectureGenPerRound; std::vector<std::string> v; v.push_back("conjecture-gen-per-round=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("conjecture-gen-uee-intro"); v.push_back(std::string(d_holder->conjectureUeeIntro ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("conjecture-no-filter"); v.push_back(std::string(d_holder->conjectureNoFilter ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dt-stc-ind"); v.push_back(std::string(d_holder->dtStcInduction ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dt-var-exp-quant"); v.push_back(std::string(d_holder->dtVarExpandQuant ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("e-matching"); v.push_back(std::string(d_holder->eMatching ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("elim-ext-arith-quant"); v.push_back(std::string(d_holder->elimExtArithQuant ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("elim-taut-quant"); v.push_back(std::string(d_holder->elimTautQuant ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("ext-rewrite-quant"); v.push_back(std::string(d_holder->extRewriteQuant ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("finite-model-find"); v.push_back(std::string(d_holder->finiteModelFind ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fmf-bound"); v.push_back(std::string(d_holder->fmfBound ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fmf-bound-int"); v.push_back(std::string(d_holder->fmfBoundInt ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fmf-bound-lazy"); v.push_back(std::string(d_holder->fmfBoundLazy ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fmf-fmc-simple"); v.push_back(std::string(d_holder->fmfFmcSimple ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fmf-fresh-dc"); v.push_back(std::string(d_holder->fmfFreshDistConst ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fmf-fun"); v.push_back(std::string(d_holder->fmfFunWellDefined ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fmf-fun-rlv"); v.push_back(std::string(d_holder->fmfFunWellDefinedRelevant ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fmf-inst-engine"); v.push_back(std::string(d_holder->fmfInstEngine ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->fmfTypeCompletionThresh; std::vector<std::string> v; v.push_back("fmf-type-completion-thresh=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fs-interleave"); v.push_back(std::string(d_holder->fullSaturateInterleave ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("fs-stratify"); v.push_back(std::string(d_holder->fullSaturateStratify ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("full-saturate-quant"); v.push_back(std::string(d_holder->fullSaturateQuant ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->fullSaturateLimit; std::vector<std::string> v; v.push_back("full-saturate-quant-limit=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("full-saturate-quant-rd"); v.push_back(std::string(d_holder->fullSaturateQuantRd ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("global-negate"); v.push_back(std::string(d_holder->globalNegate ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("ho-elim"); v.push_back(std::string(d_holder->hoElim ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("ho-elim-store-ax"); v.push_back(std::string(d_holder->hoElimStoreAx ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("ho-matching"); v.push_back(std::string(d_holder->hoMatching ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("ho-matching-var-priority"); v.push_back(std::string(d_holder->hoMatchingVarArgPriority ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("ho-merge-term-db"); v.push_back(std::string(d_holder->hoMergeTermDb ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("increment-triggers"); v.push_back(std::string(d_holder->incrementTriggers ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("inst-level-input-only"); v.push_back(std::string(d_holder->instLevelInputOnly ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->instMaxLevel; std::vector<std::string> v; v.push_back("inst-max-level=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("inst-no-entail"); v.push_back(std::string(d_holder->instNoEntail ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("inst-no-model-true"); v.push_back(std::string(d_holder->instNoModelTrue ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->instWhenPhase; std::vector<std::string> v; v.push_back("inst-when-phase=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("inst-when-strict-interleave"); v.push_back(std::string(d_holder->instWhenStrictInterleave ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("inst-when-tc-first"); v.push_back(std::string(d_holder->instWhenTcFirst ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->instWhenMode; std::vector<std::string> v; v.push_back("inst-when=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("int-wf-ind"); v.push_back(std::string(d_holder->intWfInduction ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("ite-dtt-split-quant"); v.push_back(std::string(d_holder->iteDtTesterSplitQuant ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->iteLiftQuant; std::vector<std::string> v; v.push_back("ite-lift-quant=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->literalMatchMode; std::vector<std::string> v; v.push_back("literal-matching=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("lte-restrict-inst-closure"); v.push_back(std::string(d_holder->lteRestrictInstClosure ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("macros-quant"); v.push_back(std::string(d_holder->macrosQuant ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->macrosQuantMode; std::vector<std::string> v; v.push_back("macros-quant-mode=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("mbqi-interleave"); v.push_back(std::string(d_holder->mbqiInterleave ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("mbqi-one-inst-per-round"); v.push_back(std::string(d_holder->fmfOneInstPerRound ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->mbqiMode; std::vector<std::string> v; v.push_back("mbqi=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("miniscope-quant"); v.push_back(std::string(d_holder->miniscopeQuant ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("miniscope-quant-fv"); v.push_back(std::string(d_holder->miniscopeQuantFreeVar ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("multi-trigger-cache"); v.push_back(std::string(d_holder->multiTriggerCache ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("multi-trigger-linear"); v.push_back(std::string(d_holder->multiTriggerLinear ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("multi-trigger-priority"); v.push_back(std::string(d_holder->multiTriggerPriority ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("multi-trigger-when-single"); v.push_back(std::string(d_holder->multiTriggerWhenSingle ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("partial-triggers"); v.push_back(std::string(d_holder->partialTriggers ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("pre-skolem-quant"); v.push_back(std::string(d_holder->preSkolemQuant ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("pre-skolem-quant-agg"); v.push_back(std::string(d_holder->preSkolemQuantAgg ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("pre-skolem-quant-nested"); v.push_back(std::string(d_holder->preSkolemQuantNested ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("prenex-quant-user"); v.push_back(std::string(d_holder->prenexQuantUser ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->prenexQuant; std::vector<std::string> v; v.push_back("prenex-quant=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("pure-th-triggers"); v.push_back(std::string(d_holder->pureThTriggers ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("purify-dt-triggers"); v.push_back(std::string(d_holder->purifyDtTriggers ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("purify-triggers"); v.push_back(std::string(d_holder->purifyTriggers ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("qcf-all-conflict"); v.push_back(std::string(d_holder->qcfAllConflict ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("qcf-eager-check-rd"); v.push_back(std::string(d_holder->qcfEagerCheckRd ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("qcf-eager-test"); v.push_back(std::string(d_holder->qcfEagerTest ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("qcf-nested-conflict"); v.push_back(std::string(d_holder->qcfNestedConflict ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("qcf-skip-rd"); v.push_back(std::string(d_holder->qcfSkipRd ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("qcf-tconstraint"); v.push_back(std::string(d_holder->qcfTConstraint ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("qcf-vo-exp"); v.push_back(std::string(d_holder->qcfVoExp ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("quant-alpha-equiv"); v.push_back(std::string(d_holder->quantAlphaEquiv ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("quant-anti-skolem"); v.push_back(std::string(d_holder->quantAntiSkolem ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("quant-cf"); v.push_back(std::string(d_holder->quantConflictFind ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->qcfMode; std::vector<std::string> v; v.push_back("quant-cf-mode=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->qcfWhenMode; std::vector<std::string> v; v.push_back("quant-cf-when=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->quantDynamicSplit; std::vector<std::string> v; v.push_back("quant-dsplit-mode=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("quant-epr"); v.push_back(std::string(d_holder->quantEpr ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("quant-epr-match"); v.push_back(std::string(d_holder->quantEprMatching ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("quant-fun-wd"); v.push_back(std::string(d_holder->quantFunWellDefined ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("quant-ind"); v.push_back(std::string(d_holder->quantInduction ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("quant-model-ee"); v.push_back(std::string(d_holder->quantModelEe ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->quantRepMode; std::vector<std::string> v; v.push_back("quant-rep-mode=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("quant-split"); v.push_back(std::string(d_holder->quantSplit ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("register-quant-body-terms"); v.push_back(std::string(d_holder->registerQuantBodyTerms ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("relational-triggers"); v.push_back(std::string(d_holder->relationalTriggers ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("relevant-triggers"); v.push_back(std::string(d_holder->relevantTriggers ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strict-triggers"); v.push_back(std::string(d_holder->strictTriggers ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus"); v.push_back(std::string(d_holder->sygus ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->sygusActiveGenEnumConsts; std::vector<std::string> v; v.push_back("sygus-active-gen-cfactor=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->sygusActiveGenMode; std::vector<std::string> v; v.push_back("sygus-active-gen=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-add-const-grammar"); v.push_back(std::string(d_holder->sygusAddConstGrammar ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-arg-relevant"); v.push_back(std::string(d_holder->sygusArgRelevant ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-auto-unfold"); v.push_back(std::string(d_holder->sygusInvAutoUnfold ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-bool-ite-return-const"); v.push_back(std::string(d_holder->sygusBoolIteReturnConst ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-core-connective"); v.push_back(std::string(d_holder->sygusCoreConnective ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-crepair-abort"); v.push_back(std::string(d_holder->sygusConstRepairAbort ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-eval-opt"); v.push_back(std::string(d_holder->sygusEvalOpt ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-eval-unfold"); v.push_back(std::string(d_holder->sygusEvalUnfold ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-eval-unfold-bool"); v.push_back(std::string(d_holder->sygusEvalUnfoldBool ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->sygusExprMinerCheckTimeout; std::vector<std::string> v; v.push_back("sygus-expr-miner-check-timeout=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-expr-miner-check-use-export"); v.push_back(std::string(d_holder->sygusExprMinerCheckUseExport ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-ext-rew"); v.push_back(std::string(d_holder->sygusExtRew ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-filter-sol-rev"); v.push_back(std::string(d_holder->sygusFilterSolRevSubsume ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->sygusFilterSolMode; std::vector<std::string> v; v.push_back("sygus-filter-sol=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->sygusGrammarConsMode; std::vector<std::string> v; v.push_back("sygus-grammar-cons=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-grammar-norm"); v.push_back(std::string(d_holder->sygusGrammarNorm ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-inference"); v.push_back(std::string(d_holder->sygusInference ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-inst"); v.push_back(std::string(d_holder->sygusInst ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-inv-templ-when-sg"); v.push_back(std::string(d_holder->sygusInvTemplWhenSyntax ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->sygusInvTemplMode; std::vector<std::string> v; v.push_back("sygus-inv-templ=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-min-grammar"); v.push_back(std::string(d_holder->sygusMinGrammar ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-pbe"); v.push_back(std::string(d_holder->sygusUnifPbe ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-pbe-multi-fair"); v.push_back(std::string(d_holder->sygusPbeMultiFair ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->sygusPbeMultiFairDiff; std::vector<std::string> v; v.push_back("sygus-pbe-multi-fair-diff=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-qe-preproc"); v.push_back(std::string(d_holder->sygusQePreproc ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-query-gen"); v.push_back(std::string(d_holder->sygusQueryGen ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-query-gen-check"); v.push_back(std::string(d_holder->sygusQueryGenCheck ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->sygusQueryGenDumpFiles; std::vector<std::string> v; v.push_back("sygus-query-gen-dump-files=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->sygusQueryGenThresh; std::vector<std::string> v; v.push_back("sygus-query-gen-thresh=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rec-fun"); v.push_back(std::string(d_holder->sygusRecFun ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->sygusRecFunEvalLimit; std::vector<std::string> v; v.push_back("sygus-rec-fun-eval-limit=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-repair-const"); v.push_back(std::string(d_holder->sygusRepairConst ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->sygusRepairConstTimeout; std::vector<std::string> v; v.push_back("sygus-repair-const-timeout=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr"); v.push_back(std::string(d_holder->sygusRew ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr-synth"); v.push_back(std::string(d_holder->sygusRewSynth ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr-synth-accel"); v.push_back(std::string(d_holder->sygusRewSynthAccel ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr-synth-check"); v.push_back(std::string(d_holder->sygusRewSynthCheck ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr-synth-filter-cong"); v.push_back(std::string(d_holder->sygusRewSynthFilterCong ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr-synth-filter-match"); v.push_back(std::string(d_holder->sygusRewSynthFilterMatch ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr-synth-filter-nl"); v.push_back(std::string(d_holder->sygusRewSynthFilterNonLinear ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr-synth-filter-order"); v.push_back(std::string(d_holder->sygusRewSynthFilterOrder ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr-synth-input"); v.push_back(std::string(d_holder->sygusRewSynthInput ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->sygusRewSynthInputNVars; std::vector<std::string> v; v.push_back("sygus-rr-synth-input-nvars=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr-synth-input-use-bool"); v.push_back(std::string(d_holder->sygusRewSynthInputUseBool ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr-synth-rec"); v.push_back(std::string(d_holder->sygusRewSynthRec ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr-verify"); v.push_back(std::string(d_holder->sygusRewVerify ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-rr-verify-abort"); v.push_back(std::string(d_holder->sygusRewVerifyAbort ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-sample-fp-uniform"); v.push_back(std::string(d_holder->sygusSampleFpUniform ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-sample-grammar"); v.push_back(std::string(d_holder->sygusSampleGrammar ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->sygusSamples; std::vector<std::string> v; v.push_back("sygus-samples=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-si-abort"); v.push_back(std::string(d_holder->cegqiSingleInvAbort ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-si-partial"); v.push_back(std::string(d_holder->cegqiSingleInvPartial ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->cegqiSingleInvReconstructLimit; std::vector<std::string> v; v.push_back("sygus-si-rcons-limit=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->cegqiSingleInvReconstruct; std::vector<std::string> v; v.push_back("sygus-si-rcons=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-si-reconstruct-const"); v.push_back(std::string(d_holder->cegqiSingleInvReconstructConst ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->cegqiSingleInvMode; std::vector<std::string> v; v.push_back("sygus-si=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-stream"); v.push_back(std::string(d_holder->sygusStream ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-templ-embed-grammar"); v.push_back(std::string(d_holder->sygusTemplEmbedGrammar ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-unif-cond-independent-no-repeat-sol"); v.push_back(std::string(d_holder->sygusUnifCondIndNoRepeatSol ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->sygusUnifPi; std::vector<std::string> v; v.push_back("sygus-unif-pi=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-unif-shuffle-cond"); v.push_back(std::string(d_holder->sygusUnifShuffleCond ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-verify-subcall"); v.push_back(std::string(d_holder->sygusVerifySubcall ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->termDbMode; std::vector<std::string> v; v.push_back("term-db-mode=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("track-inst-lemmas"); v.push_back(std::string(d_holder->trackInstLemmas ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->triggerActiveSelMode; std::vector<std::string> v; v.push_back("trigger-active-sel=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->triggerSelMode; std::vector<std::string> v; v.push_back("trigger-sel=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->userPatternsQuant; std::vector<std::string> v; v.push_back("user-pat=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("var-elim-quant"); v.push_back(std::string(d_holder->varElimQuant ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("var-ineq-elim-quant"); v.push_back(std::string(d_holder->varIneqElimQuant ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sep-check-neg"); v.push_back(std::string(d_holder->sepCheckNeg ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sep-child-refine"); v.push_back(std::string(d_holder->sepChildRefine ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sep-deq-c"); v.push_back(std::string(d_holder->sepDisequalC ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sep-exp"); v.push_back(std::string(d_holder->sepExp ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sep-min-refine"); v.push_back(std::string(d_holder->sepMinimalRefine ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sep-pre-skolem-emp"); v.push_back(std::string(d_holder->sepPreSkolemEmp ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sets-ext"); v.push_back(std::string(d_holder->setsExt ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sets-infer-as-lemmas"); v.push_back(std::string(d_holder->setsInferAsLemmas ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sets-proxy-lemmas"); v.push_back(std::string(d_holder->setsProxyLemmas ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("abstract-values"); v.push_back(std::string(d_holder->abstractValues ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("ackermann"); v.push_back(std::string(d_holder->ackermann ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->bitblastStep; std::vector<std::string> v; v.push_back("bitblast-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->blockModelsMode; std::vector<std::string> v; v.push_back("block-models=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->bvEagerAssertStep; std::vector<std::string> v; v.push_back("bv-eager-assert-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->bvPropagationStep; std::vector<std::string> v; v.push_back("bv-propagation-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->bvSatConflictStep; std::vector<std::string> v; v.push_back("bv-sat-conflict-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("check-abducts"); v.push_back(std::string(d_holder->checkAbducts ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("check-models"); v.push_back(std::string(d_holder->checkModels ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("check-proofs"); v.push_back(std::string(d_holder->checkProofs ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("check-synth-sol"); v.push_back(std::string(d_holder->checkSynthSol ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("check-unsat-cores"); v.push_back(std::string(d_holder->checkUnsatCores ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->cnfStep; std::vector<std::string> v; v.push_back("cnf-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("cpu-time"); v.push_back(std::string(d_holder->cpuTime ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("debug-check-models"); v.push_back(std::string(d_holder->debugCheckModels ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->decisionStep; std::vector<std::string> v; v.push_back("decision-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->diagnosticChannelName; std::vector<std::string> v; v.push_back("diagnostic-output-channel"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dump-instantiations"); v.push_back(std::string(d_holder->dumpInstantiations ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dump-models"); v.push_back(std::string(d_holder->dumpModels ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dump-proofs"); v.push_back(std::string(d_holder->dumpProofs ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dump-synth"); v.push_back(std::string(d_holder->dumpSynth ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->dumpToFileName; std::vector<std::string> v; v.push_back("dump-to"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dump-unsat-cores"); v.push_back(std::string(d_holder->dumpUnsatCores ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("dump-unsat-cores-full"); v.push_back(std::string(d_holder->dumpUnsatCoresFull ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->dumpModeString; std::vector<std::string> v; v.push_back("dump"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("expand-definitions"); v.push_back(std::string(d_holder->expandDefinitions ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("ext-rew-prep"); v.push_back(std::string(d_holder->extRewPrep ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("ext-rew-prep-agg"); v.push_back(std::string(d_holder->extRewPrepAgg ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("force-no-limit-cpu-while-dump"); v.push_back(std::string(d_holder->forceNoLimitCpuWhileDump ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("hard-limit"); v.push_back(std::string(d_holder->hardLimit ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("incremental"); v.push_back(std::string(d_holder->incrementalSolving ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("interactive-mode"); v.push_back(std::string(d_holder->interactiveMode ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("ite-simp"); v.push_back(std::string(d_holder->doITESimp ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->lemmaStep; std::vector<std::string> v; v.push_back("lemma-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->modelCoresMode; std::vector<std::string> v; v.push_back("model-cores=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("model-u-dt-enum"); v.push_back(std::string(d_holder->modelUninterpDtEnum ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("model-witness-value"); v.push_back(std::string(d_holder->modelWitnessValue ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("on-repeat-ite-simp"); v.push_back(std::string(d_holder->doITESimpOnRepeat ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->parseStep; std::vector<std::string> v; v.push_back("parse-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->preprocessStep; std::vector<std::string> v; v.push_back("preprocess-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("produce-abducts"); v.push_back(std::string(d_holder->produceAbducts ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("produce-assertions"); v.push_back(std::string(d_holder->produceAssertions ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("produce-assignments"); v.push_back(std::string(d_holder->produceAssignments ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("produce-models"); v.push_back(std::string(d_holder->produceModels ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("produce-unsat-assumptions"); v.push_back(std::string(d_holder->unsatAssumptions ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("produce-unsat-cores"); v.push_back(std::string(d_holder->unsatCores ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("produce-proofs"); v.push_back(std::string(d_holder->proof ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->quantifierStep; std::vector<std::string> v; v.push_back("quantifier-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->regularChannelName; std::vector<std::string> v; v.push_back("regular-output-channel"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("repeat-simp"); v.push_back(std::string(d_holder->repeatSimp ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->restartStep; std::vector<std::string> v; v.push_back("restart-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->rewriteStep; std::vector<std::string> v; v.push_back("rewrite-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->perCallResourceLimit; std::vector<std::string> v; v.push_back("reproducible-resource-limit"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->cumulativeResourceLimit; std::vector<std::string> v; v.push_back("rlimit"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->satConflictStep; std::vector<std::string> v; v.push_back("sat-conflict-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("simp-ite-compress"); v.push_back(std::string(d_holder->compressItes ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->zombieHuntThreshold; std::vector<std::string> v; v.push_back("simp-ite-hunt-zombies=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("simp-with-care"); v.push_back(std::string(d_holder->simplifyWithCareEnabled ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->simplificationMode; std::vector<std::string> v; v.push_back("simplification-mode"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->solveBVAsInt; std::vector<std::string> v; v.push_back("solve-bv-as-int=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->solveIntAsBV; std::vector<std::string> v; v.push_back("solve-int-as-bv=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("solve-real-as-int"); v.push_back(std::string(d_holder->solveRealAsInt ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sort-inference"); v.push_back(std::string(d_holder->sortInference ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("static-learning"); v.push_back(std::string(d_holder->doStaticLearning ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->sygusOut; std::vector<std::string> v; v.push_back("sygus-out=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("sygus-print-callbacks"); v.push_back(std::string(d_holder->sygusPrintCallbacks ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->theoryCheckStep; std::vector<std::string> v; v.push_back("theory-check-step=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->perCallMillisecondLimit; std::vector<std::string> v; v.push_back("tlimit-per"); v.push_back(ss.str()); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->cumulativeMillisecondLimit; std::vector<std::string> v; v.push_back("tlimit"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("unconstrained-simp"); v.push_back(std::string(d_holder->unconstrainedSimp ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("re-elim"); v.push_back(std::string(d_holder->regExpElim ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("re-elim-agg"); v.push_back(std::string(d_holder->regExpElimAgg ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->stringRegExpInterMode; std::vector<std::string> v; v.push_back("re-inter-mode=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-check-entail-len"); v.push_back(std::string(d_holder->stringCheckEntailLen ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-eager"); v.push_back(std::string(d_holder->stringEager ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-eager-len"); v.push_back(std::string(d_holder->stringEagerLen ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-exp"); v.push_back(std::string(d_holder->stringExp ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-ff"); v.push_back(std::string(d_holder->stringFlatForms ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-fmf"); v.push_back(std::string(d_holder->stringFMF ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-guess-model"); v.push_back(std::string(d_holder->stringGuessModel ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-infer-as-lemmas"); v.push_back(std::string(d_holder->stringInferAsLemmas ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-infer-sym"); v.push_back(std::string(d_holder->stringInferSym ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-inm"); v.push_back(std::string(d_holder->stringIgnNegMembership ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-lazy-pp"); v.push_back(std::string(d_holder->stringLazyPreproc ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-len-norm"); v.push_back(std::string(d_holder->stringLenNorm ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-lprop-csp"); v.push_back(std::string(d_holder->stringLenPropCsp ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-min-prefix-explain"); v.push_back(std::string(d_holder->stringMinPrefixExplain ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-print-ascii"); v.push_back(std::string(d_holder->stdPrintASCII ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->stringProcessLoopMode; std::vector<std::string> v; v.push_back("strings-process-loop-mode=MODE"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("strings-rexplain-lemmas"); v.push_back(std::string(d_holder->stringRExplainLemmas ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("assign-function-values"); v.push_back(std::string(d_holder->assignFunctionValues ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("condense-function-values"); v.push_back(std::string(d_holder->condenseFunctionValues ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->theoryOfMode; std::vector<std::string> v; v.push_back("theoryof-mode"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("uf-symmetry-breaker"); v.push_back(std::string(d_holder->ufSymmetryBreaker ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("uf-ho"); v.push_back(std::string(d_holder->ufHo ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("uf-ho-ext"); v.push_back(std::string(d_holder->ufHoExt ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->ufssAbortCardinality; std::vector<std::string> v; v.push_back("uf-ss-abort-card=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("uf-ss-fair"); v.push_back(std::string(d_holder->ufssFairness ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("uf-ss-fair-monotone"); v.push_back(std::string(d_holder->ufssFairnessMonotone ? "true" : "false")); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("uf-ss-totality"); v.push_back(std::string(d_holder->ufssTotality ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << std::fixed << std::setprecision(8); ss << d_holder->ufssTotalityLimited; std::vector<std::string> v; v.push_back("uf-ss-totality-limited=N"); v.push_back(ss.str()); opts.push_back(v); }
  { std::vector<std::string> v; v.push_back("uf-ss-totality-sym-break"); v.push_back(std::string(d_holder->ufssTotalitySymBreak ? "true" : "false")); opts.push_back(v); }
  { std::stringstream ss; ss << d_holder->ufssMode; std::vector<std::string> v; v.push_back("uf-ss=MODE"); v.push_back(ss.str()); opts.push_back(v); }


  return opts;
}

void Options::setOption(const std::string& key, const std::string& optionarg)
{
  options::OptionsHandler* handler = d_handler;
  Options *options = Options::current();
  Trace("options") << "SMT setOption(" << key << ", " << optionarg << ")"
                   << std::endl;

  if(key == "approx-branch-depth") {
  options->assign(options::maxApproxDepth, "approx-branch-depth", optionarg);
return;
}
if(key == "arith-brab") {
  options->assignBool(options::brabTest, "arith-brab", optionarg == "true");
return;
}
if(key == "arith-no-partial-fun") {
  options->assignBool(options::arithNoPartialFun, "arith-no-partial-fun", optionarg == "true");
return;
}
if(key == "arith-prop-clauses") {
  options->assign(options::arithPropAsLemmaLength, "arith-prop-clauses", optionarg);
return;
}
if(key == "arith-prop") {
  options->assign(options::arithPropagationMode, "arith-prop", optionarg);
return;
}
if(key == "arith-rewrite-equalities") {
  options->assignBool(options::arithRewriteEq, "arith-rewrite-equalities", optionarg == "true");
return;
}
if(key == "collect-pivot-stats") {
  options->assignBool(options::collectPivots, "collect-pivot-stats", optionarg == "true");
return;
}
if(key == "cut-all-bounded") {
  options->assignBool(options::doCutAllBounded, "cut-all-bounded", optionarg == "true");
return;
}
if(key == "dio-decomps") {
  options->assignBool(options::exportDioDecompositions, "dio-decomps", optionarg == "true");
return;
}
if(key == "dio-repeat") {
  options->assignBool(options::dioRepeat, "dio-repeat", optionarg == "true");
return;
}
if(key == "dio-solver") {
  options->assignBool(options::arithDioSolver, "dio-solver", optionarg == "true");
return;
}
if(key == "dio-turns") {
  options->assign(options::dioSolverTurns, "dio-turns", optionarg);
return;
}
if(key == "error-selection-rule") {
  options->assign(options::arithErrorSelectionRule, "error-selection-rule", optionarg);
return;
}
if(key == "fc-penalties") {
  options->assignBool(options::havePenalties, "fc-penalties", optionarg == "true");
return;
}
if(key == "heuristic-pivots") {
  options->assign(options::arithHeuristicPivots, "heuristic-pivots", optionarg);
return;
}
if(key == "lemmas-on-replay-failure") {
  options->assignBool(options::replayFailureLemma, "lemmas-on-replay-failure", optionarg == "true");
return;
}
if(key == "maxCutsInContext") {
  options->assign(options::maxCutsInContext, "maxCutsInContext", optionarg);
return;
}
if(key == "miplib-trick") {
  options->assignBool(options::arithMLTrick, "miplib-trick", optionarg == "true");
return;
}
if(key == "miplib-trick-subs") {
  options->assign(options::arithMLTrickSubstitutions, "miplib-trick-subs", optionarg);
return;
}
if(key == "new-prop") {
  options->assignBool(options::newProp, "new-prop", optionarg == "true");
return;
}
if(key == "nl-ext") {
  options->assignBool(options::nlExt, "nl-ext", optionarg == "true");
return;
}
if(key == "nl-ext-ent-conf") {
  options->assignBool(options::nlExtEntailConflicts, "nl-ext-ent-conf", optionarg == "true");
return;
}
if(key == "nl-ext-factor") {
  options->assignBool(options::nlExtFactor, "nl-ext-factor", optionarg == "true");
return;
}
if(key == "nl-ext-inc-prec") {
  options->assignBool(options::nlExtIncPrecision, "nl-ext-inc-prec", optionarg == "true");
return;
}
if(key == "nl-ext-purify") {
  options->assignBool(options::nlExtPurify, "nl-ext-purify", optionarg == "true");
return;
}
if(key == "nl-ext-rbound") {
  options->assignBool(options::nlExtResBound, "nl-ext-rbound", optionarg == "true");
return;
}
if(key == "nl-ext-rewrite") {
  options->assignBool(options::nlExtRewrites, "nl-ext-rewrite", optionarg == "true");
return;
}
if(key == "nl-ext-split-zero") {
  options->assignBool(options::nlExtSplitZero, "nl-ext-split-zero", optionarg == "true");
return;
}
if(key == "nl-ext-tf-taylor-deg") {
  options->assign(options::nlExtTfTaylorDegree, "nl-ext-tf-taylor-deg", optionarg);
return;
}
if(key == "nl-ext-tf-tplanes") {
  options->assignBool(options::nlExtTfTangentPlanes, "nl-ext-tf-tplanes", optionarg == "true");
return;
}
if(key == "nl-ext-tplanes") {
  options->assignBool(options::nlExtTangentPlanes, "nl-ext-tplanes", optionarg == "true");
return;
}
if(key == "nl-ext-tplanes-interleave") {
  options->assignBool(options::nlExtTangentPlanesInterleave, "nl-ext-tplanes-interleave", optionarg == "true");
return;
}
if(key == "pb-rewrites") {
  options->assignBool(options::pbRewrites, "pb-rewrites", optionarg == "true");
return;
}
if(key == "pivot-threshold") {
  options->assign(options::arithPivotThreshold, "pivot-threshold", optionarg);
return;
}
if(key == "pp-assert-max-sub-size") {
  options->assign(options::ppAssertMaxSubSize, "pp-assert-max-sub-size", optionarg);
return;
}
if(key == "prop-row-length") {
  options->assign(options::arithPropagateMaxLength, "prop-row-length", optionarg);
return;
}
if(key == "replay-early-close-depth") {
  options->assign(options::replayEarlyCloseDepths, "replay-early-close-depth", optionarg);
return;
}
if(key == "replay-failure-penalty") {
  options->assign(options::replayFailurePenalty, "replay-failure-penalty", optionarg);
return;
}
if(key == "replay-lemma-reject-cut") {
  options->assign(options::lemmaRejectCutSize, "replay-lemma-reject-cut", optionarg);
return;
}
if(key == "replay-num-err-penalty") {
  options->assign(options::replayNumericFailurePenalty, "replay-num-err-penalty", optionarg);
return;
}
if(key == "replay-reject-cut") {
  options->assign(options::replayRejectCutSize, "replay-reject-cut", optionarg);
return;
}
if(key == "replay-soi-major-threshold-pen") {
  options->assign(options::soiApproxMajorFailurePen, "replay-soi-major-threshold-pen", optionarg);
return;
}
if(key == "replay-soi-major-threshold") {
  options->assign(options::soiApproxMajorFailure, "replay-soi-major-threshold", optionarg);
return;
}
if(key == "replay-soi-minor-threshold-pen") {
  options->assign(options::soiApproxMinorFailurePen, "replay-soi-minor-threshold-pen", optionarg);
return;
}
if(key == "replay-soi-minor-threshold") {
  options->assign(options::soiApproxMinorFailure, "replay-soi-minor-threshold", optionarg);
return;
}
if(key == "restrict-pivots") {
  options->assignBool(options::restrictedPivots, "restrict-pivots", optionarg == "true");
return;
}
if(key == "revert-arith-models-on-unsat") {
  options->assignBool(options::revertArithModels, "revert-arith-models-on-unsat", optionarg == "true");
return;
}
if(key == "rr-turns") {
  options->assign(options::rrTurns, "rr-turns", optionarg);
return;
}
if(key == "se-solve-int") {
  options->assignBool(options::trySolveIntStandardEffort, "se-solve-int", optionarg == "true");
return;
}
if(key == "simplex-check-period") {
  options->assign(options::arithSimplexCheckPeriod, "simplex-check-period", optionarg);
return;
}
if(key == "soi-qe") {
  options->assignBool(options::soiQuickExplain, "soi-qe", optionarg == "true");
return;
}
if(key == "standard-effort-variable-order-pivots") {
  options->assign(options::arithStandardCheckVarOrderPivots, "standard-effort-variable-order-pivots", optionarg);
return;
}
if(key == "unate-lemmas") {
  options->assign(options::arithUnateLemmaMode, "unate-lemmas", optionarg);
return;
}
if(key == "use-approx") {
  options->assignBool(options::useApprox, "use-approx", optionarg == "true");
return;
}
if(key == "use-fcsimplex") {
  options->assignBool(options::useFC, "use-fcsimplex", optionarg == "true");
return;
}
if(key == "use-soi") {
  options->assignBool(options::useSOI, "use-soi", optionarg == "true");
return;
}
if(key == "arrays-config") {
  options->assign(options::arraysConfig, "arrays-config", optionarg);
return;
}
if(key == "arrays-eager-index") {
  options->assignBool(options::arraysEagerIndexSplitting, "arrays-eager-index", optionarg == "true");
return;
}
if(key == "arrays-eager-lemmas") {
  options->assignBool(options::arraysEagerLemmas, "arrays-eager-lemmas", optionarg == "true");
return;
}
if(key == "arrays-lazy-rintro1") {
  options->assignBool(options::arraysLazyRIntro1, "arrays-lazy-rintro1", optionarg == "true");
return;
}
if(key == "arrays-model-based") {
  options->assignBool(options::arraysModelBased, "arrays-model-based", optionarg == "true");
return;
}
if(key == "arrays-optimize-linear") {
  options->assignBool(options::arraysOptimizeLinear, "arrays-optimize-linear", optionarg == "true");
return;
}
if(key == "arrays-prop") {
  options->assign(options::arraysPropagate, "arrays-prop", optionarg);
return;
}
if(key == "arrays-reduce-sharing") {
  options->assignBool(options::arraysReduceSharing, "arrays-reduce-sharing", optionarg == "true");
return;
}
if(key == "arrays-weak-equiv") {
  options->assignBool(options::arraysWeakEquivalence, "arrays-weak-equiv", optionarg == "true");
return;
}
if(key == "debug") {
handler->enableDebugTag("debug", optionarg);
return;
}
if(key == "input-language" || key == "lang") {
  options->assign(options::inputLanguage, "input-language", optionarg);
return;
}
if(key == "output-lang" || key == "output-language") {
  options->assign(options::outputLanguage, "output-language", optionarg);
return;
}
if(key == "parse-only") {
  options->assignBool(options::parseOnly, "parse-only", optionarg == "true");
return;
}
if(key == "preprocess-only") {
  options->assignBool(options::preprocessOnly, "preprocess-only", optionarg == "true");
return;
}
if(key == "print-success") {
  options->assignBool(options::printSuccess, "print-success", optionarg == "true");
return;
}
if(key == "quiet") {
handler->decreaseVerbosity("quiet");
return;
}
if(key == "statistics" || key == "stats") {
  options->assignBool(options::statistics, "statistics", optionarg == "true");
return;
}
if(key == "stats-every-query") {
  options->assignBool(options::statsEveryQuery, "stats-every-query", optionarg == "true");
return;
}
if(key == "stats-hide-zeros") {
  options->assignBool(options::statsHideZeros, "stats-hide-zeros", optionarg == "true");
return;
}
if(key == "trace") {
handler->enableTraceTag("trace", optionarg);
return;
}
if(key == "verbose") {
handler->increaseVerbosity("verbose");
return;
}
if(key == "verbosity") {
  options->assign(options::verbosity, "verbosity", optionarg);
return;
}
if(key == "bitblast-aig") {
  options->assignBool(options::bitvectorAig, "bitblast-aig", optionarg == "true");
return;
}
if(key == "bitblast") {
  options->assign(options::bitblastMode, "bitblast", optionarg);
return;
}
if(key == "bitwise-eq") {
  options->assignBool(options::bitwiseEq, "bitwise-eq", optionarg == "true");
return;
}
if(key == "bool-to-bv") {
  options->assign(options::boolToBitvector, "bool-to-bv", optionarg);
return;
}
if(key == "bv-abstraction") {
  options->assignBool(options::bvAbstraction, "bv-abstraction", optionarg == "true");
return;
}
if(key == "bv-aig-simp") {
  options->assign(options::bitvectorAigSimplifications, "bv-aig-simp", optionarg);
return;
}
if(key == "bv-alg-extf") {
  options->assignBool(options::bvAlgExtf, "bv-alg-extf", optionarg == "true");
return;
}
if(key == "bv-algebraic-budget") {
  options->assign(options::bitvectorAlgebraicBudget, "bv-algebraic-budget", optionarg);
return;
}
if(key == "bv-algebraic-solver") {
  options->assignBool(options::bitvectorAlgebraicSolver, "bv-algebraic-solver", optionarg == "true");
return;
}
if(key == "bv-div-zero-const") {
  options->assignBool(options::bitvectorDivByZeroConst, "bv-div-zero-const", optionarg == "true");
return;
}
if(key == "bv-eager-explanations") {
  options->assignBool(options::bvEagerExplanations, "bv-eager-explanations", optionarg == "true");
return;
}
if(key == "bv-eq-slicer") {
  options->assign(options::bitvectorEqualitySlicer, "bv-eq-slicer", optionarg);
return;
}
if(key == "bv-eq-solver") {
  options->assignBool(options::bitvectorEqualitySolver, "bv-eq-solver", optionarg == "true");
return;
}
if(key == "bv-extract-arith") {
  options->assignBool(options::bvExtractArithRewrite, "bv-extract-arith", optionarg == "true");
return;
}
if(key == "bv-gauss-elim") {
  options->assignBool(options::bvGaussElim, "bv-gauss-elim", optionarg == "true");
return;
}
if(key == "bv-inequality-solver") {
  options->assignBool(options::bitvectorInequalitySolver, "bv-inequality-solver", optionarg == "true");
return;
}
if(key == "bv-intro-pow2") {
  options->assignBool(options::bvIntroducePow2, "bv-intro-pow2", optionarg == "true");
return;
}
if(key == "bv-lazy-reduce-extf") {
  options->assignBool(options::bvLazyReduceExtf, "bv-lazy-reduce-extf", optionarg == "true");
return;
}
if(key == "bv-lazy-rewrite-extf") {
  options->assignBool(options::bvLazyRewriteExtf, "bv-lazy-rewrite-extf", optionarg == "true");
return;
}
if(key == "bv-num-func") {
  options->assign(options::bvNumFunc, "bv-num-func", optionarg);
return;
}
if(key == "bv-optimize-sat-proof") {
  options->assign(options::bvOptimizeSatProof, "bv-optimize-sat-proof", optionarg);
return;
}
if(key == "bv-print-consts-as-indexed-symbols") {
  options->assignBool(options::bvPrintConstsAsIndexedSymbols, "bv-print-consts-as-indexed-symbols", optionarg == "true");
return;
}
if(key == "bv-proof-format") {
  options->assign(options::bvProofFormat, "bv-proof-format", optionarg);
return;
}
if(key == "bv-propagate") {
  options->assignBool(options::bitvectorPropagate, "bv-propagate", optionarg == "true");
return;
}
if(key == "bv-quick-xplain") {
  options->assignBool(options::bitvectorQuickXplain, "bv-quick-xplain", optionarg == "true");
return;
}
if(key == "bv-sat-solver") {
  options->assign(options::bvSatSolver, "bv-sat-solver", optionarg);
return;
}
if(key == "bv-skolemize") {
  options->assignBool(options::skolemizeArguments, "bv-skolemize", optionarg == "true");
return;
}
if(key == "bv-to-bool") {
  options->assignBool(options::bitvectorToBool, "bv-to-bool", optionarg == "true");
return;
}
if(key == "cdt-bisimilar") {
  options->assignBool(options::cdtBisimilar, "cdt-bisimilar", optionarg == "true");
return;
}
if(key == "dt-binary-split") {
  options->assignBool(options::dtBinarySplit, "dt-binary-split", optionarg == "true");
return;
}
if(key == "dt-blast-splits") {
  options->assignBool(options::dtBlastSplits, "dt-blast-splits", optionarg == "true");
return;
}
if(key == "dt-cyclic") {
  options->assignBool(options::dtCyclic, "dt-cyclic", optionarg == "true");
return;
}
if(key == "dt-force-assignment") {
  options->assignBool(options::dtForceAssignment, "dt-force-assignment", optionarg == "true");
return;
}
if(key == "dt-infer-as-lemmas") {
  options->assignBool(options::dtInferAsLemmas, "dt-infer-as-lemmas", optionarg == "true");
return;
}
if(key == "dt-nested-rec") {
  options->assignBool(options::dtNestedRec, "dt-nested-rec", optionarg == "true");
return;
}
if(key == "dt-rewrite-error-sel") {
  options->assignBool(options::dtRewriteErrorSel, "dt-rewrite-error-sel", optionarg == "true");
return;
}
if(key == "dt-share-sel") {
  options->assignBool(options::dtSharedSelectors, "dt-share-sel", optionarg == "true");
return;
}
if(key == "sygus-abort-size") {
  options->assign(options::sygusAbortSize, "sygus-abort-size", optionarg);
return;
}
if(key == "sygus-fair-max") {
  options->assignBool(options::sygusFairMax, "sygus-fair-max", optionarg == "true");
return;
}
if(key == "sygus-fair") {
  options->assign(options::sygusFair, "sygus-fair", optionarg);
return;
}
if(key == "sygus-sym-break") {
  options->assignBool(options::sygusSymBreak, "sygus-sym-break", optionarg == "true");
return;
}
if(key == "sygus-sym-break-agg") {
  options->assignBool(options::sygusSymBreakAgg, "sygus-sym-break-agg", optionarg == "true");
return;
}
if(key == "sygus-sym-break-dynamic") {
  options->assignBool(options::sygusSymBreakDynamic, "sygus-sym-break-dynamic", optionarg == "true");
return;
}
if(key == "sygus-sym-break-lazy") {
  options->assignBool(options::sygusSymBreakLazy, "sygus-sym-break-lazy", optionarg == "true");
return;
}
if(key == "sygus-sym-break-pbe") {
  options->assignBool(options::sygusSymBreakPbe, "sygus-sym-break-pbe", optionarg == "true");
return;
}
if(key == "sygus-sym-break-rlv") {
  options->assignBool(options::sygusSymBreakRlv, "sygus-sym-break-rlv", optionarg == "true");
return;
}
if(key == "decision-random-weight") {
  options->assign(options::decisionRandomWeight, "decision-random-weight", optionarg);
return;
}
if(key == "decision-threshold") {
  options->assign(options::decisionThreshold, "decision-threshold", optionarg);
return;
}
if(key == "decision-use-weight") {
  options->assignBool(options::decisionUseWeight, "decision-use-weight", optionarg == "true");
return;
}
if(key == "decision-weight-internal") {
  options->assign(options::decisionWeightInternal, "decision-weight-internal", optionarg);
return;
}
if(key == "decision" || key == "decision-mode") {
  options->assign(options::decisionMode, "decision-mode", optionarg);
return;
}
if(key == "dag-thresh") {
  options->assign(options::defaultDagThresh, "dag-thresh", optionarg);
return;
}
if(key == "expr-depth") {
  options->assign(options::defaultExprDepth, "expr-depth", optionarg);
return;
}
if(key == "print-expr-types") {
  options->assignBool(options::printExprTypes, "print-expr-types", optionarg == "true");
return;
}
if(key == "type-checking") {
  options->assignBool(options::typeChecking, "type-checking", optionarg == "true");
return;
}
if(key == "fp-exp") {
  options->assignBool(options::fpExp, "fp-exp", optionarg == "true");
return;
}
if(key == "copyright") {
handler->copyright("copyright");
return;
}
if(key == "early-exit") {
  options->assignBool(options::earlyExit, "early-exit", optionarg == "true");
return;
}
if(key == "help") {
  options->assignBool(options::help, "help", optionarg == "true");
return;
}
if(key == "interactive") {
  options->assignBool(options::interactive, "interactive", optionarg == "true");
return;
}
if(key == "interactive-prompt") {
  options->assignBool(options::interactivePrompt, "interactive-prompt", optionarg == "true");
return;
}
if(key == "seed") {
  options->assign(options::seed, "seed", optionarg);
return;
}
if(key == "segv-spin") {
  options->assignBool(options::segvSpin, "segv-spin", optionarg == "true");
return;
}
if(key == "show-config") {
handler->showConfiguration("show-config");
return;
}
if(key == "show-debug-tags") {
handler->showDebugTags("show-debug-tags");
return;
}
if(key == "show-trace-tags") {
handler->showTraceTags("show-trace-tags");
return;
}
if(key == "tear-down-incremental") {
  options->assign(options::tearDownIncremental, "tear-down-incremental", optionarg);
return;
}
if(key == "version") {
  options->assignBool(options::version, "version", optionarg == "true");
return;
}
if(key == "filesystem-access") {
  options->assignBool(options::filesystemAccess, "filesystem-access", optionarg == "true");
return;
}
if(key == "force-logic") {
  options->assign(options::forceLogicString, "force-logic", optionarg);
return;
}
if(key == "global-declarations") {
  options->assignBool(options::globalDeclarations, "global-declarations", optionarg == "true");
return;
}
if(key == "mmap") {
  options->assignBool(options::memoryMap, "mmap", optionarg == "true");
return;
}
if(key == "semantic-checks") {
  options->assignBool(options::semanticChecks, "semantic-checks", optionarg == "true");
return;
}
if(key == "strict-parsing") {
  options->assignBool(options::strictParsing, "strict-parsing", optionarg == "true");
return;
}
if(key == "flatten-ho-chains") {
  options->assignBool(options::flattenHOChains, "flatten-ho-chains", optionarg == "true");
return;
}
if(key == "inst-format") {
  options->assign(options::instFormatMode, "inst-format", optionarg);
return;
}
if(key == "model-format") {
  options->assign(options::modelFormatMode, "model-format", optionarg);
return;
}
if(key == "aggressive-core-min") {
  options->assignBool(options::aggressiveCoreMin, "aggressive-core-min", optionarg == "true");
return;
}
if(key == "allow-empty-dependencies") {
  options->assignBool(options::allowEmptyDependencies, "allow-empty-dependencies", optionarg == "true");
return;
}
if(key == "fewer-preprocessing-holes") {
  options->assignBool(options::fewerPreprocessingHoles, "fewer-preprocessing-holes", optionarg == "true");
return;
}
if(key == "lfsc-letification") {
  options->assignBool(options::lfscLetification, "lfsc-letification", optionarg == "true");
return;
}
if(key == "minisat-dump-dimacs") {
  options->assignBool(options::minisatDumpDimacs, "minisat-dump-dimacs", optionarg == "true");
return;
}
if(key == "minisat-elimination") {
  options->assignBool(options::minisatUseElim, "minisat-elimination", optionarg == "true");
return;
}
if(key == "random-freq" || key == "random-frequency") {
  options->assign(options::satRandomFreq, "random-frequency", optionarg);
return;
}
if(key == "random-seed") {
  options->assign(options::satRandomSeed, "random-seed", optionarg);
return;
}
if(key == "refine-conflicts") {
  options->assignBool(options::sat_refine_conflicts, "refine-conflicts", optionarg == "true");
return;
}
if(key == "restart-int-base") {
  options->assign(options::satRestartFirst, "restart-int-base", optionarg);
return;
}
if(key == "restart-int-inc") {
  options->assign(options::satRestartInc, "restart-int-inc", optionarg);
return;
}
if(key == "ag-miniscope-quant") {
  options->assignBool(options::aggressiveMiniscopeQuant, "ag-miniscope-quant", optionarg == "true");
return;
}
if(key == "cegis-sample") {
  options->assign(options::cegisSample, "cegis-sample", optionarg);
return;
}
if(key == "cegqi") {
  options->assignBool(options::cegqi, "cegqi", optionarg == "true");
return;
}
if(key == "cegqi-all") {
  options->assignBool(options::cegqiAll, "cegqi-all", optionarg == "true");
return;
}
if(key == "cegqi-bv") {
  options->assignBool(options::cegqiBv, "cegqi-bv", optionarg == "true");
return;
}
if(key == "cegqi-bv-concat-inv") {
  options->assignBool(options::cegqiBvConcInv, "cegqi-bv-concat-inv", optionarg == "true");
return;
}
if(key == "cegqi-bv-ineq") {
  options->assign(options::cegqiBvIneqMode, "cegqi-bv-ineq", optionarg);
return;
}
if(key == "cegqi-bv-interleave-value") {
  options->assignBool(options::cegqiBvInterleaveValue, "cegqi-bv-interleave-value", optionarg == "true");
return;
}
if(key == "cegqi-bv-linear") {
  options->assignBool(options::cegqiBvLinearize, "cegqi-bv-linear", optionarg == "true");
return;
}
if(key == "cegqi-bv-rm-extract") {
  options->assignBool(options::cegqiBvRmExtract, "cegqi-bv-rm-extract", optionarg == "true");
return;
}
if(key == "cegqi-bv-solve-nl") {
  options->assignBool(options::cegqiBvSolveNl, "cegqi-bv-solve-nl", optionarg == "true");
return;
}
if(key == "cegqi-full") {
  options->assignBool(options::cegqiFullEffort, "cegqi-full", optionarg == "true");
return;
}
if(key == "cegqi-innermost") {
  options->assignBool(options::cegqiInnermost, "cegqi-innermost", optionarg == "true");
return;
}
if(key == "cegqi-midpoint") {
  options->assignBool(options::cegqiMidpoint, "cegqi-midpoint", optionarg == "true");
return;
}
if(key == "cegqi-min-bounds") {
  options->assignBool(options::cegqiMinBounds, "cegqi-min-bounds", optionarg == "true");
return;
}
if(key == "cegqi-model") {
  options->assignBool(options::cegqiModel, "cegqi-model", optionarg == "true");
return;
}
if(key == "cegqi-multi-inst") {
  options->assignBool(options::cegqiMultiInst, "cegqi-multi-inst", optionarg == "true");
return;
}
if(key == "cegqi-nested-qe") {
  options->assignBool(options::cegqiNestedQE, "cegqi-nested-qe", optionarg == "true");
return;
}
if(key == "cegqi-nopt") {
  options->assignBool(options::cegqiNopt, "cegqi-nopt", optionarg == "true");
return;
}
if(key == "cegqi-prereg-inst") {
  options->assignBool(options::cegqiPreRegInst, "cegqi-prereg-inst", optionarg == "true");
return;
}
if(key == "cegqi-repeat-lit") {
  options->assignBool(options::cegqiRepeatLit, "cegqi-repeat-lit", optionarg == "true");
return;
}
if(key == "cegqi-round-up-lia") {
  options->assignBool(options::cegqiRoundUpLowerLia, "cegqi-round-up-lia", optionarg == "true");
return;
}
if(key == "cegqi-sat") {
  options->assignBool(options::cegqiSat, "cegqi-sat", optionarg == "true");
return;
}
if(key == "cegqi-use-inf-int") {
  options->assignBool(options::cegqiUseInfInt, "cegqi-use-inf-int", optionarg == "true");
return;
}
if(key == "cegqi-use-inf-real") {
  options->assignBool(options::cegqiUseInfReal, "cegqi-use-inf-real", optionarg == "true");
return;
}
if(key == "cond-var-split-agg-quant") {
  options->assignBool(options::condVarSplitQuantAgg, "cond-var-split-agg-quant", optionarg == "true");
return;
}
if(key == "cond-var-split-quant") {
  options->assignBool(options::condVarSplitQuant, "cond-var-split-quant", optionarg == "true");
return;
}
if(key == "conjecture-filter-active-terms") {
  options->assignBool(options::conjectureFilterActiveTerms, "conjecture-filter-active-terms", optionarg == "true");
return;
}
if(key == "conjecture-filter-canonical") {
  options->assignBool(options::conjectureFilterCanonical, "conjecture-filter-canonical", optionarg == "true");
return;
}
if(key == "conjecture-filter-model") {
  options->assignBool(options::conjectureFilterModel, "conjecture-filter-model", optionarg == "true");
return;
}
if(key == "conjecture-gen") {
  options->assignBool(options::conjectureGen, "conjecture-gen", optionarg == "true");
return;
}
if(key == "conjecture-gen-gt-enum") {
  options->assign(options::conjectureGenGtEnum, "conjecture-gen-gt-enum", optionarg);
return;
}
if(key == "conjecture-gen-max-depth") {
  options->assign(options::conjectureGenMaxDepth, "conjecture-gen-max-depth", optionarg);
return;
}
if(key == "conjecture-gen-per-round") {
  options->assign(options::conjectureGenPerRound, "conjecture-gen-per-round", optionarg);
return;
}
if(key == "conjecture-gen-uee-intro") {
  options->assignBool(options::conjectureUeeIntro, "conjecture-gen-uee-intro", optionarg == "true");
return;
}
if(key == "conjecture-no-filter") {
  options->assignBool(options::conjectureNoFilter, "conjecture-no-filter", optionarg == "true");
return;
}
if(key == "dt-stc-ind") {
  options->assignBool(options::dtStcInduction, "dt-stc-ind", optionarg == "true");
return;
}
if(key == "dt-var-exp-quant") {
  options->assignBool(options::dtVarExpandQuant, "dt-var-exp-quant", optionarg == "true");
return;
}
if(key == "e-matching") {
  options->assignBool(options::eMatching, "e-matching", optionarg == "true");
return;
}
if(key == "elim-ext-arith-quant") {
  options->assignBool(options::elimExtArithQuant, "elim-ext-arith-quant", optionarg == "true");
return;
}
if(key == "elim-taut-quant") {
  options->assignBool(options::elimTautQuant, "elim-taut-quant", optionarg == "true");
return;
}
if(key == "ext-rewrite-quant") {
  options->assignBool(options::extRewriteQuant, "ext-rewrite-quant", optionarg == "true");
return;
}
if(key == "finite-model-find") {
  options->assignBool(options::finiteModelFind, "finite-model-find", optionarg == "true");
return;
}
if(key == "fmf-bound") {
  options->assignBool(options::fmfBound, "fmf-bound", optionarg == "true");
return;
}
if(key == "fmf-bound-int") {
  options->assignBool(options::fmfBoundInt, "fmf-bound-int", optionarg == "true");
return;
}
if(key == "fmf-bound-lazy") {
  options->assignBool(options::fmfBoundLazy, "fmf-bound-lazy", optionarg == "true");
return;
}
if(key == "fmf-fmc-simple") {
  options->assignBool(options::fmfFmcSimple, "fmf-fmc-simple", optionarg == "true");
return;
}
if(key == "fmf-fresh-dc") {
  options->assignBool(options::fmfFreshDistConst, "fmf-fresh-dc", optionarg == "true");
return;
}
if(key == "fmf-fun") {
  options->assignBool(options::fmfFunWellDefined, "fmf-fun", optionarg == "true");
return;
}
if(key == "fmf-fun-rlv") {
  options->assignBool(options::fmfFunWellDefinedRelevant, "fmf-fun-rlv", optionarg == "true");
return;
}
if(key == "fmf-inst-engine") {
  options->assignBool(options::fmfInstEngine, "fmf-inst-engine", optionarg == "true");
return;
}
if(key == "fmf-type-completion-thresh") {
  options->assign(options::fmfTypeCompletionThresh, "fmf-type-completion-thresh", optionarg);
return;
}
if(key == "fs-interleave") {
  options->assignBool(options::fullSaturateInterleave, "fs-interleave", optionarg == "true");
return;
}
if(key == "fs-stratify") {
  options->assignBool(options::fullSaturateStratify, "fs-stratify", optionarg == "true");
return;
}
if(key == "full-saturate-quant") {
  options->assignBool(options::fullSaturateQuant, "full-saturate-quant", optionarg == "true");
return;
}
if(key == "full-saturate-quant-limit") {
  options->assign(options::fullSaturateLimit, "full-saturate-quant-limit", optionarg);
return;
}
if(key == "full-saturate-quant-rd") {
  options->assignBool(options::fullSaturateQuantRd, "full-saturate-quant-rd", optionarg == "true");
return;
}
if(key == "global-negate") {
  options->assignBool(options::globalNegate, "global-negate", optionarg == "true");
return;
}
if(key == "ho-elim") {
  options->assignBool(options::hoElim, "ho-elim", optionarg == "true");
return;
}
if(key == "ho-elim-store-ax") {
  options->assignBool(options::hoElimStoreAx, "ho-elim-store-ax", optionarg == "true");
return;
}
if(key == "ho-matching") {
  options->assignBool(options::hoMatching, "ho-matching", optionarg == "true");
return;
}
if(key == "ho-matching-var-priority") {
  options->assignBool(options::hoMatchingVarArgPriority, "ho-matching-var-priority", optionarg == "true");
return;
}
if(key == "ho-merge-term-db") {
  options->assignBool(options::hoMergeTermDb, "ho-merge-term-db", optionarg == "true");
return;
}
if(key == "increment-triggers") {
  options->assignBool(options::incrementTriggers, "increment-triggers", optionarg == "true");
return;
}
if(key == "inst-level-input-only") {
  options->assignBool(options::instLevelInputOnly, "inst-level-input-only", optionarg == "true");
return;
}
if(key == "inst-max-level") {
  options->assign(options::instMaxLevel, "inst-max-level", optionarg);
return;
}
if(key == "inst-no-entail") {
  options->assignBool(options::instNoEntail, "inst-no-entail", optionarg == "true");
return;
}
if(key == "inst-no-model-true") {
  options->assignBool(options::instNoModelTrue, "inst-no-model-true", optionarg == "true");
return;
}
if(key == "inst-when-phase") {
  options->assign(options::instWhenPhase, "inst-when-phase", optionarg);
return;
}
if(key == "inst-when-strict-interleave") {
  options->assignBool(options::instWhenStrictInterleave, "inst-when-strict-interleave", optionarg == "true");
return;
}
if(key == "inst-when-tc-first") {
  options->assignBool(options::instWhenTcFirst, "inst-when-tc-first", optionarg == "true");
return;
}
if(key == "inst-when") {
  options->assign(options::instWhenMode, "inst-when", optionarg);
return;
}
if(key == "int-wf-ind") {
  options->assignBool(options::intWfInduction, "int-wf-ind", optionarg == "true");
return;
}
if(key == "ite-dtt-split-quant") {
  options->assignBool(options::iteDtTesterSplitQuant, "ite-dtt-split-quant", optionarg == "true");
return;
}
if(key == "ite-lift-quant") {
  options->assign(options::iteLiftQuant, "ite-lift-quant", optionarg);
return;
}
if(key == "literal-matching") {
  options->assign(options::literalMatchMode, "literal-matching", optionarg);
return;
}
if(key == "lte-restrict-inst-closure") {
  options->assignBool(options::lteRestrictInstClosure, "lte-restrict-inst-closure", optionarg == "true");
return;
}
if(key == "macros-quant") {
  options->assignBool(options::macrosQuant, "macros-quant", optionarg == "true");
return;
}
if(key == "macros-quant-mode") {
  options->assign(options::macrosQuantMode, "macros-quant-mode", optionarg);
return;
}
if(key == "mbqi-interleave") {
  options->assignBool(options::mbqiInterleave, "mbqi-interleave", optionarg == "true");
return;
}
if(key == "mbqi-one-inst-per-round") {
  options->assignBool(options::fmfOneInstPerRound, "mbqi-one-inst-per-round", optionarg == "true");
return;
}
if(key == "mbqi") {
  options->assign(options::mbqiMode, "mbqi", optionarg);
return;
}
if(key == "miniscope-quant") {
  options->assignBool(options::miniscopeQuant, "miniscope-quant", optionarg == "true");
return;
}
if(key == "miniscope-quant-fv") {
  options->assignBool(options::miniscopeQuantFreeVar, "miniscope-quant-fv", optionarg == "true");
return;
}
if(key == "multi-trigger-cache") {
  options->assignBool(options::multiTriggerCache, "multi-trigger-cache", optionarg == "true");
return;
}
if(key == "multi-trigger-linear") {
  options->assignBool(options::multiTriggerLinear, "multi-trigger-linear", optionarg == "true");
return;
}
if(key == "multi-trigger-priority") {
  options->assignBool(options::multiTriggerPriority, "multi-trigger-priority", optionarg == "true");
return;
}
if(key == "multi-trigger-when-single") {
  options->assignBool(options::multiTriggerWhenSingle, "multi-trigger-when-single", optionarg == "true");
return;
}
if(key == "partial-triggers") {
  options->assignBool(options::partialTriggers, "partial-triggers", optionarg == "true");
return;
}
if(key == "pre-skolem-quant") {
  options->assignBool(options::preSkolemQuant, "pre-skolem-quant", optionarg == "true");
return;
}
if(key == "pre-skolem-quant-agg") {
  options->assignBool(options::preSkolemQuantAgg, "pre-skolem-quant-agg", optionarg == "true");
return;
}
if(key == "pre-skolem-quant-nested") {
  options->assignBool(options::preSkolemQuantNested, "pre-skolem-quant-nested", optionarg == "true");
return;
}
if(key == "prenex-quant-user") {
  options->assignBool(options::prenexQuantUser, "prenex-quant-user", optionarg == "true");
return;
}
if(key == "prenex-quant") {
  options->assign(options::prenexQuant, "prenex-quant", optionarg);
return;
}
if(key == "pure-th-triggers") {
  options->assignBool(options::pureThTriggers, "pure-th-triggers", optionarg == "true");
return;
}
if(key == "purify-dt-triggers") {
  options->assignBool(options::purifyDtTriggers, "purify-dt-triggers", optionarg == "true");
return;
}
if(key == "purify-triggers") {
  options->assignBool(options::purifyTriggers, "purify-triggers", optionarg == "true");
return;
}
if(key == "qcf-all-conflict") {
  options->assignBool(options::qcfAllConflict, "qcf-all-conflict", optionarg == "true");
return;
}
if(key == "qcf-eager-check-rd") {
  options->assignBool(options::qcfEagerCheckRd, "qcf-eager-check-rd", optionarg == "true");
return;
}
if(key == "qcf-eager-test") {
  options->assignBool(options::qcfEagerTest, "qcf-eager-test", optionarg == "true");
return;
}
if(key == "qcf-nested-conflict") {
  options->assignBool(options::qcfNestedConflict, "qcf-nested-conflict", optionarg == "true");
return;
}
if(key == "qcf-skip-rd") {
  options->assignBool(options::qcfSkipRd, "qcf-skip-rd", optionarg == "true");
return;
}
if(key == "qcf-tconstraint") {
  options->assignBool(options::qcfTConstraint, "qcf-tconstraint", optionarg == "true");
return;
}
if(key == "qcf-vo-exp") {
  options->assignBool(options::qcfVoExp, "qcf-vo-exp", optionarg == "true");
return;
}
if(key == "quant-alpha-equiv") {
  options->assignBool(options::quantAlphaEquiv, "quant-alpha-equiv", optionarg == "true");
return;
}
if(key == "quant-anti-skolem") {
  options->assignBool(options::quantAntiSkolem, "quant-anti-skolem", optionarg == "true");
return;
}
if(key == "quant-cf") {
  options->assignBool(options::quantConflictFind, "quant-cf", optionarg == "true");
return;
}
if(key == "quant-cf-mode") {
  options->assign(options::qcfMode, "quant-cf-mode", optionarg);
return;
}
if(key == "quant-cf-when") {
  options->assign(options::qcfWhenMode, "quant-cf-when", optionarg);
return;
}
if(key == "quant-dsplit-mode") {
  options->assign(options::quantDynamicSplit, "quant-dsplit-mode", optionarg);
return;
}
if(key == "quant-epr") {
  options->assignBool(options::quantEpr, "quant-epr", optionarg == "true");
return;
}
if(key == "quant-epr-match") {
  options->assignBool(options::quantEprMatching, "quant-epr-match", optionarg == "true");
return;
}
if(key == "quant-fun-wd") {
  options->assignBool(options::quantFunWellDefined, "quant-fun-wd", optionarg == "true");
return;
}
if(key == "quant-ind") {
  options->assignBool(options::quantInduction, "quant-ind", optionarg == "true");
return;
}
if(key == "quant-model-ee") {
  options->assignBool(options::quantModelEe, "quant-model-ee", optionarg == "true");
return;
}
if(key == "quant-rep-mode") {
  options->assign(options::quantRepMode, "quant-rep-mode", optionarg);
return;
}
if(key == "quant-split") {
  options->assignBool(options::quantSplit, "quant-split", optionarg == "true");
return;
}
if(key == "register-quant-body-terms") {
  options->assignBool(options::registerQuantBodyTerms, "register-quant-body-terms", optionarg == "true");
return;
}
if(key == "relational-triggers") {
  options->assignBool(options::relationalTriggers, "relational-triggers", optionarg == "true");
return;
}
if(key == "relevant-triggers") {
  options->assignBool(options::relevantTriggers, "relevant-triggers", optionarg == "true");
return;
}
if(key == "strict-triggers") {
  options->assignBool(options::strictTriggers, "strict-triggers", optionarg == "true");
return;
}
if(key == "sygus") {
  options->assignBool(options::sygus, "sygus", optionarg == "true");
return;
}
if(key == "sygus-active-gen-cfactor") {
  options->assign(options::sygusActiveGenEnumConsts, "sygus-active-gen-cfactor", optionarg);
return;
}
if(key == "sygus-active-gen") {
  options->assign(options::sygusActiveGenMode, "sygus-active-gen", optionarg);
return;
}
if(key == "sygus-add-const-grammar") {
  options->assignBool(options::sygusAddConstGrammar, "sygus-add-const-grammar", optionarg == "true");
return;
}
if(key == "sygus-arg-relevant") {
  options->assignBool(options::sygusArgRelevant, "sygus-arg-relevant", optionarg == "true");
return;
}
if(key == "sygus-auto-unfold") {
  options->assignBool(options::sygusInvAutoUnfold, "sygus-auto-unfold", optionarg == "true");
return;
}
if(key == "sygus-bool-ite-return-const") {
  options->assignBool(options::sygusBoolIteReturnConst, "sygus-bool-ite-return-const", optionarg == "true");
return;
}
if(key == "sygus-core-connective") {
  options->assignBool(options::sygusCoreConnective, "sygus-core-connective", optionarg == "true");
return;
}
if(key == "sygus-crepair-abort") {
  options->assignBool(options::sygusConstRepairAbort, "sygus-crepair-abort", optionarg == "true");
return;
}
if(key == "sygus-eval-opt") {
  options->assignBool(options::sygusEvalOpt, "sygus-eval-opt", optionarg == "true");
return;
}
if(key == "sygus-eval-unfold") {
  options->assignBool(options::sygusEvalUnfold, "sygus-eval-unfold", optionarg == "true");
return;
}
if(key == "sygus-eval-unfold-bool") {
  options->assignBool(options::sygusEvalUnfoldBool, "sygus-eval-unfold-bool", optionarg == "true");
return;
}
if(key == "sygus-expr-miner-check-timeout") {
  options->assign(options::sygusExprMinerCheckTimeout, "sygus-expr-miner-check-timeout", optionarg);
return;
}
if(key == "sygus-expr-miner-check-use-export") {
  options->assignBool(options::sygusExprMinerCheckUseExport, "sygus-expr-miner-check-use-export", optionarg == "true");
return;
}
if(key == "sygus-ext-rew") {
  options->assignBool(options::sygusExtRew, "sygus-ext-rew", optionarg == "true");
return;
}
if(key == "sygus-filter-sol-rev") {
  options->assignBool(options::sygusFilterSolRevSubsume, "sygus-filter-sol-rev", optionarg == "true");
return;
}
if(key == "sygus-filter-sol") {
  options->assign(options::sygusFilterSolMode, "sygus-filter-sol", optionarg);
return;
}
if(key == "sygus-grammar-cons") {
  options->assign(options::sygusGrammarConsMode, "sygus-grammar-cons", optionarg);
return;
}
if(key == "sygus-grammar-norm") {
  options->assignBool(options::sygusGrammarNorm, "sygus-grammar-norm", optionarg == "true");
return;
}
if(key == "sygus-inference") {
  options->assignBool(options::sygusInference, "sygus-inference", optionarg == "true");
return;
}
if(key == "sygus-inst") {
  options->assignBool(options::sygusInst, "sygus-inst", optionarg == "true");
return;
}
if(key == "sygus-inv-templ-when-sg") {
  options->assignBool(options::sygusInvTemplWhenSyntax, "sygus-inv-templ-when-sg", optionarg == "true");
return;
}
if(key == "sygus-inv-templ") {
  options->assign(options::sygusInvTemplMode, "sygus-inv-templ", optionarg);
return;
}
if(key == "sygus-min-grammar") {
  options->assignBool(options::sygusMinGrammar, "sygus-min-grammar", optionarg == "true");
return;
}
if(key == "sygus-pbe") {
  options->assignBool(options::sygusUnifPbe, "sygus-pbe", optionarg == "true");
return;
}
if(key == "sygus-pbe-multi-fair") {
  options->assignBool(options::sygusPbeMultiFair, "sygus-pbe-multi-fair", optionarg == "true");
return;
}
if(key == "sygus-pbe-multi-fair-diff") {
  options->assign(options::sygusPbeMultiFairDiff, "sygus-pbe-multi-fair-diff", optionarg);
return;
}
if(key == "sygus-qe-preproc") {
  options->assignBool(options::sygusQePreproc, "sygus-qe-preproc", optionarg == "true");
return;
}
if(key == "sygus-query-gen") {
  options->assignBool(options::sygusQueryGen, "sygus-query-gen", optionarg == "true");
return;
}
if(key == "sygus-query-gen-check") {
  options->assignBool(options::sygusQueryGenCheck, "sygus-query-gen-check", optionarg == "true");
return;
}
if(key == "sygus-query-gen-dump-files") {
  options->assign(options::sygusQueryGenDumpFiles, "sygus-query-gen-dump-files", optionarg);
return;
}
if(key == "sygus-query-gen-thresh") {
  options->assign(options::sygusQueryGenThresh, "sygus-query-gen-thresh", optionarg);
return;
}
if(key == "sygus-rec-fun") {
  options->assignBool(options::sygusRecFun, "sygus-rec-fun", optionarg == "true");
return;
}
if(key == "sygus-rec-fun-eval-limit") {
  options->assign(options::sygusRecFunEvalLimit, "sygus-rec-fun-eval-limit", optionarg);
return;
}
if(key == "sygus-repair-const") {
  options->assignBool(options::sygusRepairConst, "sygus-repair-const", optionarg == "true");
return;
}
if(key == "sygus-repair-const-timeout") {
  options->assign(options::sygusRepairConstTimeout, "sygus-repair-const-timeout", optionarg);
return;
}
if(key == "sygus-rr") {
  options->assignBool(options::sygusRew, "sygus-rr", optionarg == "true");
return;
}
if(key == "sygus-rr-synth") {
  options->assignBool(options::sygusRewSynth, "sygus-rr-synth", optionarg == "true");
return;
}
if(key == "sygus-rr-synth-accel") {
  options->assignBool(options::sygusRewSynthAccel, "sygus-rr-synth-accel", optionarg == "true");
return;
}
if(key == "sygus-rr-synth-check") {
  options->assignBool(options::sygusRewSynthCheck, "sygus-rr-synth-check", optionarg == "true");
return;
}
if(key == "sygus-rr-synth-filter-cong") {
  options->assignBool(options::sygusRewSynthFilterCong, "sygus-rr-synth-filter-cong", optionarg == "true");
return;
}
if(key == "sygus-rr-synth-filter-match") {
  options->assignBool(options::sygusRewSynthFilterMatch, "sygus-rr-synth-filter-match", optionarg == "true");
return;
}
if(key == "sygus-rr-synth-filter-nl") {
  options->assignBool(options::sygusRewSynthFilterNonLinear, "sygus-rr-synth-filter-nl", optionarg == "true");
return;
}
if(key == "sygus-rr-synth-filter-order") {
  options->assignBool(options::sygusRewSynthFilterOrder, "sygus-rr-synth-filter-order", optionarg == "true");
return;
}
if(key == "sygus-rr-synth-input") {
  options->assignBool(options::sygusRewSynthInput, "sygus-rr-synth-input", optionarg == "true");
return;
}
if(key == "sygus-rr-synth-input-nvars") {
  options->assign(options::sygusRewSynthInputNVars, "sygus-rr-synth-input-nvars", optionarg);
return;
}
if(key == "sygus-rr-synth-input-use-bool") {
  options->assignBool(options::sygusRewSynthInputUseBool, "sygus-rr-synth-input-use-bool", optionarg == "true");
return;
}
if(key == "sygus-rr-synth-rec") {
  options->assignBool(options::sygusRewSynthRec, "sygus-rr-synth-rec", optionarg == "true");
return;
}
if(key == "sygus-rr-verify") {
  options->assignBool(options::sygusRewVerify, "sygus-rr-verify", optionarg == "true");
return;
}
if(key == "sygus-rr-verify-abort") {
  options->assignBool(options::sygusRewVerifyAbort, "sygus-rr-verify-abort", optionarg == "true");
return;
}
if(key == "sygus-sample-fp-uniform") {
  options->assignBool(options::sygusSampleFpUniform, "sygus-sample-fp-uniform", optionarg == "true");
return;
}
if(key == "sygus-sample-grammar") {
  options->assignBool(options::sygusSampleGrammar, "sygus-sample-grammar", optionarg == "true");
return;
}
if(key == "sygus-samples") {
  options->assign(options::sygusSamples, "sygus-samples", optionarg);
return;
}
if(key == "sygus-si-abort") {
  options->assignBool(options::cegqiSingleInvAbort, "sygus-si-abort", optionarg == "true");
return;
}
if(key == "sygus-si-partial") {
  options->assignBool(options::cegqiSingleInvPartial, "sygus-si-partial", optionarg == "true");
return;
}
if(key == "sygus-si-rcons-limit") {
  options->assign(options::cegqiSingleInvReconstructLimit, "sygus-si-rcons-limit", optionarg);
return;
}
if(key == "sygus-si-rcons") {
  options->assign(options::cegqiSingleInvReconstruct, "sygus-si-rcons", optionarg);
return;
}
if(key == "sygus-si-reconstruct-const") {
  options->assignBool(options::cegqiSingleInvReconstructConst, "sygus-si-reconstruct-const", optionarg == "true");
return;
}
if(key == "sygus-si") {
  options->assign(options::cegqiSingleInvMode, "sygus-si", optionarg);
return;
}
if(key == "sygus-stream") {
  options->assignBool(options::sygusStream, "sygus-stream", optionarg == "true");
return;
}
if(key == "sygus-templ-embed-grammar") {
  options->assignBool(options::sygusTemplEmbedGrammar, "sygus-templ-embed-grammar", optionarg == "true");
return;
}
if(key == "sygus-unif-cond-independent-no-repeat-sol") {
  options->assignBool(options::sygusUnifCondIndNoRepeatSol, "sygus-unif-cond-independent-no-repeat-sol", optionarg == "true");
return;
}
if(key == "sygus-unif-pi") {
  options->assign(options::sygusUnifPi, "sygus-unif-pi", optionarg);
return;
}
if(key == "sygus-unif-shuffle-cond") {
  options->assignBool(options::sygusUnifShuffleCond, "sygus-unif-shuffle-cond", optionarg == "true");
return;
}
if(key == "sygus-verify-subcall") {
  options->assignBool(options::sygusVerifySubcall, "sygus-verify-subcall", optionarg == "true");
return;
}
if(key == "term-db-mode") {
  options->assign(options::termDbMode, "term-db-mode", optionarg);
return;
}
if(key == "track-inst-lemmas") {
  options->assignBool(options::trackInstLemmas, "track-inst-lemmas", optionarg == "true");
return;
}
if(key == "trigger-active-sel") {
  options->assign(options::triggerActiveSelMode, "trigger-active-sel", optionarg);
return;
}
if(key == "trigger-sel") {
  options->assign(options::triggerSelMode, "trigger-sel", optionarg);
return;
}
if(key == "user-pat") {
  options->assign(options::userPatternsQuant, "user-pat", optionarg);
return;
}
if(key == "var-elim-quant") {
  options->assignBool(options::varElimQuant, "var-elim-quant", optionarg == "true");
return;
}
if(key == "var-ineq-elim-quant") {
  options->assignBool(options::varIneqElimQuant, "var-ineq-elim-quant", optionarg == "true");
return;
}
if(key == "sep-check-neg") {
  options->assignBool(options::sepCheckNeg, "sep-check-neg", optionarg == "true");
return;
}
if(key == "sep-child-refine") {
  options->assignBool(options::sepChildRefine, "sep-child-refine", optionarg == "true");
return;
}
if(key == "sep-deq-c") {
  options->assignBool(options::sepDisequalC, "sep-deq-c", optionarg == "true");
return;
}
if(key == "sep-exp") {
  options->assignBool(options::sepExp, "sep-exp", optionarg == "true");
return;
}
if(key == "sep-min-refine") {
  options->assignBool(options::sepMinimalRefine, "sep-min-refine", optionarg == "true");
return;
}
if(key == "sep-pre-skolem-emp") {
  options->assignBool(options::sepPreSkolemEmp, "sep-pre-skolem-emp", optionarg == "true");
return;
}
if(key == "sets-ext") {
  options->assignBool(options::setsExt, "sets-ext", optionarg == "true");
return;
}
if(key == "sets-infer-as-lemmas") {
  options->assignBool(options::setsInferAsLemmas, "sets-infer-as-lemmas", optionarg == "true");
return;
}
if(key == "sets-proxy-lemmas") {
  options->assignBool(options::setsProxyLemmas, "sets-proxy-lemmas", optionarg == "true");
return;
}
if(key == "abstract-values") {
  options->assignBool(options::abstractValues, "abstract-values", optionarg == "true");
return;
}
if(key == "ackermann") {
  options->assignBool(options::ackermann, "ackermann", optionarg == "true");
return;
}
if(key == "bitblast-step") {
  options->assign(options::bitblastStep, "bitblast-step", optionarg);
return;
}
if(key == "block-models") {
  options->assign(options::blockModelsMode, "block-models", optionarg);
return;
}
if(key == "bv-eager-assert-step") {
  options->assign(options::bvEagerAssertStep, "bv-eager-assert-step", optionarg);
return;
}
if(key == "bv-propagation-step") {
  options->assign(options::bvPropagationStep, "bv-propagation-step", optionarg);
return;
}
if(key == "bv-sat-conflict-step") {
  options->assign(options::bvSatConflictStep, "bv-sat-conflict-step", optionarg);
return;
}
if(key == "check-abducts") {
  options->assignBool(options::checkAbducts, "check-abducts", optionarg == "true");
return;
}
if(key == "check-models") {
  options->assignBool(options::checkModels, "check-models", optionarg == "true");
return;
}
if(key == "check-proofs") {
  options->assignBool(options::checkProofs, "check-proofs", optionarg == "true");
return;
}
if(key == "check-synth-sol") {
  options->assignBool(options::checkSynthSol, "check-synth-sol", optionarg == "true");
return;
}
if(key == "check-unsat-cores") {
  options->assignBool(options::checkUnsatCores, "check-unsat-cores", optionarg == "true");
return;
}
if(key == "cnf-step") {
  options->assign(options::cnfStep, "cnf-step", optionarg);
return;
}
if(key == "cpu-time") {
  options->assignBool(options::cpuTime, "cpu-time", optionarg == "true");
return;
}
if(key == "debug-check-models") {
  options->assignBool(options::debugCheckModels, "debug-check-models", optionarg == "true");
return;
}
if(key == "decision-step") {
  options->assign(options::decisionStep, "decision-step", optionarg);
return;
}
if(key == "diagnostic-output-channel") {
  options->assign(options::diagnosticChannelName, "diagnostic-output-channel", optionarg);
return;
}
if(key == "dump-instantiations") {
  options->assignBool(options::dumpInstantiations, "dump-instantiations", optionarg == "true");
return;
}
if(key == "dump-models") {
  options->assignBool(options::dumpModels, "dump-models", optionarg == "true");
return;
}
if(key == "dump-proofs") {
  options->assignBool(options::dumpProofs, "dump-proofs", optionarg == "true");
return;
}
if(key == "dump-synth") {
  options->assignBool(options::dumpSynth, "dump-synth", optionarg == "true");
return;
}
if(key == "dump-to") {
  options->assign(options::dumpToFileName, "dump-to", optionarg);
return;
}
if(key == "dump-unsat-cores") {
  options->assignBool(options::dumpUnsatCores, "dump-unsat-cores", optionarg == "true");
return;
}
if(key == "dump-unsat-cores-full") {
  options->assignBool(options::dumpUnsatCoresFull, "dump-unsat-cores-full", optionarg == "true");
return;
}
if(key == "dump") {
  options->assign(options::dumpModeString, "dump", optionarg);
return;
}
if(key == "expand-definitions") {
  options->assignBool(options::expandDefinitions, "expand-definitions", optionarg == "true");
return;
}
if(key == "ext-rew-prep") {
  options->assignBool(options::extRewPrep, "ext-rew-prep", optionarg == "true");
return;
}
if(key == "ext-rew-prep-agg") {
  options->assignBool(options::extRewPrepAgg, "ext-rew-prep-agg", optionarg == "true");
return;
}
if(key == "force-no-limit-cpu-while-dump") {
  options->assignBool(options::forceNoLimitCpuWhileDump, "force-no-limit-cpu-while-dump", optionarg == "true");
return;
}
if(key == "hard-limit") {
  options->assignBool(options::hardLimit, "hard-limit", optionarg == "true");
return;
}
if(key == "incremental") {
  options->assignBool(options::incrementalSolving, "incremental", optionarg == "true");
return;
}
if(key == "interactive-mode") {
  options->assignBool(options::interactiveMode, "interactive-mode", optionarg == "true");
return;
}
if(key == "ite-simp") {
  options->assignBool(options::doITESimp, "ite-simp", optionarg == "true");
return;
}
if(key == "lemma-step") {
  options->assign(options::lemmaStep, "lemma-step", optionarg);
return;
}
if(key == "model-cores") {
  options->assign(options::modelCoresMode, "model-cores", optionarg);
return;
}
if(key == "model-u-dt-enum") {
  options->assignBool(options::modelUninterpDtEnum, "model-u-dt-enum", optionarg == "true");
return;
}
if(key == "model-witness-value") {
  options->assignBool(options::modelWitnessValue, "model-witness-value", optionarg == "true");
return;
}
if(key == "on-repeat-ite-simp") {
  options->assignBool(options::doITESimpOnRepeat, "on-repeat-ite-simp", optionarg == "true");
return;
}
if(key == "parse-step") {
  options->assign(options::parseStep, "parse-step", optionarg);
return;
}
if(key == "preprocess-step") {
  options->assign(options::preprocessStep, "preprocess-step", optionarg);
return;
}
if(key == "produce-abducts") {
  options->assignBool(options::produceAbducts, "produce-abducts", optionarg == "true");
return;
}
if(key == "produce-assertions") {
  options->assignBool(options::produceAssertions, "produce-assertions", optionarg == "true");
return;
}
if(key == "produce-assignments") {
  options->assignBool(options::produceAssignments, "produce-assignments", optionarg == "true");
return;
}
if(key == "produce-models") {
  options->assignBool(options::produceModels, "produce-models", optionarg == "true");
return;
}
if(key == "produce-unsat-assumptions") {
  options->assignBool(options::unsatAssumptions, "produce-unsat-assumptions", optionarg == "true");
return;
}
if(key == "produce-unsat-cores") {
  options->assignBool(options::unsatCores, "produce-unsat-cores", optionarg == "true");
return;
}
if(key == "produce-proofs" || key == "proof") {
  options->assignBool(options::proof, "produce-proofs", optionarg == "true");
return;
}
if(key == "quantifier-step") {
  options->assign(options::quantifierStep, "quantifier-step", optionarg);
return;
}
if(key == "regular-output-channel") {
  options->assign(options::regularChannelName, "regular-output-channel", optionarg);
return;
}
if(key == "repeat-simp") {
  options->assignBool(options::repeatSimp, "repeat-simp", optionarg == "true");
return;
}
if(key == "restart-step") {
  options->assign(options::restartStep, "restart-step", optionarg);
return;
}
if(key == "rewrite-step") {
  options->assign(options::rewriteStep, "rewrite-step", optionarg);
return;
}
if(key == "reproducible-resource-limit" || key == "rlimit-per") {
  options->assign(options::perCallResourceLimit, "reproducible-resource-limit", optionarg);
return;
}
if(key == "rlimit") {
  options->assign(options::cumulativeResourceLimit, "rlimit", optionarg);
return;
}
if(key == "sat-conflict-step") {
  options->assign(options::satConflictStep, "sat-conflict-step", optionarg);
return;
}
if(key == "simp-ite-compress") {
  options->assignBool(options::compressItes, "simp-ite-compress", optionarg == "true");
return;
}
if(key == "simp-ite-hunt-zombies") {
  options->assign(options::zombieHuntThreshold, "simp-ite-hunt-zombies", optionarg);
return;
}
if(key == "simp-with-care") {
  options->assignBool(options::simplifyWithCareEnabled, "simp-with-care", optionarg == "true");
return;
}
if(key == "simplification" || key == "simplification-mode") {
  options->assign(options::simplificationMode, "simplification-mode", optionarg);
return;
}
if(key == "solve-bv-as-int") {
  options->assign(options::solveBVAsInt, "solve-bv-as-int", optionarg);
return;
}
if(key == "solve-int-as-bv") {
  options->assign(options::solveIntAsBV, "solve-int-as-bv", optionarg);
return;
}
if(key == "solve-real-as-int") {
  options->assignBool(options::solveRealAsInt, "solve-real-as-int", optionarg == "true");
return;
}
if(key == "sort-inference") {
  options->assignBool(options::sortInference, "sort-inference", optionarg == "true");
return;
}
if(key == "static-learning") {
  options->assignBool(options::doStaticLearning, "static-learning", optionarg == "true");
return;
}
if(key == "sygus-out") {
  options->assign(options::sygusOut, "sygus-out", optionarg);
return;
}
if(key == "sygus-print-callbacks") {
  options->assignBool(options::sygusPrintCallbacks, "sygus-print-callbacks", optionarg == "true");
return;
}
if(key == "theory-check-step") {
  options->assign(options::theoryCheckStep, "theory-check-step", optionarg);
return;
}
if(key == "tlimit-per") {
  options->assign(options::perCallMillisecondLimit, "tlimit-per", optionarg);
return;
}
if(key == "tlimit") {
  options->assign(options::cumulativeMillisecondLimit, "tlimit", optionarg);
return;
}
if(key == "unconstrained-simp") {
  options->assignBool(options::unconstrainedSimp, "unconstrained-simp", optionarg == "true");
return;
}
if(key == "re-elim") {
  options->assignBool(options::regExpElim, "re-elim", optionarg == "true");
return;
}
if(key == "re-elim-agg") {
  options->assignBool(options::regExpElimAgg, "re-elim-agg", optionarg == "true");
return;
}
if(key == "re-inter-mode") {
  options->assign(options::stringRegExpInterMode, "re-inter-mode", optionarg);
return;
}
if(key == "strings-check-entail-len") {
  options->assignBool(options::stringCheckEntailLen, "strings-check-entail-len", optionarg == "true");
return;
}
if(key == "strings-eager") {
  options->assignBool(options::stringEager, "strings-eager", optionarg == "true");
return;
}
if(key == "strings-eager-len") {
  options->assignBool(options::stringEagerLen, "strings-eager-len", optionarg == "true");
return;
}
if(key == "strings-exp") {
  options->assignBool(options::stringExp, "strings-exp", optionarg == "true");
return;
}
if(key == "strings-ff") {
  options->assignBool(options::stringFlatForms, "strings-ff", optionarg == "true");
return;
}
if(key == "strings-fmf") {
  options->assignBool(options::stringFMF, "strings-fmf", optionarg == "true");
return;
}
if(key == "strings-guess-model") {
  options->assignBool(options::stringGuessModel, "strings-guess-model", optionarg == "true");
return;
}
if(key == "strings-infer-as-lemmas") {
  options->assignBool(options::stringInferAsLemmas, "strings-infer-as-lemmas", optionarg == "true");
return;
}
if(key == "strings-infer-sym") {
  options->assignBool(options::stringInferSym, "strings-infer-sym", optionarg == "true");
return;
}
if(key == "strings-inm") {
  options->assignBool(options::stringIgnNegMembership, "strings-inm", optionarg == "true");
return;
}
if(key == "strings-lazy-pp") {
  options->assignBool(options::stringLazyPreproc, "strings-lazy-pp", optionarg == "true");
return;
}
if(key == "strings-len-norm") {
  options->assignBool(options::stringLenNorm, "strings-len-norm", optionarg == "true");
return;
}
if(key == "strings-lprop-csp") {
  options->assignBool(options::stringLenPropCsp, "strings-lprop-csp", optionarg == "true");
return;
}
if(key == "strings-min-prefix-explain") {
  options->assignBool(options::stringMinPrefixExplain, "strings-min-prefix-explain", optionarg == "true");
return;
}
if(key == "strings-print-ascii") {
  options->assignBool(options::stdPrintASCII, "strings-print-ascii", optionarg == "true");
return;
}
if(key == "strings-process-loop-mode") {
  options->assign(options::stringProcessLoopMode, "strings-process-loop-mode", optionarg);
return;
}
if(key == "strings-rexplain-lemmas") {
  options->assignBool(options::stringRExplainLemmas, "strings-rexplain-lemmas", optionarg == "true");
return;
}
if(key == "assign-function-values") {
  options->assignBool(options::assignFunctionValues, "assign-function-values", optionarg == "true");
return;
}
if(key == "condense-function-values") {
  options->assignBool(options::condenseFunctionValues, "condense-function-values", optionarg == "true");
return;
}
if(key == "theoryof-mode") {
  options->assign(options::theoryOfMode, "theoryof-mode", optionarg);
return;
}
if(key == "symmetry-breaker" || key == "uf-symmetry-breaker") {
  options->assignBool(options::ufSymmetryBreaker, "uf-symmetry-breaker", optionarg == "true");
return;
}
if(key == "uf-ho") {
  options->assignBool(options::ufHo, "uf-ho", optionarg == "true");
return;
}
if(key == "uf-ho-ext") {
  options->assignBool(options::ufHoExt, "uf-ho-ext", optionarg == "true");
return;
}
if(key == "uf-ss-abort-card") {
  options->assign(options::ufssAbortCardinality, "uf-ss-abort-card", optionarg);
return;
}
if(key == "uf-ss-fair") {
  options->assignBool(options::ufssFairness, "uf-ss-fair", optionarg == "true");
return;
}
if(key == "uf-ss-fair-monotone") {
  options->assignBool(options::ufssFairnessMonotone, "uf-ss-fair-monotone", optionarg == "true");
return;
}
if(key == "uf-ss-totality") {
  options->assignBool(options::ufssTotality, "uf-ss-totality", optionarg == "true");
return;
}
if(key == "uf-ss-totality-limited") {
  options->assign(options::ufssTotalityLimited, "uf-ss-totality-limited", optionarg);
return;
}
if(key == "uf-ss-totality-sym-break") {
  options->assignBool(options::ufssTotalitySymBreak, "uf-ss-totality-sym-break", optionarg == "true");
return;
}
if(key == "uf-ss") {
  options->assign(options::ufssMode, "uf-ss", optionarg);
return;
}


  throw UnrecognizedOptionException(key);
}

std::string Options::getOption(const std::string& key) const
{
  Trace("options") << "SMT getOption(" << key << ")" << std::endl;

  if (key == "approx-branch-depth") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::maxApproxDepth();
return ss.str();
}
if (key == "arith-brab") {
return options::brabTest() ? "true" : "false";
}
if (key == "arith-no-partial-fun") {
return options::arithNoPartialFun() ? "true" : "false";
}
if (key == "arith-prop-clauses") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::arithPropAsLemmaLength();
return ss.str();
}
if (key == "arith-prop") {
std::stringstream ss;
ss << options::arithPropagationMode();
return ss.str();
}
if (key == "arith-rewrite-equalities") {
return options::arithRewriteEq() ? "true" : "false";
}
if (key == "collect-pivot-stats") {
return options::collectPivots() ? "true" : "false";
}
if (key == "cut-all-bounded") {
return options::doCutAllBounded() ? "true" : "false";
}
if (key == "dio-decomps") {
return options::exportDioDecompositions() ? "true" : "false";
}
if (key == "dio-repeat") {
return options::dioRepeat() ? "true" : "false";
}
if (key == "dio-solver") {
return options::arithDioSolver() ? "true" : "false";
}
if (key == "dio-turns") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::dioSolverTurns();
return ss.str();
}
if (key == "error-selection-rule") {
std::stringstream ss;
ss << options::arithErrorSelectionRule();
return ss.str();
}
if (key == "fc-penalties") {
return options::havePenalties() ? "true" : "false";
}
if (key == "heuristic-pivots") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::arithHeuristicPivots();
return ss.str();
}
if (key == "lemmas-on-replay-failure") {
return options::replayFailureLemma() ? "true" : "false";
}
if (key == "maxCutsInContext") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::maxCutsInContext();
return ss.str();
}
if (key == "miplib-trick") {
return options::arithMLTrick() ? "true" : "false";
}
if (key == "miplib-trick-subs") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::arithMLTrickSubstitutions();
return ss.str();
}
if (key == "new-prop") {
return options::newProp() ? "true" : "false";
}
if (key == "nl-ext") {
return options::nlExt() ? "true" : "false";
}
if (key == "nl-ext-ent-conf") {
return options::nlExtEntailConflicts() ? "true" : "false";
}
if (key == "nl-ext-factor") {
return options::nlExtFactor() ? "true" : "false";
}
if (key == "nl-ext-inc-prec") {
return options::nlExtIncPrecision() ? "true" : "false";
}
if (key == "nl-ext-purify") {
return options::nlExtPurify() ? "true" : "false";
}
if (key == "nl-ext-rbound") {
return options::nlExtResBound() ? "true" : "false";
}
if (key == "nl-ext-rewrite") {
return options::nlExtRewrites() ? "true" : "false";
}
if (key == "nl-ext-split-zero") {
return options::nlExtSplitZero() ? "true" : "false";
}
if (key == "nl-ext-tf-taylor-deg") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::nlExtTfTaylorDegree();
return ss.str();
}
if (key == "nl-ext-tf-tplanes") {
return options::nlExtTfTangentPlanes() ? "true" : "false";
}
if (key == "nl-ext-tplanes") {
return options::nlExtTangentPlanes() ? "true" : "false";
}
if (key == "nl-ext-tplanes-interleave") {
return options::nlExtTangentPlanesInterleave() ? "true" : "false";
}
if (key == "pb-rewrites") {
return options::pbRewrites() ? "true" : "false";
}
if (key == "pivot-threshold") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::arithPivotThreshold();
return ss.str();
}
if (key == "pp-assert-max-sub-size") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::ppAssertMaxSubSize();
return ss.str();
}
if (key == "prop-row-length") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::arithPropagateMaxLength();
return ss.str();
}
if (key == "replay-early-close-depth") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::replayEarlyCloseDepths();
return ss.str();
}
if (key == "replay-failure-penalty") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::replayFailurePenalty();
return ss.str();
}
if (key == "replay-lemma-reject-cut") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::lemmaRejectCutSize();
return ss.str();
}
if (key == "replay-num-err-penalty") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::replayNumericFailurePenalty();
return ss.str();
}
if (key == "replay-reject-cut") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::replayRejectCutSize();
return ss.str();
}
if (key == "replay-soi-major-threshold-pen") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::soiApproxMajorFailurePen();
return ss.str();
}
if (key == "replay-soi-major-threshold") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::soiApproxMajorFailure();
return ss.str();
}
if (key == "replay-soi-minor-threshold-pen") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::soiApproxMinorFailurePen();
return ss.str();
}
if (key == "replay-soi-minor-threshold") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::soiApproxMinorFailure();
return ss.str();
}
if (key == "restrict-pivots") {
return options::restrictedPivots() ? "true" : "false";
}
if (key == "revert-arith-models-on-unsat") {
return options::revertArithModels() ? "true" : "false";
}
if (key == "rr-turns") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::rrTurns();
return ss.str();
}
if (key == "se-solve-int") {
return options::trySolveIntStandardEffort() ? "true" : "false";
}
if (key == "simplex-check-period") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::arithSimplexCheckPeriod();
return ss.str();
}
if (key == "soi-qe") {
return options::soiQuickExplain() ? "true" : "false";
}
if (key == "standard-effort-variable-order-pivots") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::arithStandardCheckVarOrderPivots();
return ss.str();
}
if (key == "unate-lemmas") {
std::stringstream ss;
ss << options::arithUnateLemmaMode();
return ss.str();
}
if (key == "use-approx") {
return options::useApprox() ? "true" : "false";
}
if (key == "use-fcsimplex") {
return options::useFC() ? "true" : "false";
}
if (key == "use-soi") {
return options::useSOI() ? "true" : "false";
}
if (key == "arrays-config") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::arraysConfig();
return ss.str();
}
if (key == "arrays-eager-index") {
return options::arraysEagerIndexSplitting() ? "true" : "false";
}
if (key == "arrays-eager-lemmas") {
return options::arraysEagerLemmas() ? "true" : "false";
}
if (key == "arrays-lazy-rintro1") {
return options::arraysLazyRIntro1() ? "true" : "false";
}
if (key == "arrays-model-based") {
return options::arraysModelBased() ? "true" : "false";
}
if (key == "arrays-optimize-linear") {
return options::arraysOptimizeLinear() ? "true" : "false";
}
if (key == "arrays-prop") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::arraysPropagate();
return ss.str();
}
if (key == "arrays-reduce-sharing") {
return options::arraysReduceSharing() ? "true" : "false";
}
if (key == "arrays-weak-equiv") {
return options::arraysWeakEquivalence() ? "true" : "false";
}
if (key == "input-language" || key == "lang") {
std::stringstream ss;
ss << options::inputLanguage();
return ss.str();
}
if (key == "output-lang" || key == "output-language") {
std::stringstream ss;
ss << options::outputLanguage();
return ss.str();
}
if (key == "parse-only") {
return options::parseOnly() ? "true" : "false";
}
if (key == "preprocess-only") {
return options::preprocessOnly() ? "true" : "false";
}
if (key == "print-success") {
return options::printSuccess() ? "true" : "false";
}
if (key == "statistics" || key == "stats") {
return options::statistics() ? "true" : "false";
}
if (key == "stats-every-query") {
return options::statsEveryQuery() ? "true" : "false";
}
if (key == "stats-hide-zeros") {
return options::statsHideZeros() ? "true" : "false";
}
if (key == "verbosity") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::verbosity();
return ss.str();
}
if (key == "bitblast-aig") {
return options::bitvectorAig() ? "true" : "false";
}
if (key == "bitblast") {
std::stringstream ss;
ss << options::bitblastMode();
return ss.str();
}
if (key == "bitwise-eq") {
return options::bitwiseEq() ? "true" : "false";
}
if (key == "bool-to-bv") {
std::stringstream ss;
ss << options::boolToBitvector();
return ss.str();
}
if (key == "bv-abstraction") {
return options::bvAbstraction() ? "true" : "false";
}
if (key == "bv-aig-simp") {
std::stringstream ss;
ss << options::bitvectorAigSimplifications();
return ss.str();
}
if (key == "bv-alg-extf") {
return options::bvAlgExtf() ? "true" : "false";
}
if (key == "bv-algebraic-budget") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::bitvectorAlgebraicBudget();
return ss.str();
}
if (key == "bv-algebraic-solver") {
return options::bitvectorAlgebraicSolver() ? "true" : "false";
}
if (key == "bv-div-zero-const") {
return options::bitvectorDivByZeroConst() ? "true" : "false";
}
if (key == "bv-eager-explanations") {
return options::bvEagerExplanations() ? "true" : "false";
}
if (key == "bv-eq-slicer") {
std::stringstream ss;
ss << options::bitvectorEqualitySlicer();
return ss.str();
}
if (key == "bv-eq-solver") {
return options::bitvectorEqualitySolver() ? "true" : "false";
}
if (key == "bv-extract-arith") {
return options::bvExtractArithRewrite() ? "true" : "false";
}
if (key == "bv-gauss-elim") {
return options::bvGaussElim() ? "true" : "false";
}
if (key == "bv-inequality-solver") {
return options::bitvectorInequalitySolver() ? "true" : "false";
}
if (key == "bv-intro-pow2") {
return options::bvIntroducePow2() ? "true" : "false";
}
if (key == "bv-lazy-reduce-extf") {
return options::bvLazyReduceExtf() ? "true" : "false";
}
if (key == "bv-lazy-rewrite-extf") {
return options::bvLazyRewriteExtf() ? "true" : "false";
}
if (key == "bv-num-func") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::bvNumFunc();
return ss.str();
}
if (key == "bv-optimize-sat-proof") {
std::stringstream ss;
ss << options::bvOptimizeSatProof();
return ss.str();
}
if (key == "bv-print-consts-as-indexed-symbols") {
return options::bvPrintConstsAsIndexedSymbols() ? "true" : "false";
}
if (key == "bv-proof-format") {
std::stringstream ss;
ss << options::bvProofFormat();
return ss.str();
}
if (key == "bv-propagate") {
return options::bitvectorPropagate() ? "true" : "false";
}
if (key == "bv-quick-xplain") {
return options::bitvectorQuickXplain() ? "true" : "false";
}
if (key == "bv-sat-solver") {
std::stringstream ss;
ss << options::bvSatSolver();
return ss.str();
}
if (key == "bv-skolemize") {
return options::skolemizeArguments() ? "true" : "false";
}
if (key == "bv-to-bool") {
return options::bitvectorToBool() ? "true" : "false";
}
if (key == "cdt-bisimilar") {
return options::cdtBisimilar() ? "true" : "false";
}
if (key == "dt-binary-split") {
return options::dtBinarySplit() ? "true" : "false";
}
if (key == "dt-blast-splits") {
return options::dtBlastSplits() ? "true" : "false";
}
if (key == "dt-cyclic") {
return options::dtCyclic() ? "true" : "false";
}
if (key == "dt-force-assignment") {
return options::dtForceAssignment() ? "true" : "false";
}
if (key == "dt-infer-as-lemmas") {
return options::dtInferAsLemmas() ? "true" : "false";
}
if (key == "dt-nested-rec") {
return options::dtNestedRec() ? "true" : "false";
}
if (key == "dt-rewrite-error-sel") {
return options::dtRewriteErrorSel() ? "true" : "false";
}
if (key == "dt-share-sel") {
return options::dtSharedSelectors() ? "true" : "false";
}
if (key == "sygus-abort-size") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::sygusAbortSize();
return ss.str();
}
if (key == "sygus-fair-max") {
return options::sygusFairMax() ? "true" : "false";
}
if (key == "sygus-fair") {
std::stringstream ss;
ss << options::sygusFair();
return ss.str();
}
if (key == "sygus-sym-break") {
return options::sygusSymBreak() ? "true" : "false";
}
if (key == "sygus-sym-break-agg") {
return options::sygusSymBreakAgg() ? "true" : "false";
}
if (key == "sygus-sym-break-dynamic") {
return options::sygusSymBreakDynamic() ? "true" : "false";
}
if (key == "sygus-sym-break-lazy") {
return options::sygusSymBreakLazy() ? "true" : "false";
}
if (key == "sygus-sym-break-pbe") {
return options::sygusSymBreakPbe() ? "true" : "false";
}
if (key == "sygus-sym-break-rlv") {
return options::sygusSymBreakRlv() ? "true" : "false";
}
if (key == "decision-random-weight") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::decisionRandomWeight();
return ss.str();
}
if (key == "decision-threshold") {
std::stringstream ss;
ss << options::decisionThreshold();
return ss.str();
}
if (key == "decision-use-weight") {
return options::decisionUseWeight() ? "true" : "false";
}
if (key == "decision-weight-internal") {
std::stringstream ss;
ss << options::decisionWeightInternal();
return ss.str();
}
if (key == "decision" || key == "decision-mode") {
std::stringstream ss;
ss << options::decisionMode();
return ss.str();
}
if (key == "dag-thresh") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::defaultDagThresh();
return ss.str();
}
if (key == "expr-depth") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::defaultExprDepth();
return ss.str();
}
if (key == "print-expr-types") {
return options::printExprTypes() ? "true" : "false";
}
if (key == "type-checking") {
return options::typeChecking() ? "true" : "false";
}
if (key == "fp-exp") {
return options::fpExp() ? "true" : "false";
}
if (key == "early-exit") {
return options::earlyExit() ? "true" : "false";
}
if (key == "help") {
return options::help() ? "true" : "false";
}
if (key == "interactive") {
return options::interactive() ? "true" : "false";
}
if (key == "interactive-prompt") {
return options::interactivePrompt() ? "true" : "false";
}
if (key == "seed") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::seed();
return ss.str();
}
if (key == "segv-spin") {
return options::segvSpin() ? "true" : "false";
}
if (key == "tear-down-incremental") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::tearDownIncremental();
return ss.str();
}
if (key == "version") {
return options::version() ? "true" : "false";
}
if (key == "filesystem-access") {
return options::filesystemAccess() ? "true" : "false";
}
if (key == "force-logic") {
std::stringstream ss;
ss << options::forceLogicString();
return ss.str();
}
if (key == "global-declarations") {
return options::globalDeclarations() ? "true" : "false";
}
if (key == "mmap") {
return options::memoryMap() ? "true" : "false";
}
if (key == "semantic-checks") {
return options::semanticChecks() ? "true" : "false";
}
if (key == "strict-parsing") {
return options::strictParsing() ? "true" : "false";
}
if (key == "flatten-ho-chains") {
return options::flattenHOChains() ? "true" : "false";
}
if (key == "inst-format") {
std::stringstream ss;
ss << options::instFormatMode();
return ss.str();
}
if (key == "model-format") {
std::stringstream ss;
ss << options::modelFormatMode();
return ss.str();
}
if (key == "aggressive-core-min") {
return options::aggressiveCoreMin() ? "true" : "false";
}
if (key == "allow-empty-dependencies") {
return options::allowEmptyDependencies() ? "true" : "false";
}
if (key == "fewer-preprocessing-holes") {
return options::fewerPreprocessingHoles() ? "true" : "false";
}
if (key == "lfsc-letification") {
return options::lfscLetification() ? "true" : "false";
}
if (key == "minisat-dump-dimacs") {
return options::minisatDumpDimacs() ? "true" : "false";
}
if (key == "minisat-elimination") {
return options::minisatUseElim() ? "true" : "false";
}
if (key == "random-freq" || key == "random-frequency") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::satRandomFreq();
return ss.str();
}
if (key == "random-seed") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::satRandomSeed();
return ss.str();
}
if (key == "refine-conflicts") {
return options::sat_refine_conflicts() ? "true" : "false";
}
if (key == "restart-int-base") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::satRestartFirst();
return ss.str();
}
if (key == "restart-int-inc") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::satRestartInc();
return ss.str();
}
if (key == "ag-miniscope-quant") {
return options::aggressiveMiniscopeQuant() ? "true" : "false";
}
if (key == "cegis-sample") {
std::stringstream ss;
ss << options::cegisSample();
return ss.str();
}
if (key == "cegqi") {
return options::cegqi() ? "true" : "false";
}
if (key == "cegqi-all") {
return options::cegqiAll() ? "true" : "false";
}
if (key == "cegqi-bv") {
return options::cegqiBv() ? "true" : "false";
}
if (key == "cegqi-bv-concat-inv") {
return options::cegqiBvConcInv() ? "true" : "false";
}
if (key == "cegqi-bv-ineq") {
std::stringstream ss;
ss << options::cegqiBvIneqMode();
return ss.str();
}
if (key == "cegqi-bv-interleave-value") {
return options::cegqiBvInterleaveValue() ? "true" : "false";
}
if (key == "cegqi-bv-linear") {
return options::cegqiBvLinearize() ? "true" : "false";
}
if (key == "cegqi-bv-rm-extract") {
return options::cegqiBvRmExtract() ? "true" : "false";
}
if (key == "cegqi-bv-solve-nl") {
return options::cegqiBvSolveNl() ? "true" : "false";
}
if (key == "cegqi-full") {
return options::cegqiFullEffort() ? "true" : "false";
}
if (key == "cegqi-innermost") {
return options::cegqiInnermost() ? "true" : "false";
}
if (key == "cegqi-midpoint") {
return options::cegqiMidpoint() ? "true" : "false";
}
if (key == "cegqi-min-bounds") {
return options::cegqiMinBounds() ? "true" : "false";
}
if (key == "cegqi-model") {
return options::cegqiModel() ? "true" : "false";
}
if (key == "cegqi-multi-inst") {
return options::cegqiMultiInst() ? "true" : "false";
}
if (key == "cegqi-nested-qe") {
return options::cegqiNestedQE() ? "true" : "false";
}
if (key == "cegqi-nopt") {
return options::cegqiNopt() ? "true" : "false";
}
if (key == "cegqi-prereg-inst") {
return options::cegqiPreRegInst() ? "true" : "false";
}
if (key == "cegqi-repeat-lit") {
return options::cegqiRepeatLit() ? "true" : "false";
}
if (key == "cegqi-round-up-lia") {
return options::cegqiRoundUpLowerLia() ? "true" : "false";
}
if (key == "cegqi-sat") {
return options::cegqiSat() ? "true" : "false";
}
if (key == "cegqi-use-inf-int") {
return options::cegqiUseInfInt() ? "true" : "false";
}
if (key == "cegqi-use-inf-real") {
return options::cegqiUseInfReal() ? "true" : "false";
}
if (key == "cond-var-split-agg-quant") {
return options::condVarSplitQuantAgg() ? "true" : "false";
}
if (key == "cond-var-split-quant") {
return options::condVarSplitQuant() ? "true" : "false";
}
if (key == "conjecture-filter-active-terms") {
return options::conjectureFilterActiveTerms() ? "true" : "false";
}
if (key == "conjecture-filter-canonical") {
return options::conjectureFilterCanonical() ? "true" : "false";
}
if (key == "conjecture-filter-model") {
return options::conjectureFilterModel() ? "true" : "false";
}
if (key == "conjecture-gen") {
return options::conjectureGen() ? "true" : "false";
}
if (key == "conjecture-gen-gt-enum") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::conjectureGenGtEnum();
return ss.str();
}
if (key == "conjecture-gen-max-depth") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::conjectureGenMaxDepth();
return ss.str();
}
if (key == "conjecture-gen-per-round") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::conjectureGenPerRound();
return ss.str();
}
if (key == "conjecture-gen-uee-intro") {
return options::conjectureUeeIntro() ? "true" : "false";
}
if (key == "conjecture-no-filter") {
return options::conjectureNoFilter() ? "true" : "false";
}
if (key == "dt-stc-ind") {
return options::dtStcInduction() ? "true" : "false";
}
if (key == "dt-var-exp-quant") {
return options::dtVarExpandQuant() ? "true" : "false";
}
if (key == "e-matching") {
return options::eMatching() ? "true" : "false";
}
if (key == "elim-ext-arith-quant") {
return options::elimExtArithQuant() ? "true" : "false";
}
if (key == "elim-taut-quant") {
return options::elimTautQuant() ? "true" : "false";
}
if (key == "ext-rewrite-quant") {
return options::extRewriteQuant() ? "true" : "false";
}
if (key == "finite-model-find") {
return options::finiteModelFind() ? "true" : "false";
}
if (key == "fmf-bound") {
return options::fmfBound() ? "true" : "false";
}
if (key == "fmf-bound-int") {
return options::fmfBoundInt() ? "true" : "false";
}
if (key == "fmf-bound-lazy") {
return options::fmfBoundLazy() ? "true" : "false";
}
if (key == "fmf-fmc-simple") {
return options::fmfFmcSimple() ? "true" : "false";
}
if (key == "fmf-fresh-dc") {
return options::fmfFreshDistConst() ? "true" : "false";
}
if (key == "fmf-fun") {
return options::fmfFunWellDefined() ? "true" : "false";
}
if (key == "fmf-fun-rlv") {
return options::fmfFunWellDefinedRelevant() ? "true" : "false";
}
if (key == "fmf-inst-engine") {
return options::fmfInstEngine() ? "true" : "false";
}
if (key == "fmf-type-completion-thresh") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::fmfTypeCompletionThresh();
return ss.str();
}
if (key == "fs-interleave") {
return options::fullSaturateInterleave() ? "true" : "false";
}
if (key == "fs-stratify") {
return options::fullSaturateStratify() ? "true" : "false";
}
if (key == "full-saturate-quant") {
return options::fullSaturateQuant() ? "true" : "false";
}
if (key == "full-saturate-quant-limit") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::fullSaturateLimit();
return ss.str();
}
if (key == "full-saturate-quant-rd") {
return options::fullSaturateQuantRd() ? "true" : "false";
}
if (key == "global-negate") {
return options::globalNegate() ? "true" : "false";
}
if (key == "ho-elim") {
return options::hoElim() ? "true" : "false";
}
if (key == "ho-elim-store-ax") {
return options::hoElimStoreAx() ? "true" : "false";
}
if (key == "ho-matching") {
return options::hoMatching() ? "true" : "false";
}
if (key == "ho-matching-var-priority") {
return options::hoMatchingVarArgPriority() ? "true" : "false";
}
if (key == "ho-merge-term-db") {
return options::hoMergeTermDb() ? "true" : "false";
}
if (key == "increment-triggers") {
return options::incrementTriggers() ? "true" : "false";
}
if (key == "inst-level-input-only") {
return options::instLevelInputOnly() ? "true" : "false";
}
if (key == "inst-max-level") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::instMaxLevel();
return ss.str();
}
if (key == "inst-no-entail") {
return options::instNoEntail() ? "true" : "false";
}
if (key == "inst-no-model-true") {
return options::instNoModelTrue() ? "true" : "false";
}
if (key == "inst-when-phase") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::instWhenPhase();
return ss.str();
}
if (key == "inst-when-strict-interleave") {
return options::instWhenStrictInterleave() ? "true" : "false";
}
if (key == "inst-when-tc-first") {
return options::instWhenTcFirst() ? "true" : "false";
}
if (key == "inst-when") {
std::stringstream ss;
ss << options::instWhenMode();
return ss.str();
}
if (key == "int-wf-ind") {
return options::intWfInduction() ? "true" : "false";
}
if (key == "ite-dtt-split-quant") {
return options::iteDtTesterSplitQuant() ? "true" : "false";
}
if (key == "ite-lift-quant") {
std::stringstream ss;
ss << options::iteLiftQuant();
return ss.str();
}
if (key == "literal-matching") {
std::stringstream ss;
ss << options::literalMatchMode();
return ss.str();
}
if (key == "lte-restrict-inst-closure") {
return options::lteRestrictInstClosure() ? "true" : "false";
}
if (key == "macros-quant") {
return options::macrosQuant() ? "true" : "false";
}
if (key == "macros-quant-mode") {
std::stringstream ss;
ss << options::macrosQuantMode();
return ss.str();
}
if (key == "mbqi-interleave") {
return options::mbqiInterleave() ? "true" : "false";
}
if (key == "mbqi-one-inst-per-round") {
return options::fmfOneInstPerRound() ? "true" : "false";
}
if (key == "mbqi") {
std::stringstream ss;
ss << options::mbqiMode();
return ss.str();
}
if (key == "miniscope-quant") {
return options::miniscopeQuant() ? "true" : "false";
}
if (key == "miniscope-quant-fv") {
return options::miniscopeQuantFreeVar() ? "true" : "false";
}
if (key == "multi-trigger-cache") {
return options::multiTriggerCache() ? "true" : "false";
}
if (key == "multi-trigger-linear") {
return options::multiTriggerLinear() ? "true" : "false";
}
if (key == "multi-trigger-priority") {
return options::multiTriggerPriority() ? "true" : "false";
}
if (key == "multi-trigger-when-single") {
return options::multiTriggerWhenSingle() ? "true" : "false";
}
if (key == "partial-triggers") {
return options::partialTriggers() ? "true" : "false";
}
if (key == "pre-skolem-quant") {
return options::preSkolemQuant() ? "true" : "false";
}
if (key == "pre-skolem-quant-agg") {
return options::preSkolemQuantAgg() ? "true" : "false";
}
if (key == "pre-skolem-quant-nested") {
return options::preSkolemQuantNested() ? "true" : "false";
}
if (key == "prenex-quant-user") {
return options::prenexQuantUser() ? "true" : "false";
}
if (key == "prenex-quant") {
std::stringstream ss;
ss << options::prenexQuant();
return ss.str();
}
if (key == "pure-th-triggers") {
return options::pureThTriggers() ? "true" : "false";
}
if (key == "purify-dt-triggers") {
return options::purifyDtTriggers() ? "true" : "false";
}
if (key == "purify-triggers") {
return options::purifyTriggers() ? "true" : "false";
}
if (key == "qcf-all-conflict") {
return options::qcfAllConflict() ? "true" : "false";
}
if (key == "qcf-eager-check-rd") {
return options::qcfEagerCheckRd() ? "true" : "false";
}
if (key == "qcf-eager-test") {
return options::qcfEagerTest() ? "true" : "false";
}
if (key == "qcf-nested-conflict") {
return options::qcfNestedConflict() ? "true" : "false";
}
if (key == "qcf-skip-rd") {
return options::qcfSkipRd() ? "true" : "false";
}
if (key == "qcf-tconstraint") {
return options::qcfTConstraint() ? "true" : "false";
}
if (key == "qcf-vo-exp") {
return options::qcfVoExp() ? "true" : "false";
}
if (key == "quant-alpha-equiv") {
return options::quantAlphaEquiv() ? "true" : "false";
}
if (key == "quant-anti-skolem") {
return options::quantAntiSkolem() ? "true" : "false";
}
if (key == "quant-cf") {
return options::quantConflictFind() ? "true" : "false";
}
if (key == "quant-cf-mode") {
std::stringstream ss;
ss << options::qcfMode();
return ss.str();
}
if (key == "quant-cf-when") {
std::stringstream ss;
ss << options::qcfWhenMode();
return ss.str();
}
if (key == "quant-dsplit-mode") {
std::stringstream ss;
ss << options::quantDynamicSplit();
return ss.str();
}
if (key == "quant-epr") {
return options::quantEpr() ? "true" : "false";
}
if (key == "quant-epr-match") {
return options::quantEprMatching() ? "true" : "false";
}
if (key == "quant-fun-wd") {
return options::quantFunWellDefined() ? "true" : "false";
}
if (key == "quant-ind") {
return options::quantInduction() ? "true" : "false";
}
if (key == "quant-model-ee") {
return options::quantModelEe() ? "true" : "false";
}
if (key == "quant-rep-mode") {
std::stringstream ss;
ss << options::quantRepMode();
return ss.str();
}
if (key == "quant-split") {
return options::quantSplit() ? "true" : "false";
}
if (key == "register-quant-body-terms") {
return options::registerQuantBodyTerms() ? "true" : "false";
}
if (key == "relational-triggers") {
return options::relationalTriggers() ? "true" : "false";
}
if (key == "relevant-triggers") {
return options::relevantTriggers() ? "true" : "false";
}
if (key == "strict-triggers") {
return options::strictTriggers() ? "true" : "false";
}
if (key == "sygus") {
return options::sygus() ? "true" : "false";
}
if (key == "sygus-active-gen-cfactor") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::sygusActiveGenEnumConsts();
return ss.str();
}
if (key == "sygus-active-gen") {
std::stringstream ss;
ss << options::sygusActiveGenMode();
return ss.str();
}
if (key == "sygus-add-const-grammar") {
return options::sygusAddConstGrammar() ? "true" : "false";
}
if (key == "sygus-arg-relevant") {
return options::sygusArgRelevant() ? "true" : "false";
}
if (key == "sygus-auto-unfold") {
return options::sygusInvAutoUnfold() ? "true" : "false";
}
if (key == "sygus-bool-ite-return-const") {
return options::sygusBoolIteReturnConst() ? "true" : "false";
}
if (key == "sygus-core-connective") {
return options::sygusCoreConnective() ? "true" : "false";
}
if (key == "sygus-crepair-abort") {
return options::sygusConstRepairAbort() ? "true" : "false";
}
if (key == "sygus-eval-opt") {
return options::sygusEvalOpt() ? "true" : "false";
}
if (key == "sygus-eval-unfold") {
return options::sygusEvalUnfold() ? "true" : "false";
}
if (key == "sygus-eval-unfold-bool") {
return options::sygusEvalUnfoldBool() ? "true" : "false";
}
if (key == "sygus-expr-miner-check-timeout") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::sygusExprMinerCheckTimeout();
return ss.str();
}
if (key == "sygus-expr-miner-check-use-export") {
return options::sygusExprMinerCheckUseExport() ? "true" : "false";
}
if (key == "sygus-ext-rew") {
return options::sygusExtRew() ? "true" : "false";
}
if (key == "sygus-filter-sol-rev") {
return options::sygusFilterSolRevSubsume() ? "true" : "false";
}
if (key == "sygus-filter-sol") {
std::stringstream ss;
ss << options::sygusFilterSolMode();
return ss.str();
}
if (key == "sygus-grammar-cons") {
std::stringstream ss;
ss << options::sygusGrammarConsMode();
return ss.str();
}
if (key == "sygus-grammar-norm") {
return options::sygusGrammarNorm() ? "true" : "false";
}
if (key == "sygus-inference") {
return options::sygusInference() ? "true" : "false";
}
if (key == "sygus-inst") {
return options::sygusInst() ? "true" : "false";
}
if (key == "sygus-inv-templ-when-sg") {
return options::sygusInvTemplWhenSyntax() ? "true" : "false";
}
if (key == "sygus-inv-templ") {
std::stringstream ss;
ss << options::sygusInvTemplMode();
return ss.str();
}
if (key == "sygus-min-grammar") {
return options::sygusMinGrammar() ? "true" : "false";
}
if (key == "sygus-pbe") {
return options::sygusUnifPbe() ? "true" : "false";
}
if (key == "sygus-pbe-multi-fair") {
return options::sygusPbeMultiFair() ? "true" : "false";
}
if (key == "sygus-pbe-multi-fair-diff") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::sygusPbeMultiFairDiff();
return ss.str();
}
if (key == "sygus-qe-preproc") {
return options::sygusQePreproc() ? "true" : "false";
}
if (key == "sygus-query-gen") {
return options::sygusQueryGen() ? "true" : "false";
}
if (key == "sygus-query-gen-check") {
return options::sygusQueryGenCheck() ? "true" : "false";
}
if (key == "sygus-query-gen-dump-files") {
std::stringstream ss;
ss << options::sygusQueryGenDumpFiles();
return ss.str();
}
if (key == "sygus-query-gen-thresh") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::sygusQueryGenThresh();
return ss.str();
}
if (key == "sygus-rec-fun") {
return options::sygusRecFun() ? "true" : "false";
}
if (key == "sygus-rec-fun-eval-limit") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::sygusRecFunEvalLimit();
return ss.str();
}
if (key == "sygus-repair-const") {
return options::sygusRepairConst() ? "true" : "false";
}
if (key == "sygus-repair-const-timeout") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::sygusRepairConstTimeout();
return ss.str();
}
if (key == "sygus-rr") {
return options::sygusRew() ? "true" : "false";
}
if (key == "sygus-rr-synth") {
return options::sygusRewSynth() ? "true" : "false";
}
if (key == "sygus-rr-synth-accel") {
return options::sygusRewSynthAccel() ? "true" : "false";
}
if (key == "sygus-rr-synth-check") {
return options::sygusRewSynthCheck() ? "true" : "false";
}
if (key == "sygus-rr-synth-filter-cong") {
return options::sygusRewSynthFilterCong() ? "true" : "false";
}
if (key == "sygus-rr-synth-filter-match") {
return options::sygusRewSynthFilterMatch() ? "true" : "false";
}
if (key == "sygus-rr-synth-filter-nl") {
return options::sygusRewSynthFilterNonLinear() ? "true" : "false";
}
if (key == "sygus-rr-synth-filter-order") {
return options::sygusRewSynthFilterOrder() ? "true" : "false";
}
if (key == "sygus-rr-synth-input") {
return options::sygusRewSynthInput() ? "true" : "false";
}
if (key == "sygus-rr-synth-input-nvars") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::sygusRewSynthInputNVars();
return ss.str();
}
if (key == "sygus-rr-synth-input-use-bool") {
return options::sygusRewSynthInputUseBool() ? "true" : "false";
}
if (key == "sygus-rr-synth-rec") {
return options::sygusRewSynthRec() ? "true" : "false";
}
if (key == "sygus-rr-verify") {
return options::sygusRewVerify() ? "true" : "false";
}
if (key == "sygus-rr-verify-abort") {
return options::sygusRewVerifyAbort() ? "true" : "false";
}
if (key == "sygus-sample-fp-uniform") {
return options::sygusSampleFpUniform() ? "true" : "false";
}
if (key == "sygus-sample-grammar") {
return options::sygusSampleGrammar() ? "true" : "false";
}
if (key == "sygus-samples") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::sygusSamples();
return ss.str();
}
if (key == "sygus-si-abort") {
return options::cegqiSingleInvAbort() ? "true" : "false";
}
if (key == "sygus-si-partial") {
return options::cegqiSingleInvPartial() ? "true" : "false";
}
if (key == "sygus-si-rcons-limit") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::cegqiSingleInvReconstructLimit();
return ss.str();
}
if (key == "sygus-si-rcons") {
std::stringstream ss;
ss << options::cegqiSingleInvReconstruct();
return ss.str();
}
if (key == "sygus-si-reconstruct-const") {
return options::cegqiSingleInvReconstructConst() ? "true" : "false";
}
if (key == "sygus-si") {
std::stringstream ss;
ss << options::cegqiSingleInvMode();
return ss.str();
}
if (key == "sygus-stream") {
return options::sygusStream() ? "true" : "false";
}
if (key == "sygus-templ-embed-grammar") {
return options::sygusTemplEmbedGrammar() ? "true" : "false";
}
if (key == "sygus-unif-cond-independent-no-repeat-sol") {
return options::sygusUnifCondIndNoRepeatSol() ? "true" : "false";
}
if (key == "sygus-unif-pi") {
std::stringstream ss;
ss << options::sygusUnifPi();
return ss.str();
}
if (key == "sygus-unif-shuffle-cond") {
return options::sygusUnifShuffleCond() ? "true" : "false";
}
if (key == "sygus-verify-subcall") {
return options::sygusVerifySubcall() ? "true" : "false";
}
if (key == "term-db-mode") {
std::stringstream ss;
ss << options::termDbMode();
return ss.str();
}
if (key == "track-inst-lemmas") {
return options::trackInstLemmas() ? "true" : "false";
}
if (key == "trigger-active-sel") {
std::stringstream ss;
ss << options::triggerActiveSelMode();
return ss.str();
}
if (key == "trigger-sel") {
std::stringstream ss;
ss << options::triggerSelMode();
return ss.str();
}
if (key == "user-pat") {
std::stringstream ss;
ss << options::userPatternsQuant();
return ss.str();
}
if (key == "var-elim-quant") {
return options::varElimQuant() ? "true" : "false";
}
if (key == "var-ineq-elim-quant") {
return options::varIneqElimQuant() ? "true" : "false";
}
if (key == "sep-check-neg") {
return options::sepCheckNeg() ? "true" : "false";
}
if (key == "sep-child-refine") {
return options::sepChildRefine() ? "true" : "false";
}
if (key == "sep-deq-c") {
return options::sepDisequalC() ? "true" : "false";
}
if (key == "sep-exp") {
return options::sepExp() ? "true" : "false";
}
if (key == "sep-min-refine") {
return options::sepMinimalRefine() ? "true" : "false";
}
if (key == "sep-pre-skolem-emp") {
return options::sepPreSkolemEmp() ? "true" : "false";
}
if (key == "sets-ext") {
return options::setsExt() ? "true" : "false";
}
if (key == "sets-infer-as-lemmas") {
return options::setsInferAsLemmas() ? "true" : "false";
}
if (key == "sets-proxy-lemmas") {
return options::setsProxyLemmas() ? "true" : "false";
}
if (key == "abstract-values") {
return options::abstractValues() ? "true" : "false";
}
if (key == "ackermann") {
return options::ackermann() ? "true" : "false";
}
if (key == "bitblast-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::bitblastStep();
return ss.str();
}
if (key == "block-models") {
std::stringstream ss;
ss << options::blockModelsMode();
return ss.str();
}
if (key == "bv-eager-assert-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::bvEagerAssertStep();
return ss.str();
}
if (key == "bv-propagation-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::bvPropagationStep();
return ss.str();
}
if (key == "bv-sat-conflict-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::bvSatConflictStep();
return ss.str();
}
if (key == "check-abducts") {
return options::checkAbducts() ? "true" : "false";
}
if (key == "check-models") {
return options::checkModels() ? "true" : "false";
}
if (key == "check-proofs") {
return options::checkProofs() ? "true" : "false";
}
if (key == "check-synth-sol") {
return options::checkSynthSol() ? "true" : "false";
}
if (key == "check-unsat-cores") {
return options::checkUnsatCores() ? "true" : "false";
}
if (key == "cnf-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::cnfStep();
return ss.str();
}
if (key == "cpu-time") {
return options::cpuTime() ? "true" : "false";
}
if (key == "debug-check-models") {
return options::debugCheckModels() ? "true" : "false";
}
if (key == "decision-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::decisionStep();
return ss.str();
}
if (key == "diagnostic-output-channel") {
std::stringstream ss;
ss << options::diagnosticChannelName();
return ss.str();
}
if (key == "dump-instantiations") {
return options::dumpInstantiations() ? "true" : "false";
}
if (key == "dump-models") {
return options::dumpModels() ? "true" : "false";
}
if (key == "dump-proofs") {
return options::dumpProofs() ? "true" : "false";
}
if (key == "dump-synth") {
return options::dumpSynth() ? "true" : "false";
}
if (key == "dump-to") {
std::stringstream ss;
ss << options::dumpToFileName();
return ss.str();
}
if (key == "dump-unsat-cores") {
return options::dumpUnsatCores() ? "true" : "false";
}
if (key == "dump-unsat-cores-full") {
return options::dumpUnsatCoresFull() ? "true" : "false";
}
if (key == "dump") {
std::stringstream ss;
ss << options::dumpModeString();
return ss.str();
}
if (key == "expand-definitions") {
return options::expandDefinitions() ? "true" : "false";
}
if (key == "ext-rew-prep") {
return options::extRewPrep() ? "true" : "false";
}
if (key == "ext-rew-prep-agg") {
return options::extRewPrepAgg() ? "true" : "false";
}
if (key == "force-no-limit-cpu-while-dump") {
return options::forceNoLimitCpuWhileDump() ? "true" : "false";
}
if (key == "hard-limit") {
return options::hardLimit() ? "true" : "false";
}
if (key == "incremental") {
return options::incrementalSolving() ? "true" : "false";
}
if (key == "interactive-mode") {
return options::interactiveMode() ? "true" : "false";
}
if (key == "ite-simp") {
return options::doITESimp() ? "true" : "false";
}
if (key == "lemma-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::lemmaStep();
return ss.str();
}
if (key == "model-cores") {
std::stringstream ss;
ss << options::modelCoresMode();
return ss.str();
}
if (key == "model-u-dt-enum") {
return options::modelUninterpDtEnum() ? "true" : "false";
}
if (key == "model-witness-value") {
return options::modelWitnessValue() ? "true" : "false";
}
if (key == "on-repeat-ite-simp") {
return options::doITESimpOnRepeat() ? "true" : "false";
}
if (key == "parse-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::parseStep();
return ss.str();
}
if (key == "preprocess-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::preprocessStep();
return ss.str();
}
if (key == "produce-abducts") {
return options::produceAbducts() ? "true" : "false";
}
if (key == "produce-assertions") {
return options::produceAssertions() ? "true" : "false";
}
if (key == "produce-assignments") {
return options::produceAssignments() ? "true" : "false";
}
if (key == "produce-models") {
return options::produceModels() ? "true" : "false";
}
if (key == "produce-unsat-assumptions") {
return options::unsatAssumptions() ? "true" : "false";
}
if (key == "produce-unsat-cores") {
return options::unsatCores() ? "true" : "false";
}
if (key == "produce-proofs" || key == "proof") {
return options::proof() ? "true" : "false";
}
if (key == "quantifier-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::quantifierStep();
return ss.str();
}
if (key == "regular-output-channel") {
std::stringstream ss;
ss << options::regularChannelName();
return ss.str();
}
if (key == "repeat-simp") {
return options::repeatSimp() ? "true" : "false";
}
if (key == "restart-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::restartStep();
return ss.str();
}
if (key == "rewrite-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::rewriteStep();
return ss.str();
}
if (key == "reproducible-resource-limit" || key == "rlimit-per") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::perCallResourceLimit();
return ss.str();
}
if (key == "rlimit") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::cumulativeResourceLimit();
return ss.str();
}
if (key == "sat-conflict-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::satConflictStep();
return ss.str();
}
if (key == "simp-ite-compress") {
return options::compressItes() ? "true" : "false";
}
if (key == "simp-ite-hunt-zombies") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::zombieHuntThreshold();
return ss.str();
}
if (key == "simp-with-care") {
return options::simplifyWithCareEnabled() ? "true" : "false";
}
if (key == "simplification" || key == "simplification-mode") {
std::stringstream ss;
ss << options::simplificationMode();
return ss.str();
}
if (key == "solve-bv-as-int") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::solveBVAsInt();
return ss.str();
}
if (key == "solve-int-as-bv") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::solveIntAsBV();
return ss.str();
}
if (key == "solve-real-as-int") {
return options::solveRealAsInt() ? "true" : "false";
}
if (key == "sort-inference") {
return options::sortInference() ? "true" : "false";
}
if (key == "static-learning") {
return options::doStaticLearning() ? "true" : "false";
}
if (key == "sygus-out") {
std::stringstream ss;
ss << options::sygusOut();
return ss.str();
}
if (key == "sygus-print-callbacks") {
return options::sygusPrintCallbacks() ? "true" : "false";
}
if (key == "theory-check-step") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::theoryCheckStep();
return ss.str();
}
if (key == "tlimit-per") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::perCallMillisecondLimit();
return ss.str();
}
if (key == "tlimit") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::cumulativeMillisecondLimit();
return ss.str();
}
if (key == "unconstrained-simp") {
return options::unconstrainedSimp() ? "true" : "false";
}
if (key == "re-elim") {
return options::regExpElim() ? "true" : "false";
}
if (key == "re-elim-agg") {
return options::regExpElimAgg() ? "true" : "false";
}
if (key == "re-inter-mode") {
std::stringstream ss;
ss << options::stringRegExpInterMode();
return ss.str();
}
if (key == "strings-check-entail-len") {
return options::stringCheckEntailLen() ? "true" : "false";
}
if (key == "strings-eager") {
return options::stringEager() ? "true" : "false";
}
if (key == "strings-eager-len") {
return options::stringEagerLen() ? "true" : "false";
}
if (key == "strings-exp") {
return options::stringExp() ? "true" : "false";
}
if (key == "strings-ff") {
return options::stringFlatForms() ? "true" : "false";
}
if (key == "strings-fmf") {
return options::stringFMF() ? "true" : "false";
}
if (key == "strings-guess-model") {
return options::stringGuessModel() ? "true" : "false";
}
if (key == "strings-infer-as-lemmas") {
return options::stringInferAsLemmas() ? "true" : "false";
}
if (key == "strings-infer-sym") {
return options::stringInferSym() ? "true" : "false";
}
if (key == "strings-inm") {
return options::stringIgnNegMembership() ? "true" : "false";
}
if (key == "strings-lazy-pp") {
return options::stringLazyPreproc() ? "true" : "false";
}
if (key == "strings-len-norm") {
return options::stringLenNorm() ? "true" : "false";
}
if (key == "strings-lprop-csp") {
return options::stringLenPropCsp() ? "true" : "false";
}
if (key == "strings-min-prefix-explain") {
return options::stringMinPrefixExplain() ? "true" : "false";
}
if (key == "strings-print-ascii") {
return options::stdPrintASCII() ? "true" : "false";
}
if (key == "strings-process-loop-mode") {
std::stringstream ss;
ss << options::stringProcessLoopMode();
return ss.str();
}
if (key == "strings-rexplain-lemmas") {
return options::stringRExplainLemmas() ? "true" : "false";
}
if (key == "assign-function-values") {
return options::assignFunctionValues() ? "true" : "false";
}
if (key == "condense-function-values") {
return options::condenseFunctionValues() ? "true" : "false";
}
if (key == "theoryof-mode") {
std::stringstream ss;
ss << options::theoryOfMode();
return ss.str();
}
if (key == "symmetry-breaker" || key == "uf-symmetry-breaker") {
return options::ufSymmetryBreaker() ? "true" : "false";
}
if (key == "uf-ho") {
return options::ufHo() ? "true" : "false";
}
if (key == "uf-ho-ext") {
return options::ufHoExt() ? "true" : "false";
}
if (key == "uf-ss-abort-card") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::ufssAbortCardinality();
return ss.str();
}
if (key == "uf-ss-fair") {
return options::ufssFairness() ? "true" : "false";
}
if (key == "uf-ss-fair-monotone") {
return options::ufssFairnessMonotone() ? "true" : "false";
}
if (key == "uf-ss-totality") {
return options::ufssTotality() ? "true" : "false";
}
if (key == "uf-ss-totality-limited") {
std::stringstream ss;
ss << std::fixed << std::setprecision(8);
ss << options::ufssTotalityLimited();
return ss.str();
}
if (key == "uf-ss-totality-sym-break") {
return options::ufssTotalitySymBreak() ? "true" : "false";
}
if (key == "uf-ss") {
std::stringstream ss;
ss << options::ufssMode();
return ss.str();
}


  throw UnrecognizedOptionException(key);
}

#undef USE_EARLY_TYPE_CHECKING_BY_DEFAULT
#undef DO_SEMANTIC_CHECKS_BY_DEFAULT

}  // namespace CVC4
