// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef EXTENSIONS_RENDERER_EXTENSION_URL_LOADER_THROTTLE_H_
#define EXTENSIONS_RENDERER_EXTENSION_URL_LOADER_THROTTLE_H_

#include <string>
#include <vector>

#include "base/memory/raw_ptr.h"
#include "base/scoped_observation.h"
#include "extensions/renderer/extension_throttle_manager.h"
#include "third_party/blink/public/common/loader/url_loader_throttle.h"
#include "url/gurl.h"

namespace extensions {

class ExtensionThrottleManager;

// This class monitors requests issued by extensions and throttles the request
// if there are too many requests made within a short time to urls with the same
// scheme, host, port and path. For the exact criteria for throttling, please
// also see extension_throttle_manager.cc.
class ExtensionURLLoaderThrottle
    : public blink::URLLoaderThrottle,
      public ExtensionThrottleManager::ExtensionThrottleManagerObserver {
 public:
  explicit ExtensionURLLoaderThrottle(ExtensionThrottleManager* manager);

  ExtensionURLLoaderThrottle(const ExtensionURLLoaderThrottle&) = delete;
  ExtensionURLLoaderThrottle& operator=(const ExtensionURLLoaderThrottle&) =
      delete;

  ~ExtensionURLLoaderThrottle() override;

  // blink::URLLoaderThrottle:
  void WillStartRequest(network::ResourceRequest* request,
                        bool* defer) override;
  void WillRedirectRequest(
      net::RedirectInfo* redirect_info,
      const network::mojom::URLResponseHead& response_head,
      bool* defer,
      std::vector<std::string>* to_be_removed_request_headers,
      net::HttpRequestHeaders* modified_request_headers,
      net::HttpRequestHeaders* modified_cors_exempt_request_headers) override;
  void WillProcessResponse(const GURL& response_url,
                           network::mojom::URLResponseHead* response_head,
                           bool* defer) override;

  // ExtensionThrottleManagerObserver.
  void OnExtensionThrottleManagerDestruct(
      ExtensionThrottleManager* manager) override;

 private:
  // blink::URLLoaderThrottle:
  void DetachFromCurrentSequence() override;

  raw_ptr<ExtensionThrottleManager> manager_ = nullptr;
  GURL start_request_url_;

  base::ScopedObservation<ExtensionThrottleManager,
                          ExtensionThrottleManagerObserver>
      manager_observation_{this};
};

}  // namespace extensions

#endif  // EXTENSIONS_RENDERER_EXTENSION_URL_LOADER_THROTTLE_H_
